#!/usr/bin/env python
# Copyright (c) 2008, 2009, 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import sys, os, glob
import argparse
import Swi

defaultYumConfig = "/etc/Ayum.conf"

def swiUpdate( swiVariant=None, swiFlavor=None, packages=None, yumConfig=None,
               enableRepos=None, disableRepos=None, swiMaxHwEpoch=None,
               targets=None, extraExclude=None, zipRoot=False ):
   if extraExclude is None:
      extraExclude = {}
   def updateWithinExtractedSwi( d ):
      rootDir = glob.glob( os.path.join( d, "rootfs-*.dir" ) )[ 0 ]

      def updateSwiConfig( label, config ):
         printData = { "config": config,
                       "label": label }
         print "Updating %(label)s=%(config)s" % printData
         Swi.run( [ 'sed', '-i',
                    's/%(label)s=.*$/%(label)s=%(config)s/g' % printData,
                    '%s/version' % d ] )
         Swi.run( [ 'sudo', 'sed', '-i',
                    's/%(label)s=.*$/%(label)s=%(config)s/g' % printData,
                    '%s/etc/swi-version' % rootDir ] )

      if swiVariant:
         updateSwiConfig( "SWI_VARIANT", swiVariant )
      if swiFlavor:
         updateSwiConfig( "SWI_FLAVOR", swiFlavor )
      if swiMaxHwEpoch:
         updateSwiConfig( "SWI_MAX_HWEPOCH", swiMaxHwEpoch )

      if packages:
         # yum install packages into rootfs
         cmd = [ "sudo" ]
         if not yumConfig:
            cmd += [ "a4", "yum" ]
         else:
            cmd += [ "yum", "-c", yumConfig ]
         for repo in enableRepos:
            cmd += [ "--enablerepo=%s" % ( repo ) ]
         for repo in disableRepos:
            cmd += [ "--disablerepo=%s" % ( repo ) ]
         Swi.run( cmd + [ "-y", "--installroot=%s" % rootDir, "install" ] +
                  packages )
         # Make sure packages were really installed
         Swi.run( [ "rpm", "--root=%s" % rootDir, "-q" ] + packages )
         # Clean up yum cache
         Swi.run( [ "sudo", "sh", "-c", "rm -rf %s/var/cache/yum/*" % rootDir ] )

      for originalPackage, extraPackages in extraExclude.iteritems():
         newPackages = '\\n'.join( [ originalPackage ] + extraPackages )
         sedStr = "s/%s$/%s/g" % ( originalPackage, newPackages )
         path = '%s/etc/EosPkgExclude.d/*.exclude' % rootDir
         Swi.run( [ 'sudo', 'sed', '-i', sedStr, path ] )

      if targets:
         for f in targets:
            source, dest = f 
            baseDir = d if zipRoot else rootDir
            Swi.run( [ "sudo", "cp", "-ar", source,
                       "%s/%s" % ( baseDir, dest ) ] )
   return updateWithinExtractedSwi

def updateHandler( args=sys.argv[1:] ):
   parser = argparse.ArgumentParser( description="Updates files or attributes of a"
                                                 " swi" )
   parser.add_argument( "image", type=str, nargs=1, help="EOS.swi image to update" )
   parser.add_argument( "--swiVariant", help="Update EOS SWI variant" )
   parser.add_argument( '--swiFlavor', help="Update EOS SWI flavor" )
   parser.add_argument( "--swiMaxHwEpoch", help="Update EOS SWI max HW epoch" )
   parser.add_argument( "--yum-config", "-c", metavar="CONFIG",
                        help="use yum configuration file CONFIG" )
   parser.add_argument( "--enablerepo", action="append",
                        help="enable repository (may be specified multiple times)" )
   parser.add_argument( "--disablerepo", action="append",
                        help="disable repository (may be specified multiple times)" )
   parser.add_argument( "--zipRoot", action="store_true",
                        help="Update file(s) in zip root instead of rootfs" )

   group = parser.add_mutually_exclusive_group()
   group.add_argument( "--target", "-t", type=str, nargs=2, action="append",
                        help="SOURCE and DESTINATION of a file or directory (may be"
                             " specified multiple times" )
   group.add_argument( "--package", "-p", action="append",
                     help="install PACKAGE (may be specified multiple times)" )
   args = parser.parse_args( args )

   # Provide defaults for enablerepo and disablerepo, don't use
   # op.set_defaults(), as then anything provided by the user gets
   # appended to the defaults, the user specified ones does not
   # replace the defaults.
   if args.enablerepo or args.disablerepo:
      enablerepo = args.enablerepo if args.enablerepo else []
      disablerepo = args.disablerepo if args.disablerepo else []
   else:
      enablerepo = [ 'local' ]
      disablerepo = []

   # The zipRoot option does not work with specifying a package.  There isn't
   # an easy way to express this with argparse, so check that here:
   if args.zipRoot and args.package:
      raise ValueError( "--zipRoot can not be used with --package" )

   swiUpdateFn = swiUpdate( swiVariant=args.swiVariant,
                            swiFlavor=args.swiFlavor,
                            packages=args.package,
                            yumConfig=args.yum_config,
                            enableRepos=enablerepo,
                            disableRepos=disablerepo,
                            swiMaxHwEpoch=args.swiMaxHwEpoch,
                            targets=args.target,
                            zipRoot=args.zipRoot )

   Swi.inSwi( args.image[ 0 ], [ swiUpdateFn ], fast=False )

if __name__ == "__main__":
   updateHandler()
