#!/usr/bin/env python
# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
import Swi, Swi.extract, Swi.flavor
import argparse, os, re, sys

def stripSwi( image, output, products, rpms, verbose=False ):
   input_bytes = os.stat( image ).st_size if os.path.exists( image ) else 0
   def deleteRpms( tmpdir ):
      rootDir = os.path.join( tmpdir, 'rootfs-i386.dir' )
      excludePackagesPath = os.path.join( rootDir, 'etc/EosPkgExclude.d' )

      rpmExcludes = []
      if products:
         productRe = "|".join( products )
         excludes = []
         for f in os.listdir( excludePackagesPath ):
            if f.endswith( ".exclude" ):
               if re.search( productRe, f ):
                  # Matching list, read its rpms
                  print( "Matched " + f.split( '.exclude' )[ 0 ] )
                  with open( os.path.join( excludePackagesPath, f ) ) as fr:
                     excludes.append( set( fr.read().splitlines() ) )
               else:
                  # Non-matching list, delete it since that product will not
                  # be supported in the stripped swi
                  Swi.run( [ "sudo", "rm", os.path.join( excludePackagesPath, f ) ] )

         if not excludes:
            print( "Found no matching products supported in this swi. Aborting" )
            sys.exit( 1 )
         rpmExcludes += list( set.intersection( *excludes ) )

      if rpms:
         rpmExcludes += rpms

      # filter for rpms that actually exist in the image
      rpmExcludes = Swi.flavor.getRpmsToExclude( rootDir, rpmExcludes )
      if not rpmExcludes:
         print( "Output swi would be identical to input swi. Aborting" )
         sys.exit( 1 )

      print( "Removing " + str( len( rpmExcludes ) ) + " rpms" )
      if verbose:
         print( '\n'.join( sorted( rpmExcludes ) ) )
      Swi.run( [ "sudo", "rpm", "--root=%s" % rootDir, "--nodeps", "-e" ] +
               list( rpmExcludes ) )
   Swi.inSwi( image, [ deleteRpms ], outputfile=output )
   output_bytes = os.stat( output ).st_size if output else os.stat( image ).st_size
   input_mbytes = int( round( input_bytes / 1024 / 1024 ) )
   output_mbytes = int( round( output_bytes / 1024 / 1024 ) )
   print( "Reduced swi size from %sMB to %sMB" %
          ( str( input_mbytes ), str( output_mbytes ) ) )

def stripHandler( args=None ):
   if args is None:
      args = sys.argv[ 1: ]

   op = argparse.ArgumentParser(
      prog="swi strip",
      description="Create a stripped swi from a full swi that only supports "
                  "specified hardware",
      usage="%(prog)s EOS.swi [-o target] [-p product] [-r rpm]" )
   op.add_argument( '-o', '--output',
                    help="target swi for output (default overwrites input swi)",
                    action='store' )
   op.add_argument( '-p', '--product',
                    help="product to support in the stripped swi"
                    " (may be specified multiple times)",
                    action='append' )
   op.add_argument( '-r', '--rpm',
                    help="rpm to remove from the stripped swi"
                    " (may be specified multiple times)",
                    action='append' )
   op.add_argument( '-v', '--verbose',
                    help="print all rpms removed from the stripped swi",
                    action='store_true' )

   opts, args = op.parse_known_args( args )

   if len( args ) != 1:
      op.error( "Please provide exactly one input swi" )
   if not opts.product and not opts.rpm:
      op.error( "Please specify the product(s) you wish to support with "
                "'-p product' or rpm(s) to remove with '-r rpm'" )

   stripSwi( args[ 0 ], opts.output, opts.product, opts.rpm, verbose=opts.verbose )

if __name__ == "__main__":
   stripHandler()
