#!/usr/bin/env python
# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import print_function

import argparse
from enum import Enum
import os
import sys

import SignatureFile
import SignatureRequest
import Swi
import SwiSignLib
import Tac
import VerifySwi

class SWI_SIGN_RESULT:
   SUCCESS = 0
   ALREADY_SIGNED_INVALID = 1
   ALREADY_SIGNED_VALID = 2
   SERVER_ERROR = 3
   ERROR_NOT_A_SWI = 4 # We just check the version has "SWI_VERSION"
   ERROR_SWI_NOT_BLESSED = 5

def alreadySigned( swiFile ):
   ''' Returns a tuple of ( isSigned, validSignature ) where "isSigned"
       is True if the swiFile is signed and False otherwise, and 
       "validSignature" is True if the signature is valid, False otherwise'''
   signed = SwiSignLib.swiSignatureExists( swiFile )
   swiSigValid, _ = SwiSignLib.verifySwiSignature( swiFile )
   return ( signed, swiSigValid )

def querySignNotBlessed( force ):
   if force:
      return True
   question = "SWI is not blessed. Are you sure you wish to sign it? [y/N] "
   queryBless = raw_input( question )
   # Not signing is the default
   if queryBless.lower().startswith( 'y' ):
      return True
   return False 

class UPDATE_SIG_RESULT( Enum ):
   UPDATED_SIG = 0
   NO_SIG = 1
   INVALID_SIG = 2
   ERROR_UPDATING = 3
   ERROR_RELEASE_CERT = 4
   ERROR_UNKNOWN_CERT = 5

UPDATE_SIG_MESSAGE = {
   UPDATE_SIG_RESULT.UPDATED_SIG : "SWI signature updated.",
   UPDATE_SIG_RESULT.NO_SIG : "SWI not signed, signature update not needed.",
   UPDATE_SIG_RESULT.INVALID_SIG : "Invalid SWI signature, " \
                                   "signature update aborted.",
   UPDATE_SIG_RESULT.ERROR_UPDATING : "Failed to update SWI signature, please" \
                                      " re-sign the SWI.",
   UPDATE_SIG_RESULT.ERROR_RELEASE_CERT : "Signed with release CA, please manually" \
                                          " re-sign the SWI.",
   UPDATE_SIG_RESULT.ERROR_UNKNOWN_CERT : "Signed with an unknown certificate," \
                                          " please manually re-sign the SWI.",
}

def updateSwiSignature( swiFile, allowReleaseCA=False ):
   if not SwiSignLib.swiSignatureExists( swiFile ):
      return UPDATE_SIG_RESULT.NO_SIG

   swiSig = VerifySwi.getSwiSignatureData( swiFile )
   if swiSig is None:
      return UPDATE_SIG_RESULT.INVALID_SIG

   try:
      aristaCaX509 = VerifySwi.loadRootCert( VerifySwi.ARISTA_ROOT_CA_FILE_NAME )
      result = VerifySwi.checkSigningCert( swiSig, aristaCaX509 )
   except VerifySwi.X509CertException:
      return UPDATE_SIG_RESULT.INVALID_SIG

   if result == VerifySwi.VERIFY_SWI_RESULT.SUCCESS:
      if allowReleaseCA:
         result = sign( swiFile, forceSign=True )
         if result == SWI_SIGN_RESULT.SUCCESS:
            return UPDATE_SIG_RESULT.UPDATED_SIG
         else:
            return UPDATE_SIG_RESULT.ERROR_UPDATING
      return UPDATE_SIG_RESULT.ERROR_RELEASE_CERT

   if os.path.exists( VerifySwi.DEV_ROOT_CA_FILE_NAME ):
      devCaX509 = VerifySwi.loadRootCert( VerifySwi.DEV_ROOT_CA_FILE_NAME )
      result = VerifySwi.checkSigningCert( swiSig, devCaX509 )

      if result == VerifySwi.VERIFY_SWI_RESULT.SUCCESS:
         result = sign( swiFile, forceSign=True, useDevCA=True )
         if result == SWI_SIGN_RESULT.SUCCESS:
            return UPDATE_SIG_RESULT.UPDATED_SIG
         else:
            return UPDATE_SIG_RESULT.ERROR_UPDATING

   return UPDATE_SIG_RESULT.ERROR_UNKNOWN_CERT

def sign( swiFile, forceSign=False, useDevCA=False, devCaKeyPair=None ):
   signed, validSig = alreadySigned( swiFile )
   if signed:
      if not forceSign:
         if validSig:
            print( 'SWI is already signed with a valid signature.' )
            return SWI_SIGN_RESULT.ALREADY_SIGNED_VALID
         else:
            print( 'Warning: SWI is signed with an invalid signature.' )
            return SWI_SIGN_RESULT.ALREADY_SIGNED_INVALID
      else:
         # Force sign. Remove the swi-signature file from the SWI.
         Tac.run( [ 'zip', '-d', swiFile, SwiSignLib.SIG_FILE_NAME ] )
   blessed, version = Swi.getBlessedAndVersion( swiFile )
   if version is None:
      print( 'Error: Version file not found in SWI' )
      return SWI_SIGN_RESULT.ERROR_NOT_A_SWI
   if not useDevCA and not blessed and not querySignNotBlessed( forceSign ):
      print( "Not signing SWI that is not blessed." )
      return SWI_SIGN_RESULT.ERROR_SWI_NOT_BLESSED
   swiSignature = SignatureFile.Signature()
   swiData = SignatureFile.prepareDataForServer( swiFile, version, swiSignature )
   try:
      if useDevCA:
         signatureData = SignatureRequest.getDataFromDevCA( swiFile, swiData,
                                                        devCaKeyPair=devCaKeyPair )
      else:
         signatureData = SignatureRequest.getDataFromServer( swiFile, swiData )
      SignatureFile.generateSigFileFromServer( signatureData, swiFile, swiSignature )
   except SignatureRequest.SigningServerError, e:
      print( e )
      # Remove the null signature
      Tac.run( [ 'zip', '-d', swiFile, SwiSignLib.SIG_FILE_NAME ] )
      return SWI_SIGN_RESULT.SERVER_ERROR
   return SWI_SIGN_RESULT.SUCCESS

def signHandler( args=sys.argv[1:] ):
   parser = argparse.ArgumentParser( prog="swi sign" )
   parser.add_argument( "swi", metavar="EOS.swi",
                        help="Path of the SWI to be signed" )
   parser.add_argument( "--force-sign", help="Force signing the SWI",
                        action="store_true")
   parser.add_argument( "--dev-ca", help="Use development certificates for signing",
                        action="store_true" )

   args = parser.parse_args( args )
   returnCode = sign( args.swi, args.force_sign, args.dev_ca )
   exit( returnCode )
