#!/usr/bin/env python
# Copyright (c) 2008-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import optparse, os, re, socket, subprocess, sys, tempfile, time

from EosVersion import swimVariantFlavorMap, swiFlavorDefault
import EpochConsts

def run( argv, asRoot=False, captureStdout=False, verbose=True,
         ignoreReturnCode=False ):
   if asRoot:
      argv = ["sudo"] + list( argv )
   if verbose:
      sys.stderr.write( "+ %s\n" % " ".join( argv ) )
   p = subprocess.Popen( argv, stdout=subprocess.PIPE )
   stdoutdata = ""
   for line in iter( p.stdout.readline, b'' ):
      print line,
      if not ignoreReturnCode:
         assert not re.search( "exit status [1-9]", line )
      if captureStdout:
         stdoutdata += line
   p.wait()
   if not ignoreReturnCode:
      assert p.returncode == 0, "'%s' returned code %d" % ( " ".join( argv ),
                                                            p.returncode )
   if captureStdout:
      return stdoutdata

def getDocsToRemove( rootdir ):
   docsToRemove = run( [ "find", "%s/usr/share/doc" % rootdir,
                         "-iname", "change*", "-o",
                         "-iname", "news*", "-o",
                         "-iname", "demo*", "-o",
                         "-iname", "example*", "-o",
                         "-iname", "*faq*" ],
                       asRoot=True, captureStdout=True,
                       ignoreReturnCode=True ).splitlines()
   docsToRemove.append( "%s/usr/share/man" % rootdir )
   return docsToRemove

def removeDocs( rootdir ):
   oldcwd = os.getcwd()
   os.chdir( rootdir )
   for f in os.listdir( "." ):
      if f.endswith( ".dir" ):
         docsToRemove = getDocsToRemove( f )
         run( [ 'rm', '-rf' ] + docsToRemove, asRoot=True )
   os.chdir( oldcwd )

def installrootfs( rootdir, arch, packages, groups, yumConfig, erasePattern,
                   enableRepos, disableRepos, keepdoc=False ):
   assert os.path.exists( rootdir )
   rootdir = os.path.abspath( rootdir )
   tempdir = tempfile.mkdtemp()

   try:
      # Suppress glibc locales other than en_US.utf8 to save space
      file( os.path.join( tempdir, ".rpmmacros" ), "w" ).write(
         "%_install_langs C:en_US.utf8" )

      # Create essential device nodes and mountpoints in rootfs
      for d in ["/dev", "/proc", "/.deltas"]:
         run( ["mkdir", "-p", rootdir + d], asRoot=True )
      for n, a in [("/dev/null", ["c", "1", "3"]),
                   ("/dev/console", ["c", "5", "1"]),
                   ("/dev/random", ["c", "1", "8"]),
                   ("/dev/urandom", ["c", "1", "9"])]:
         if not os.path.exists( rootdir + n ):
            run( ["mknod", "--mode=0666", rootdir + n] + a, asRoot=True )
      run( ["mount", "-t", "proc", "proc", rootdir + "/proc"], asRoot=True )

      # yum install packages or groups into rootfs
      cmd = ["env", "HOME=%s" % tempdir]
      if yumConfig is None:
         cmd += ["a4", "yum"]
      else:
         cmd += ["yum", "-c", yumConfig]
      for repo in enableRepos:
         cmd += [ "--enablerepo=%s" % ( repo ) ]
      for repo in disableRepos:
         cmd += [ "--disablerepo=%s" % ( repo ) ]
      # package list too long may cause python pipe error, so break it into
      # shorter list
      for i in range( 0, len( packages ), 30 ):
         pkgs = packages[ i : i+30 ]
         run( cmd + [ "-y", "--installroot=%s" % rootdir,
                      "install" ] + pkgs, asRoot=True )
         # Make sure packages were really installed
         run( ["rpm", "--root=%s" % rootdir, "-q"] + pkgs )

      # Install yum groups
      if groups:
         run( cmd + [ "-y", "--installroot=%s" % rootdir, "install" ] + groups,
              asRoot=True )

      if arch == 'i686' and os.path.exists( rootdir + "/etc/rpm" ):
         with file( os.path.join( tempdir, "rpmplatform" ), "w" ) as f:
            # set rpm platform to x86_64 because customers want to install 64-bit
            # extensions on 32-bit EOS.
            f.write( "x86_64-redhat-linux\n" )
            f.flush()
         run( [ "cp", tempdir + "/rpmplatform", rootdir + "/etc/rpm/platform" ],
              asRoot=True )

      # Remove erasePackages
      erasePackages = [ p for p in run(
         ["rpm", "--root=%s" % rootdir, "-qa", "--qf=%{NAME} "],
         captureStdout=True ).split() if re.match( erasePattern, p ) ]
      if erasePackages:
         run( ["rpm", "--root=%s" % rootdir, "-e", "--nodeps"] + erasePackages,
               asRoot=True )

      # remove some docs to free up space
      docsToRemove = getDocsToRemove( rootdir )
      if not keepdoc:
         run( [ 'rm', '-rf' ] + docsToRemove, asRoot=True )

      # Clean up yum cache
      run( ["sh", "-c", "rm -rf %s/var/cache/yum/*" % rootdir], asRoot=True )

   finally:
      run( ["umount", rootdir + "/proc"], asRoot=True, ignoreReturnCode=True )
      run( ["rm", "-rf", tempdir], asRoot=True, verbose=False )

def installmodfs( moddir, arch, lowerdirs, pkgs, groups, yumConfig, erasePattern,
                  enablerepos, disablerepos, trace, keepdoc=False ):
   run( [ "mkdir", "-p", moddir ], asRoot=True )
   workdir = ""
   uniondir = ""
   try:
      if lowerdirs:
         workdir = '%s-work' % moddir
         uniondir = '%s-union' % moddir
         run( [ "mkdir", "-p", workdir, uniondir ], asRoot=True )
         run( [ "mount", "-t", "overlay", "overlay-%s" % moddir, "-o",
                "lowerdir=%s,upperdir=%s,workdir=%s" % ( lowerdirs, moddir,
                                                         workdir ), uniondir ],
              asRoot=True )
         # workaround for following yum issue with overlayfs
         #    Rpmdb checksum is invalid: dCDPT(pkg checksums): package_name
         for d in os.listdir( "%s/var/lib/rpm" % uniondir ):
            run( [ "touch", "%s/var/lib/rpm/%s" % ( uniondir, d ) ], asRoot=True )
         moddir = uniondir

      if pkgs or groups:
         installrootfs( moddir, arch, pkgs, groups, yumConfig, erasePattern,
                        enablerepos, disablerepos, keepdoc=keepdoc )
   finally:
      # show mount points to help with debugging
      run( [ "mount" ], asRoot=True )
      if lowerdirs and not trace:
         run( [ "umount", "-R", uniondir ], asRoot=True )
         if workdir or uniondir:
            run( [ "rm", "-rf", workdir, uniondir ], asRoot=True )

def writeSwiVersion( rootdir, version, arch, release, swiVariant='US',
                     swiFlavor=swiFlavorDefault,
                     swiMaxHwEpoch=EpochConsts.SwiMaxHwEpoch ):
   assert os.path.exists( rootdir )
   rootdir = os.path.abspath( rootdir )
   f = tempfile.NamedTemporaryFile()
   f.write( "SWI_VERSION=%s\n" % version )
   f.write( "SWI_ARCH=%s\n" % arch )
   f.write( "SWI_RELEASE=%s\n" % release )
   f.write( "BUILD_DATE=%s\n" % time.strftime( "%Y%m%dT%H%M%SZ", time.gmtime() ) )
   f.write( "BUILD_HOST=%s\n" % socket.gethostname() )
   f.write( "SERIALNUM=%s\n" %
            file( "/proc/sys/kernel/random/uuid" ).read().strip() )
   f.write( "SWI_MAX_HWEPOCH=%d\n" % swiMaxHwEpoch )
   f.write( "SWI_VARIANT=%s\n" % swiVariant )
   f.write( "SWI_FLAVOR=%s\n" % swiFlavor )
   f.flush()
   sys.stdout.write( file( f.name ).read() )
   run( [ "mkdir", "-p", rootdir + "/etc" ], asRoot=True )
   run( [ "cp", f.name, rootdir + "/etc/swi-version" ], asRoot=True )
   run( [ "chmod", "0644", rootdir + "/etc/swi-version" ], asRoot=True )

def setSwiVersion( rootdir, swi_version, swi_arch, swi_release ):
   arch = swi_arch or run( [ "arch" ], captureStdout=True ).rstrip( "\n" )
   swiVariant = "US"
   swiFlavor = swiFlavorDefault
   swi_variant_flavor_opts = os.path.basename( rootdir ).split( '.' )[ 0 ]
   if swi_variant_flavor_opts in swimVariantFlavorMap:
      swiVariant = swimVariantFlavorMap[ swi_variant_flavor_opts ][ 0 ]
      swiFlavor = swimVariantFlavorMap[ swi_variant_flavor_opts ][ 1 ]

   writeSwiVersion( rootdir, swi_version, arch, swi_release,
                    swiVariant=swiVariant, swiFlavor=swiFlavor )

def installrootfsHandler( args=sys.argv[1:] ):
   op = optparse.OptionParser(
         prog="swi installrootfs",
         usage="usage: %prog [OPTIONS] ROOTFSDIR" )
   op.add_option( "-l", "--lowerdirs", action="store",
                  help="directories to be union-mounted as lowerdirs, "
                       "separated by ':', must be in full path, "
                       "with top layer on left and bottom layer on right" )
   op.add_option( '-f', '--force', action='store_true',
                  help="clean up existing installation" )
   op.add_option( "-t", "--trace", action="store_true",
                  help="trace mode, no cleaning up intermediate files." )
   op.add_option( "--keepdoc", action="store_true",
                  help="keep doc directories" )
   op.add_option( "-p", "--package", action="append", default=[],
                  help="install PACKAGE (may be specified multiple times)" )
   op.add_option( "-g", "--group", action="append", default=[],
                  help="install GROUP @<yum-group-id> "
                       "(may be specified multiple times) " )
   op.add_option( "-c", "--yum-config", action="store", metavar="CONFIG",
                  help="use yum configuration file CONFIG" )
   op.add_option( "-e", "--erase-pattern", action="store", metavar="REGEX",
                  help="erase packages matching REGEX after yum install "
                  "(default=%default)" )
   op.add_option( "--swi-version", action="store",
                  help="set the SWI_VERSION in /etc/swi-version" )
   op.add_option( "--swi-arch", action="store",
                  help="set the SWI_ARCH in /etc/swi-version" )
   op.add_option( "--swi-release", action="store",
                  help="set the SWI_RELEASE in /etc/swi-version" )
   op.add_option( "--enablerepo", action="append",
                  help="enable repository (may be specified multiple times)" )
   op.add_option( "--disablerepo", action="append",
                  help="disable repository (may be specified multiple times)" )
   op.set_defaults( erase_pattern="$" )
   opts, args = op.parse_args( args )

   if len( args ) != 1:
      op.error( "Missing ROOTFSDIR argument" )
   if not opts.package and not opts.group:
      op.error( "Must specify at least one package or group to install" )
   if len( args ) == 1:
      rootdir = args[0]
   if opts.group:
      if not all( [ grp.startswith( "@" ) for grp in opts.group ] ):
         op.error( "Each GROUP argument must be prepended with @" )

   # Provide defaults for enablerepo and disablerepo, don't use
   # op.set_defaults(), as then anything provided by the user gets
   # appended to the defaults, the user specified ones does not
   # replace the defaults.
   #
   # XXX - 'local' and 'ToolsV2' repos don't exist outside of workspace,
   #       i.e. in Jenkins.
   if opts.enablerepo is None and opts.disablerepo is None:
      enablerepo = [ 'local' ]
      disablerepo = [ 'ToolsV2' ]
   else:
      enablerepo = [] if opts.enablerepo is None else opts.enablerepo
      disablerepo = [] if opts.disablerepo is None else opts.disablerepo

   arch = opts.swi_arch or run( [ "arch" ], captureStdout=True ).rstrip( "\n" )
   if opts.lowerdirs:
      if opts.force:
         # make sure to start with clean slate
         run( [ "rm", "-rf", rootdir ], asRoot=True )
      installmodfs( rootdir, arch, opts.lowerdirs, opts.package, opts.group,
                    opts.yum_config, opts.erase_pattern, enablerepo, disablerepo,
                    opts.trace, keepdoc=opts.keepdoc )
   else :
      installrootfs( rootdir, arch, opts.package, opts.group, opts.yum_config,
                     opts.erase_pattern, enablerepo, disablerepo,
                     keepdoc=opts.keepdoc )

   if opts.swi_version and rootdir:
      setSwiVersion( rootdir, opts.swi_version, opts.swi_arch, opts.swi_release )

if __name__ == "__main__":
   installrootfsHandler()
