#!/usr/bin/env python
# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import
from pprint import pprint
import os, re, sys
import rpm as baseRpmModule
import Swi, Swi.rpm, Swi.update
import tempfile
import argparse
import platform
from EosVersion import swiFlavor2GB, swiFlavorDPE, swiFlavorPdp, swiFlavorCloud
import subprocess

internationalPackages = [ 'Eos-international' ]

extraPkgs2GB = [ 'Bodega-EosImage-2gb' ]

# Copied from EosImage/Makefile.am
extraPkgsVEosLab = [ 'VEosSku-EosImage', 'Eos-veos', 'Eos-etba' ]

oliveFirmware = [ "Jericho2Fixed-olive-firmware", "DenaliModular-olive-firmware" ]

dpePackages = [ "Macsec-dpe", "Ipsec-dpe", "Jericho2Fixed-dpe", "Tunnelsec-dpe",
                "DenaliModular-dpe" ]

cloudPackages = [ "LicenseLoader-cloud", "LicenseLoader-azureCloudCerts" ]

bareMetalPackages = [ "Baremetal-EosImage" ]

# if the name of the cEOS host kernel rpm changes, please
# redefine this accordingly.
cEOSHostKernelRpm = 'EosKernel-ceos'

# List of rpms we need for the host OS on which
# the cEOS container will be instantiated.
cEOSHostRpms = [ cEOSHostKernelRpm ]

# List of rpms that need to be included in the
# cEOS docker container.
cEOSPackages = [ "CEosUtils-etc" ]

# list of platforms to be included in 2GB swi
systems2GB = "Cloverdale"

# Generate list of unused packages
# - 'platform' specifies a list of platforms
# - 'isWhiteList' specifies if the list of platforms is white-listed or black-listed
def excludePlatformPackages( excludePackagesDataPath, platforms, isWhiteList=True ):
   '''
   Generate a list of excluded packages of excluded platforms based on
   /etc/EosPkgExclude.d
   '''
   excludes = []
   for filename in os.listdir( excludePackagesDataPath ):
      if filename.endswith( '.exclude' ):
         match = re.search( platforms, filename )
         if ( ( isWhiteList and match ) or
              ( not isWhiteList and not match ) ):
            with open( os.path.join( excludePackagesDataPath, filename ) ) as f:
               excludes.append( set( f.read().splitlines() ) )

   if excludes:
      return list( set.intersection( *excludes ) )
   else:
      return []

def sharedExcludesDpe( excludePackagesDataPath, rootDir ):
   return oliveFirmware

def sharedExcludesCloud( excludePackagesDataPath, rootDir ):
   return excludePlatformPackages( excludePackagesDataPath, "Sfa|Sfe" )

def sharedExcludes2GB( excludePackagesDataPath, rootDir ):
   # Get RPMs installed
   cmdOut = Swi.runAndReturnOutput( [ "rpm", "--root=%s" % rootDir, "-qa",
                                      "--queryformat=%{NAME}\n" ],
                                      printStdout=False )
   rpmsInstalled = cmdOut.splitlines()
   sharedExcludes = excludePlatformPackages( excludePackagesDataPath, systems2GB )
   # BUG185629 - Ideally we would generate StrataApi* as part of
   # CheckExcludePackages.py. For now, add them manually to the list of
   # packages to remove
   sharedExcludes.extend( [ 'StrataApi', 'StrataApiV2', 'StrataApiNew' ] )
   # Exlcude Sand/Strata RPMs not needed for platforms in the systems2GB
   # string. These are added to the excludes list to reduce the size of
   # the EOS-2GB image.
   sharedExcludes.extend( [ 'SandSdkV4', 'SandSdkV4Helper', ] )

   # Remove all StrataSdkShim RPMs which aren't needed
   sdkRequired = [ 'StrataSdkShim-v1', 'StrataSdkShim-lib', ]
   sdkNonRequired = [ rpm for rpm in rpmsInstalled
                        if ( rpm.startswith( 'StrataSdkShim-' )
                             and rpm not in sdkRequired ) ]
   sharedExcludes.extend( sdkNonRequired )
   # Remove bf-* RPMs which are not needed on 2GB platforms
   sharedExcludes.extend( [ 'bf-drivers', 'bf-syslibs', 'bf-utils' ] )
   # Remove vEOS related RPMs
   sharedExcludes.extend( [ 'CloudHa', 'CloudHa-license', 'CloudVirtualEnv',
                            'KernelFirewall', 'KernelFirewall-lib' ] )
   # Remove bess & Sfe RPMs
   sharedExcludes.extend( [ 'BessMgr', 'BessMgr-lib', 'BessMgr-cli',
                            'Sfe-lib', 'Sfe-cli', 'SfeModules-lib', 'bess' ] )
   # Remove Platform Dependent RPMs not needed for 2GB platforms
   sharedExcludes.extend( [ 'TimestampKeyframe', 'TimestampKeyframe-lib',
                            'TimestampKeyframe-cli' , 'Sai', 'Sai-lib' ] )
   # Exclude Bodega-firmware since it is being replaced by Bodega-firmware-2gb
   # See BUG295869
   sharedExcludes.extend( [ 'Bodega-firmware' ] )

   # Remove some of the larger unnecessary rpms missed by the exclusion mechanism
   sharedExcludes.extend( [
   'EvoraApi',
   'PhyApm',
   'PhyAvago',
   'SflowAccel-firmware',
   'StrataFlowTracker',
   'StrataFlowTracker-cli',
   'StrataFlowTracker-lib',
   'StrataEm',
   'StrataEm-cli',
   'StrataEm-lib',
   'StrataInt-cli',
   'StrataInt-lib',
   'StrataNat-cli',
   'StrataNat-lib',
   'python-boto',
   ] )

   return sharedExcludes

def sharedExcludesVEosLab( excludePackagesDataPath, rootDir ):
   # We don't yet have a platform for vEOS-lab, so we cheat by using Sfe
   # and then removing the CloudEOS packages.
   sharedExcludes = excludePlatformPackages( excludePackagesDataPath, "Sfe" )
   sharedExcludes.extend( [ 'Sfe' ] )
   # These are copied from 2GB above
   # Remove vEOS related RPMs
   sharedExcludes.extend( [ 'CloudHa', 'CloudHa-license', 'CloudVirtualEnv',
                            'KernelFirewall', 'KernelFirewall-lib' ] )
   # Remove bess & Sfe RPMs
   sharedExcludes.extend( [ 'BessMgr', 'BessMgr-lib', 'BessMgr-cli',
                            'Sfe-lib', 'Sfe-cli', 'SfeModules-lib', 'bess' ] )
   return sharedExcludes


def getRpmsToExclude( rootDir, pkgsToExclude ):
   # Find out which RPMs are actually present in the swi ...
   rpmDbPath = os.path.normpath( os.path.join( rootDir, 'var/lib/rpm' ) )
   # pylint: disable=E1101
   baseRpmModule.addMacro( "_dbpath", rpmDbPath )
   ts = baseRpmModule.ts()
   swiRpms = set( [ rpm[ 'name' ] for rpm in ts.dbMatch() ] )
   # ... and only consider those for removal
   setDiff = set( pkgsToExclude ) - swiRpms
   if setDiff:
      print "-----------------------------------------------------------------------"
      print "Warning: there are RPMs we want to remove that are not in image"
      pprint( setDiff )
      print "-----------------------------------------------------------------------"
   return list( swiRpms.intersection( pkgsToExclude ) )

def getRepoList( opts ):
   # The default repository to be used in build environments is 'local'.
   # For AUTOTEST/STEST environments we should use the Abuild repository.
   # Also check for enablerepo & disablerepo args which can be used to
   # override these defaults
   if 'AUTOTEST' in os.environ or 'STEST' in os.environ or 'A4_MOCK' in os.environ:
      enablerepo = [ 'Abuild' ]
   else:
      enablerepo = [ 'local' ]

   enablerepo =  opts.enablerepo if opts.enablerepo else enablerepo
   disablerepo = opts.disablerepo if opts.disablerepo else []

   return enablerepo, disablerepo

def flavorCommon( opts, extraPkgs, extraExclude, sharedExcludesFunc,
                  swiMaxHwEpoch, image ):
   '''
   Generate a new flavor SWI from EOS.swi.  The extraPkgs parameter is used to
   specify a list of extra RPM packages that should be installed.  If it is None,
   then no extra packages are required.  If the swiMaxHwEpoch parameter is not None,
   then the SWI_MAX_HWEPOCH of EOS.swi is updated to the new value.
   '''
   excludePackagesPath = opts.excludePkgs
   tmpdir = tempfile.mkdtemp()
   print "Extracting %s to %s" % ( image, tmpdir )
   Swi.extract.extract( image, Swi.SwiOptions( dirname=tmpdir,
                                               use_existing=True ) )
   rootDir = os.path.join( tmpdir, 'rootfs-i386.dir' )
   if not opts.excludePkgs:
      # Default to using the /etc/EosPkgExclude.d files in the swi if caller
      # does not specify one
      print "excludePkgs not specified, using the one in %s" % image
      excludePackagesPath = os.path.join( rootDir, 'etc/EosPkgExclude.d' )

   sharedExcludes = sharedExcludesFunc( excludePackagesPath, rootDir )

   sharedRpmExcludes = getRpmsToExclude( rootDir, sharedExcludes )

   print "Cleaning up", tmpdir
   Swi.run( [ 'sudo', 'rm', '-rf', tmpdir ] )

   enablerepo, disablerepo = getRepoList( opts )

   if sharedRpmExcludes:
      rpmArgs = [ '--nodeps', '-e' ]
      rpmArgs.extend( sharedRpmExcludes )
      # Erase the packages that this flavor excludes
      eraseFunc = Swi.rpm.rpmInSwiFunc( *rpmArgs )
      # Update swi flavor in the version file, yumConfig,
      # enableRepos and disableRepos are not used if packages is passed
      # as None.
      updateFunc = Swi.update.swiUpdate( swiVariant=opts.swiVariant,
                                         swiFlavor=opts.swiFlavor,
                                         packages=extraPkgs, yumConfig=None,
                                         enableRepos=enablerepo,
                                         disableRepos=disablerepo,
                                         swiMaxHwEpoch=swiMaxHwEpoch,
                                         extraExclude=extraExclude )
      print "Erasing uneeded RPMs and updating version file"
      Swi.inSwi( image, [ eraseFunc, updateFunc ], outputfile=opts.output )
   else:
      print "Nothing to do. Unneeded RPMs have already been removed from %s" % image

def flavorWithCloud( opts, extraPkgs, extraExclude, swiMaxHwEpoch, image ):
   flavorCommon( opts, extraPkgs, extraExclude, sharedExcludesCloud,
                 swiMaxHwEpoch, image )

def flavorWith2GB( opts, extraPkgs, extraExclude, swiMaxHwEpoch, image ):
   flavorCommon( opts, extraPkgs, extraExclude, sharedExcludes2GB,
                 swiMaxHwEpoch, image )

def flavorWithVEosLab( opts, extraPkgs, extraExclude, swiMaxHwEpoch, image ):
   flavorCommon( opts, extraPkgs, extraExclude, sharedExcludesVEosLab,
                 swiMaxHwEpoch, image )

def flavorWithDpe( opts, extraPkgs, extraExclude, swiMaxHwEpoch, image ):
   flavorCommon( opts, extraPkgs, extraExclude, sharedExcludesDpe,
                 swiMaxHwEpoch, image )

def cleanup( workdir ):
   print "Cleaning up", workdir
   Swi.run( [ 'sudo', 'rm', '-rf', workdir ] )

# Remove unused packages
# - 'platform' specifies a list of platforms
# - 'isWhiteList' specifies if the list of platforms is white-listed or black-listed
def removeUnusedPackages( image, tmpdir, rootFsDir, platforms, isWhiteList=True ):
   excludePackagesPath = os.path.join( rootFsDir, 'etc/EosPkgExclude.d' )
   if not os.path.exists( excludePackagesPath ):
      print "No etc/EosPkgExclude.d in image %s" % image
   else:
      pkgsToExclude = excludePlatformPackages( excludePackagesPath,
                                               platforms,
                                               isWhiteList )
      cEOSRpmExcludes = getRpmsToExclude( rootFsDir, pkgsToExclude )
      if cEOSRpmExcludes:
         rpmArgs = [ '--nodeps', '-e' ]
         rpmArgs.extend( cEOSRpmExcludes )
         print "Excluding packages:"
         print cEOSRpmExcludes
         eraseFunc = Swi.rpm.rpmInSwiFunc( *rpmArgs )
         eraseFunc( tmpdir )

def repackCEosImage( rootFsDir, outputFile, tmpFile ):
   # Use --xattrs to ensure that capabilities (see "man capabilities")
   # are packed into the tarball.
   Swi.runAndReturnOutput( [ 'sudo', 'tar', '--xattrs', '-cf', tmpFile,
                             '-C', rootFsDir, '.' ] )
   # Install pxz if it doesn't exist
   try:
      Swi.run( [ 'which', 'pxz' ] )
   except AssertionError:
      Swi.run( [ 'a4', 'yum', 'install', '-y', 'pxz' ] )
   Swi.runAndReturnOutput( [ 'sudo', 'pxz', '-T30', '-k', tmpFile ] )
   Swi.runAndReturnOutput( [ 'sudo', 'mv', tmpFile + '.xz', outputFile ] )

def flavorWithCEOS( opts, image, extraPackages=[] ):
   tmpdir = tempfile.mkdtemp()
   outputFile = opts.output
   if not outputFile:
      outputFile = '/images/cEOS.tar.xz'
   Swi.extract.extract( image, Swi.SwiOptions( dirname=tmpdir, use_existing=True ) )
   rootFsDir = "%s/rootfs-i386.dir" % tmpdir

   enablerepo, disablerepo = getRepoList( opts )

   cEOSPackages.extend( extraPackages )

   updateFunc = Swi.update.swiUpdate( swiVariant=None, swiFlavor='cEOS',
                                      packages=cEOSPackages, yumConfig=None,
                                      enableRepos=enablerepo,
                                      disableRepos=disablerepo )
   updateFunc( tmpdir )

   systemConfFileName = os.path.join( rootFsDir, "etc/sysctl.d/99-eos.conf" )
   print " Removing core file setting in cEOS. The OS is responsible for this."
   Swi.run( [ 'sudo', 'sed', '-i', '/kernel.core/d', systemConfFileName ] )

   systemUnitFiles = [
         # Udevd is not supported currently by even latest release of systemd for
         # containers. Disabling udevd.
         "systemd-udevd-kernel.socket",
         "systemd-udevd-control.socket",
         "systemd-udev-trigger.service",
         "systemd-udev-settle.service",
         "systemd-udevd.service",

         # Disable Login and agetty services
         "console-getty.service",
         "getty.target",
         ]
   for sFile in systemUnitFiles:
      filePath = os.path.join( rootFsDir, "lib/systemd/system", sFile )
      Swi.run( [ 'sudo', 'rm', filePath ] )

   for pkg in cEOSHostRpms:
      localPkgRpm = "/RPMS/%s.%s.rpm" % ( pkg, platform.machine() )
      if os.path.exists( localPkgRpm ):
         Swi.run( [ 'sudo', 'cp', '-r', localPkgRpm, '%s/usr/share/' % rootFsDir ] )
      else:
         # '--downloadonly' would not take effect if package is already installed
         # so erase package first to make sure it would be downloaded
         # rpm would not be downloaded if it is in 'local' repository, so disable
         # local repository
         Swi.run( [ 'a4', 'yum', 'erase', '-y', pkg ] )
         Swi.run( [ 'a4', 'yum', 'install', '-y', '--disablerepo=local',
                    '--downloadonly', '--downloaddir=%s/usr/share/'
                    % rootFsDir, pkg ] )

   # Disable ZTP
   with tempfile.NamedTemporaryFile() as cfg:
      cfg.write( 'DISABLE=True\n' )
      cfg.flush()
      flashDir = os.path.join( rootFsDir, 'mnt', 'flash' )
      Swi.run( [ 'sudo', 'mkdir', '-p', '%s' % flashDir ] )
      Swi.run( [ 'sudo', 'cp', cfg.name,
         os.path.join( flashDir, 'zerotouch-config' ) ] )

   # Systemd-poweroff.service is executed when RTMIN+4 is received by systemd.
   # We configured by default that RTMIN+4 will be sent to cEOS when a user
   # types "docker stop ceos", effectively rebooting the system if cEOS
   # is stopped.
   with tempfile.NamedTemporaryFile() as powerCycleSystemdUnit:
      powerCycleSystemdUnit.write( "[Unit]\nDescription=Power-Off\n\n"
                                   "[Service]\nType=oneshot\n"
                                   "ExecStart=/usr/bin/powercycle" )
      powerCycleSystemdUnit.flush()
      Swi.run( [ 'sudo', 'cp', powerCycleSystemdUnit.name,
                 '%s/lib/systemd/system/systemd-poweroff.service' % rootFsDir ] )

   if opts.swiFlavor == 'cEOSAristaSAI':
      # We need to enable the AristaSAI toggle just for this image
      saiEnable = [ "sudo", "sed", "-i", "s/\"enabled\":false/\"enabled\":true/",
                    "%s/etc/toggles.d/AristaSAI.json" % rootFsDir ]
      Swi.runAndReturnOutput( saiEnable )
      # we only care about Strata for now
      removeUnusedPackages( image, tmpdir, rootFsDir, "Sand|Sfa|Sfe|Bfn|Xp" )

   # check EOS.swi and EosKernel-ceos compatibility
   try:
      if opts.swiFlavor == 'cEOS' or opts.swiFlavor == 'cEOSAristaSAI':
         stdoutdata = Swi.runAndReturnOutput( [ 'sudo', 'rpm', '-q', '-p',
                        '%s/usr/share/%s.%s.rpm'
                        % ( rootFsDir, cEOSHostKernelRpm, platform.machine() ),
                        '--info' ] )

         lines = stdoutdata.split( '\n' )
         kernelVer = ''
         kernelRel = ''
         for line in lines:
            tokens = line.split( ':' )
            name = tokens[0].strip()
            if name == 'Version':
               kernelVer = tokens[1].strip()
            elif name == 'Release':
               kernelRel = tokens[1].strip()

         kernelModulePath = "%s/usr/lib/modules/%s-%s" % ( rootFsDir, kernelVer, 
                                                           kernelRel )
         assert os.path.exists( kernelModulePath ), \
            "Error: %s not found. EOS.swi and EosKernel-ceos.%s.rpm mismatch." \
            % ( kernelModulePath, platform.machine() )
         Swi.run( [ 'sudo', 'cp', '-r', '%s/psp' % tmpdir, 
                    '%s/usr/share/' % rootFsDir ] )

      etcCeosPath = "%s/etc/cEOS-release" % rootFsDir
      assert os.path.exists( etcCeosPath ) , "Error: /etc/cEOS-release not found."
      # Iptables are for temporary hack to get ssh working.
      # Please refer to BUG188760.
      # Also disabling reboot commands inside the container, since semantics don't
      # really make sense.
      filesToDelete = [ 'lib/systemd/system/iptables.service',
                        'lib/systemd/system/ip6tables.service',
                        'usr/sbin/reboot',
                        'usr/sbin/poweroff',
                        'usr/sbin/shutdown',
                        'usr/sbin/halt' ]
      for fileToDelete in filesToDelete:
         Swi.run( [ 'sudo', 'rm', os.path.join( rootFsDir, fileToDelete ) ] )

      tarFile = tempfile.NamedTemporaryFile().name
      repackCEosImage( rootFsDir, outputFile, tarFile )
      Swi.run( [ 'sudo', 'rm', tarFile ] )
   except AssertionError:
      # Debug info for BUG239880 before cleanup
      # Are we out of space in our Autotest workspaces?
      _outputDf = Swi.runAndReturnOutput( [ 'df', '-h' ] )
      _outputDu = Swi.runAndReturnOutput( [ 'du', '-h', '/tmp' ] )
      raise
   finally:
      cleanup( tmpdir )

def flavorWithCEOSSingleHW( opts, image ):
   outputFile = opts.output
   if not outputFile:
      outputFile = image
   pwd = os.getcwd()
   rootFsDir = "rootfs-i386.dir"
   Swi.run( [ 'sudo', 'mkdir', '-p', rootFsDir ] )
   Swi.run( [ 'sudo', 'tar', '-Jxf', image , '-C', rootFsDir,
              '--warning=no-timestamp' ] )
   if not opts.keep:
      Swi.run( [ 'sudo', 'rm', '-f', image ] )
   rootFsDir = os.path.abspath( rootFsDir )

   removeUnusedPackages( image, pwd, rootFsDir, opts.hw, True )
   repackCEosImage( rootFsDir, outputFile, outputFile + '.tmp' )
   cleanup( rootFsDir )
   Swi.runAndReturnOutput( [ 'sudo', 'rm', '-f', outputFile + '.tmp' ] )

def flavorWithCEOSLab( opts, image ):
   outputFile = opts.output
   if not outputFile:
      opts.output = '/images/cEOS-lab.tar.xz'
   flavorWithCEOS( opts, image )

def flavorWithCEOSR( opts, image ):
   outputFile = opts.output
   if not outputFile:
      opts.output = '/images/cEOSR.tar.xz'
   rootFsDir = tempfile.mkdtemp()
   Swi.run( [ 'sudo', 'tar', '-Jxf', image, '-C', rootFsDir,
              '--warning=no-timestamp' ] )

   unusedPackages = [ 'EosInit-veos', 'LinuxBridge', 'LinuxBridge-lib' ]
   # remove all etba packages
   etbaPackages = subprocess.check_output( [ "sudo", "rpm", "--root=%s" % rootFsDir,
                                             "-qa", r"*etba*" ] )
   if etbaPackages:
      unusedPackages += etbaPackages.splitlines()
   Swi.run( [ "sudo", "rpm", "--root=%s" % rootFsDir, "--nodeps", "-e" ] +
            unusedPackages )
   additonalPkgs = [ 'KubeController' ]
   Swi.run( [ "a4", "yum", "--enablerepo=local", "--installroot=%s" % rootFsDir, 
            "install" , "-y" ] + additonalPkgs )
   repackCEosImage( rootFsDir, outputFile, outputFile + '.tmp' )
   Swi.run( [ 'sudo', 'rm', '-f', outputFile + '.tmp' ] )
   cleanup( rootFsDir )

def flavorWithExtraPkgs( opts, extraPkgs, extraExclude, swiMaxHwEpoch, image ):
   '''
   Generate a new flavor SWI from EOS.swi.  The extraPkgs parameter is used to
   specify a list of extra RPM packages that should be installed.  If it is None,
   then no extra packages are required.  If the swiMaxHwEpoch parameter is not None,
   then the SWI_MAX_HWEPOCH of EOS.swi is updated to the new value.
   '''

   enablerepo, disablerepo = getRepoList( opts )

   updateFunc = Swi.update.swiUpdate( swiVariant=opts.swiVariant,
                                      swiFlavor=opts.swiFlavor,
                                      packages=extraPkgs, yumConfig=None,
                                      enableRepos=enablerepo,
                                      disableRepos=disablerepo,
                                      swiMaxHwEpoch=swiMaxHwEpoch,
                                      extraExclude=extraExclude )

   print "Updating image, %s, with flavor=%s, swiVariant=%s" % \
      ( image, opts.swiFlavor, opts.swiVariant )
   Swi.inSwi( image, [ updateFunc ], outputfile=opts.output )

def flavorHandler( args=sys.argv[ 1: ] ):
   op = argparse.ArgumentParser(
      prog="swi flavor",
      description="Create a flavor of EOS from full EOS.swi",
      usage="usage: %(prog)s EOS.swi [--with swiflavor] [-o target] " +
            "[--swiVariant International] [--hw product] [--excludePkgs path] " )
   op.add_argument( '--with',
                    help='desired flavor',
                    dest='swiFlavor' )
   op.add_argument( "--enablerepo", action="append",
                    help="enable repository (may be specified multiple times)" )
   op.add_argument( "--disablerepo", action="append",
                    help="disable repository (may be specified multiple times)" )
   op.add_argument( '-o', '--output',
                    help="target swi for swiFlavor EOS",
                    action='store' )
   op.add_argument( '--swiVariant', action='store',
                    help="Update EOS SWI variant" )
   op.add_argument( '--hw', action='store',
                    help="product for which cEOS image size should be optimized" )
   op.add_argument( '--excludePkgs',
                    help="path to dir containing package exclude info, "
                         "default uses /etc/EosPkgExclude.d/ within EOS.swi",
                    action='store' )
   op.add_argument( '-k', '--keep',
                    help="keep the original cEOS image",
                    action='store_true' )

   opts, args = op.parse_known_args( args )
   if len( args ) != 1:
      op.error( 'Please give me exactly one swi file!' )
   extraExclude = None
   if opts.swiFlavor == None:
      assert opts.swiVariant == 'International'
      flavorWithExtraPkgs( opts, internationalPackages,
                           extraExclude, None, *args )
   elif opts.swiFlavor == swiFlavorCloud:
      # Include BareMetal RPMs in the CloudEOS.swi
      extraPkgs = cloudPackages + bareMetalPackages
      extraExclude = None
      if opts.swiVariant == 'International':
         extraPkgs += internationalPackages
      flavorWithCloud( opts, extraPkgs, extraExclude, None, *args )
   elif opts.swiFlavor == swiFlavor2GB:
      extraPkgs = extraPkgs2GB
      if opts.swiVariant == 'International':
         extraPkgs += internationalPackages
      flavorWith2GB( opts, extraPkgs, extraExclude, None, *args )
   elif opts.swiFlavor == swiFlavorDPE:
      assert opts.swiVariant != 'International'
      flavorWithDpe( opts, dpePackages, None, None, *args )
   elif opts.swiFlavor == swiFlavorPdp:
      assert opts.swiVariant != 'International'
      flavorWithExtraPkgs( opts, None, None, None, *args )
   elif opts.swiFlavor == swiFlavorPdp + '-' + swiFlavorDPE:
      assert opts.swiVariant != 'International'
      flavorWithDpe( opts, dpePackages, None, None, *args )
   elif opts.swiFlavor == swiFlavor2GB + '-' + swiFlavorPdp:
      assert opts.swiVariant != 'International'
      flavorWith2GB( opts, extraPkgs2GB, None, None, *args )
   elif opts.swiFlavor == 'vEOS-lab':
      # Until we make sure that we delete all of the packages
      # that we don't want to distribute.  For now this feature
      # is just to generate images to run in autotest.
      print '-------------------------------------------------------'
      print 'Warning: images generated this way are only for testing'
      print '         and should not be distributed to customers'
      print '-------------------------------------------------------'
      flavorWithVEosLab( opts, extraPkgsVEosLab, None, None, *args )
   elif opts.swiFlavor == 'cEOS':
      flavorWithCEOS( opts, *args )
   elif opts.swiFlavor == 'cEOSLab':
      flavorWithCEOSLab( opts, *args )
   elif opts.swiFlavor == 'cEOSR':
      flavorWithCEOSR( opts, *args )
   elif opts.swiFlavor == 'cEOSSingleHW':
      if not opts.output and opts.keep:
         op.error( 'Cannot use --keep without --output' )
      flavorWithCEOSSingleHW( opts, *args )
   elif opts.swiFlavor == 'cEOSAristaSAI':
      flavorWithCEOS( opts, *args,
                      extraPackages=[ "AristaSAI-lib",
                                      "AristaSAI",
                                      "sonic-sairedis" ] )
   else:
      op.error( 'Invalid flavor!' )

if __name__ == "__main__":
   flavorHandler()
