#!/usr/bin/env python
# Copyright (c) 2008-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import errno, os, re, sys
import glob
import optparse
import subprocess
from Swi import installrootfs

def mksquashfs( f, outfilename, fast, rootfsRpmDbOnly ): 
   # Run mksquashfs as root so that it has permissions to read all the
   # files in the rootfs directory.
   squashfsArgs = [ "sudo", "mksquashfs", f, outfilename,
                    "-no-progress", "-noappend" ]
   # If fast is not specified, compress the swi (this is slow!)
   if not fast:
      squashfsArgs += [ "-comp", "xz", "-Xbcj", "x86" ]
   # only include rpmdb of top layer
   if rootfsRpmDbOnly and ( "rootfs" not in f ) and \
         os.path.exists( "%s/var/lib/rpm" % f ):
      subprocess.check_call( [ "sudo", "mv", "%s/var/lib/rpm" % f,
                               "%s.rpmdir" % f ] )
   subprocess.check_call( squashfsArgs )
   if rootfsRpmDbOnly and ( "rootfs" not in f ) and \
         os.path.exists( "%s.rpmdir" % f ):
      subprocess.check_call( [ "sudo", "mv", "%s.rpmdir" % f,
                               "%s/var/lib/rpm" % f ] )
   # An unfortunate side-effect of running mksquashfs as root is that the
   # output file is owned by root, so chown it to the user running the
   # script.
   uid = os.geteuid() 
   gid = os.getegid()
   subprocess.check_call( [ "sudo", "chown", "%d:%d" %(uid, gid),
                            outfilename ] ) 

def createSquashfs( dir, opts ):
   installrootfs.removeDocs( dir )
   try:
      oldcwd = os.getcwd()
   except OSError, e:
      if e.errno != errno.ENOENT:
         raise
      oldcwd = "/tmp"
   os.chdir( dir )
   filesToZip = set()
   filesToExclude = set()
   for f in os.listdir( "." ):
      if f.endswith( ".dir" ):
         if not os.path.exists( "%s/bin/bash" % f ):
            if opts and opts.force:
               pass
            else:
               print "%s/%s/bin/bash does not exist;" % ( dir, f ), \
                     "is this really a suitable image?"
               print "Use --force to force."
               return

         base = f[:-4]
         # In the case where you extract a swi containing rootfs-i386 and then
         # use that extracted data as the source to create a swi with a
         # squashfs rootfs you don't want the new swi to contain the original
         # rootfs-i386.
         filesToExclude.add( base )
         # squash content of <variant/flavor>.rootfs-i386.dir into
         # <variant/flavor>.rootfs-i386.sqsh
         outfilename = base + ".sqsh"
         try:
            os.unlink( outfilename )
         except OSError, e:
            if e.errno != errno.ENOENT:
               sys.stderr.write( "%s: %s\n" %( outfilename, e ))
               sys.exit( 1 )
         if not opts.modules or os.path.exists( "%s/boot" % f ):
            for f1 in os.listdir( "%s/boot" % f ):
               m = re.match( r"vmlinuz-EosKernel(-kdump|)", f1 )
               if m:
                  dst = "linux-i386" + m.group( 1 )
                  subprocess.check_call( [ "sudo", "cp", "%s/boot/%s" % ( f, f1 ),
                                           dst ] )
                  filesToZip.add( dst )
         if not opts.modules or os.path.exists( "%s/etc/swi-version" % f ):
            subprocess.check_call( [ "sudo", "cp", "%s/etc/swi-version" % f,
                                     "version" ] )
            filesToZip.add( "version" )
         # Now add files as listed in swi-meta dir. We expect files to 
         # contain just 2 lines with first line being path of the file in the rootfs
         # and second line listing the destination file name as it would appear in 
         # the swi file. The original file is also contained in the embedded 
         # squashfs. 
         if not opts.modules or os.path.exists( "%s/etc/swi-metadata" % f ):
            for specFileName in glob.glob( "%s/etc/swi-metadata/*" % f ):
               with file( specFileName ) as specFile : 
                  contents = specFile.read().strip().split( '\n' )
                  assert len( contents ) == 2
                  filePath = os.path.normpath( '/' + contents[ 0 ] )
                  nameInZip = os.path.normpath( contents[ 1 ] )
                  subprocess.check_call( [ "sudo", "cp", f + filePath, nameInZip ] )
                  filesToZip.add( nameInZip )
         mksquashfs( f, outfilename, opts and opts.fast, opts.modules )
         filesToZip.add( outfilename )
      else:
         if os.path.isdir( f ):
            for root, _, files in os.walk( f ):
               for name in files:
                  filesToZip.add( os.path.join( root, name ) )
         elif not opts.hdrfile or f is not opts.hdrfile:
            filesToZip.add( f )
   os.chdir( oldcwd )
   return filesToZip - filesToExclude

def create( filename, opts=None ):
   if opts and opts.dirname:
      dir = opts.dirname
   else:
      dir = os.path.basename( filename ).rsplit('.', 1)[0]

   # Store files in EOS.swi sorted by size to speed extraction of individual files
   # (stage 0 Aboot needs only initrd and linux)
   files = sorted( createSquashfs( dir, opts ),
                   key=lambda f: os.stat( os.path.join( dir, f ) ).st_size )
   if opts.hdrfile:
      hdrFileOut = 'swimSqshMap'
      files = [ hdrFileOut ] + files
      # header file is formatted for use by swim build process with lines
      # in the following format
      #   <flavor>=<path>/<moduleA>.dir[:<path>/<moduleN>.dir]*
      # rename .dir to .sqsh here for use by swim adaptor and boot process with lines
      # in the following format
      #   <flavor>=<moduleA>.sqsh[:<moduleN>.sqsh]*
      oldcwd = os.getcwd()
      os.chdir( dir )
      subprocess.check_call( [ "rm", "-f", "%s.tmp" % opts.hdrfile ] )
      subprocess.check_call( [ "mv", opts.hdrfile, "%s.tmp" % opts.hdrfile ] )
      with open( hdrFileOut, "w" ) as outhdrfile:
         with open( opts.hdrfile + ".tmp", "r" ) as inhdrfile:
            for line in inhdrfile:
               key, value = line.split( '=', 1 )
               dirs = value.split( ':' )
               sqshes = []
               for d in dirs:
                  sqshes += [ os.path.basename( d ).replace( 'dir', 'sqsh' ) ]
               value = ':'.join( sqshes )
               line = '%s=%s' % ( key, value )
               outhdrfile.write( line )
      os.chdir( oldcwd )

   if not opts.installfs_only:
      try:
         # Don't unlink since creating new file needs write permission to parent
         # directory. write mode anyway truncates the file for us.
         outfile = file( filename, "w" )
      except Exception, e:
         sys.stderr.write( "while writing to %s: %s\n" %( filename, e ))
         sys.exit( 1 )

      subprocess.check_call( [ "zip", "-", "-0" ] + files, stdout=outfile, cwd=dir )

def createHandler( args=sys.argv[1:] ):
   op = optparse.OptionParser(
         prog="swi create",
         usage="usage: %prog [-d dir] [-f] [-c] [-m [--hdrfile hdrfile]] EOS.swi" )
   op.add_option( '-d', '--dirname', action='store' )
   op.add_option( '--hdrfile', action='store', help="first file in image" )
   op.add_option( '-m', '--modules', action='store_true',
                  help="create EOS image modules" )
   op.add_option( '-f', '--force', action='store_true' )
   op.add_option( '-t', '--trace', action='store_true', 
                  help="trace mode, no cleaning up of intermediate files" )
   op.add_option( '--fast', action='store_true' )
   op.add_option( '--installfs-only', action='store_true',
                  help="Only install rpms, but do not generate squashfs and image." )
   op.add_option( '-s', '--squashfs', action='store_true',
                  help="Create the root filesystem in squashfs format.  This "
                       "is the default, and thus this option is useless." )
   op.set_defaults( squashfs=True )
   op.set_defaults( fast=False )
   opts, args = op.parse_args( args )

   if not opts.installfs_only and len( args ) != 1:
      op.error( 'Please give me exactly one swi file!' )

   if opts.hdrfile and not opts.modules:
      op.error( '--hdrfile must be used together with -m or --modules' )

   create( args[0], opts )

if __name__ == "__main__":
   createHandler()
