# Copyright (c) 2008, 2009 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

bindMounts = ["/proc", "/selinux", "/sys", "/dev", "/dev/pts", "/dev/shm",
              "/tmp/.X11-unix"]

import sys, os, subprocess

def runMountedCmd( swiRootDir, mounted, lazy, args ):
   '''Run a mount or umount command.
   If mounted is True, runs the mount command, otherwise runs the umount command.
   If lazy is True, perform lazy unmounting. Irrelevant when mounting.
   args are the file arguments to a mount command. umount will use a subset of this.
   Returns whether or not the command was successful.'''
   execPrefix = os.environ.get( "EXECPREFIX" )
   cmd = []
   if execPrefix:
      cmd.append( execPrefix )
   if mounted:
      cmd += [ "%s/bin/mount" % swiRootDir, "-n", "--bind" ] + args
   else:
      cmd += [ "%s/bin/umount" % swiRootDir, "-n" ] + \
          ( [ "-l" ] if lazy else [] ) + [ args[ 1 ] ]
   process = subprocess.Popen( cmd, stderr=subprocess.PIPE )
   errorData = process.communicate()[ 1 ]
   if process.wait() != 0:
      # don't fail if the specified directory wasn't mounted to begin with.
      # Assume the specified directory has been unmounted already.
      if mounted or "not mounted" not in errorData:
         if mounted:
            sys.stderr.write( "bind-mounting %s on %s failed: %s\n" %
                              ( args[ 0 ], args[ 1 ], errorData ) )
         else:
            sys.stderr.write( "unmounting %s failed: %s\n" %
                              ( args[ 1 ], errorData ) )
         return False
   return True

def yumReposMountedIs( rootDir, mounted, lazy=True ):
   '''If there is repo mount information in the default yum repository, perform
   the specified mounts/unmounts.'''
   repoMountsFile = os.path.join( rootDir, "etc", "yum.repos.d", "repoMounts" )
   success = True
   if os.path.exists( repoMountsFile ):
      with open( repoMountsFile, "r" ) as repoMounts:
         for line in repoMounts:
            # [dir to mount] --> [dir to mount over]
            repoData = line.strip().split( " --> " )
            if len( repoData ) == 2:
               if not runMountedCmd( rootDir, mounted, lazy, repoData ):
                  success = False
   return success

def mountedIs( dir, mounted, lazy=True ):
   mounts = bindMounts
   if not mounted: mounts = reversed(mounts)
   success = True
   for bm in mounts:
      if not os.path.isdir( bm ): continue
      target = os.path.join( dir, bm[1:] )
      if os.path.isdir( target ):
         if os.path.samefile( bm, target ) == mounted:
            continue
      else:
         if mounted:
            os.makedirs( target )
         else:
            # The mount point does not even exist.  So there is no way it can be
            # mounted!
            continue
      # swi-wrapper sets PREFIX to the path to the chroot and EXECPREFIX to
      # PREFIX/ld-linux.so.2 so that I can use it here to execute mount & umount.
      # This is necessary when the arch outside the chroot differs from the arch
      # inside, e.g. when using an i386 EOS-devel on an x86_64 machine.
      prefix = os.environ.get( "PREFIX", "" )
      if not runMountedCmd( prefix, mounted, lazy, [ bm, target ] ):
         success = False
   success = yumReposMountedIs( dir, mounted, lazy ) and success
   return success

def chroot( dir, inMountNamespace ):
   
   if not inMountNamespace:
      # If swi chroot does not have its own mount namespace, make it so. This is so
      # that a rm -rf outside the chroot won't see that, say, /dev is mounted within
      # the swi chroot at "dir"/dev and try to delete those files.
      os.execvp( "chns", [ "chns", "-m" ] + sys.argv + [ "--in-mount-namespace" ] )
      sys.exit( 1 )

   sh = os.path.join( dir, "bin/sh" )
   if not os.path.exists( sh ):
      sys.stderr.write( "%s: not a plausible chroot environment (%s not found)\n" %(
         dir, sh ))
      sys.exit( 1 )
   if not mountedIs( dir, True ):
      sys.exit( 1 )
   try:
      os.chroot( dir )
      os.chdir( "/" )
   except OSError, e:
      sys.stderr.write( "chroot: %s\n" % e )
      sys.exit( 1 )
   try:
      os.environ[ "INSTALL_AROSTEST_PLUGINS" ] = "no"
      if "A4_CHROOT" in os.environ:
         # Update A4_CHROOT, if it exists, as it points to the outer directory which
         # will not be accessible in this chroot.
         os.environ[ "A4_CHROOT" ] = os.getcwd()
      os.execvp( "su", [ "su", "root"] )
   except OSError, e:
      sys.stderr.write( "execv: %s\n" % e )
      sys.exit( 1 )

def umount( dir ):
   if not mountedIs( dir, False ):
      sys.exit( 1 )

def chrootOrUmountHandler( args, cmd ):
   if os.geteuid() != 0:
      os.execvp( "sudo", [ "sudo", "swi", cmd ] + args )
   import optparse
   op = optparse.OptionParser(
         prog="swi %s" % cmd,
         usage="usage: %prog dir|swi" )
   # Don't use this option; it is used by the command internally to denote whether
   # or not swi chroot is in its own mount namespace (it has no effect on swi
   # umount).
   op.add_option( "--in-mount-namespace", dest="inMountNamespace",
                  action="store_true", help=optparse.SUPPRESS_HELP )
   opts, args = op.parse_args( args )

   if len( args ) != 1:
      op.error( 'missing directory or swi-file to %s' % cmd )
   dir = args[0]
   found = None
   if os.path.exists( os.path.join( dir, "bin/sh" )):
      found = dir
   else:
      # maybe "dir" is not really a directory --- maybe it is a swi file,
      # or the directory the swi was exploded into.
      import glob
      pattern = os.path.splitext( dir )[0] + "/rootfs*.dir"
      globbed = glob.glob( pattern )
      if len(globbed) > 0:
         dir2 = sorted(globbed)[0]
         if os.path.isdir( dir2 ):
            found = dir2
   if not found:
      sys.stderr.write( '%s: no chroot environment found\n' %( dir, ))
      sys.exit( 1 )
   # If $HOME is set, bind-mount it too -- it's very useful
   # to have it inside the chroot.
   if 'HOME' in os.environ:
      bindMounts.append( os.environ[ 'HOME' ] )
   if cmd == 'chroot':
      chroot( found, opts.inMountNamespace )
   elif cmd == 'umount':
      umount( found )
   else:
      assert False, "bad command: %s" % cmd

def chrootHandler( args ):
   chrootOrUmountHandler( args, "chroot" )

def umountHandler( args ):
   chrootOrUmountHandler( args, "umount" )
   
