# Copyright (c) 2008, 2009 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os
import re
import Swi
import glob
from optparse import OptionParser, SUPPRESS_HELP
from collections import namedtuple

CopyTriplet = namedtuple( 'CopyTriplet', [ 'file', 'perm', 'dest' ] )

USAGE = \
'''%prog <swi-file> [ (<file> <perm> <dest>)+ [--nobash] ]

This will extract the swi, chroot into it, mount your home dir under /ws/ and start
bash so you can massage the image (edit file, copy files from your workspace) and
when you exit bash it will repackage the swi with the new changes.

For non-interactive mode, use --nobash
This will copy the given files and repack the image without any intervention.
If --nobash is used, but no files are specified, no work will be done.

Example:
swi bash /images/EOS.swi ~/proj/usr/bin/bla 755 /usr/bin/bla
  ~/proj/comp/CliPlugin/bla.py 644 /usr/lib/python2.7/site-packages/CliPlugin/

For the <swi-file>, enter . as shortcut to /images/EOS.swi, and .i for EOS-INT.swi.
For <perm>, enter . for 777, .. for 755 (exe), ... for 644 (non-exe).'''

COPYING = '\nCopying the files that you passed.'
DEFAULT_INT_SWI = '/images/EOS-INT.swi'
DEFAULT_SWI = '/images/EOS.swi'
FILE_NO_EXIST = '<%s> does not seem to exist'
INVALID_MODE = 'Invalid UNIX-style permissions/mode. Got <%s>'
NO_SWI_FILE = 'missing argument: swi file'
NOT_SWI_FILE = '<%s> is not a .swi file'
NOT_A_TRIPLET = 'expected a triplet, but got <%s> <%s> <%s>'
ROOTFS = 'rootfs-*.dir'
WELCOME_INTERACTIVE = '''
A new shell was created.
Massage the image (you are in its root dir),
then 'exit' to start its re-packaging,
or 'exit -1' to skip the packaging.
Your %s has been mounted to /ws/'''
NOTHING_TO_DO = \
'''Cowardly refusing to proceed.
'--nobash' (non-interactive) is enabled, yet no files were provided to be copied.'''
SWI_PROMPT = \
r'''$(if [[ $? == 0 ]]; then echo -e "\e[00;32m"; else echo -e "\e[1;31m"; fi)
[SWI \u@\h \W]\e[00m> '''

def maybeRerunAsRoot( args ):
   ''' Runs the same command as sudo if not su. '''
   if os.geteuid() != 0:
      os.execvp( 'sudo', [ 'sudo', 'swi', 'bash' ] + args )

def parseOptions( args ):
   op = OptionParser( prog='swi bash',
                      usage=USAGE )
   op.add_option( '--nobash',
                  dest = 'interactive',
                  action = 'store_false',
                  default = True,
                  help = SUPPRESS_HELP )
   opts, args = op.parse_args( args )

   return opts, args, op

def validateSwiArg( optParser, args ):
   ''' Makes sure that first argument is an existing .swi file. '''
   if not len( args ):
      optParser.error( NO_SWI_FILE )
   filename = args.pop( 0 )
   if filename == '.':
      filename = DEFAULT_SWI
   elif filename == '.i':
      filename = DEFAULT_INT_SWI
   elif not filename.endswith( '.swi' ):
      optParser.error( NOT_SWI_FILE % filename )
   if not os.path.exists( filename ):
      optParser.error( FILE_NO_EXIST % filename )
   return filename

def validatePerm( op, candidate ):
   ''' Makes sure that mode/perm is valid like 0777, 1234, 655, etc.
       Per spec, it replaces . with 777, .. with 755, and ... with 644. '''
   if candidate == '.':
      return '777'
   if candidate == '..':
      return '755'
   if candidate == '...':
      return '644'
   if not re.match( "^[0-7]?[0-7]{3}$", candidate ):
      op.error( INVALID_MODE % candidate )
   return candidate

def validateCopyArgs( optParser, args ):
   ''' Makes sure that arguments for file copy come in triples.
       Checks that arg1 is a file and that arg2 is valid UNIX mode/permission. '''
   result = list()
   iterator = iter( args )
   for fyle in iterator:
      if not os.path.exists( fyle ):
         optParser.error( FILE_NO_EXIST % fyle )
      perm = '???'
      dest = '???'
      try:
         perm = validatePerm( optParser, iterator.next() )
         dest = iterator.next()
         triplet = CopyTriplet( fyle, perm, dest )
         result.append( triplet )
      except StopIteration:
         optParser.error( NOT_A_TRIPLET % ( fyle, perm, dest ) )
   return result

def chdirToImg( workDir ):
   os.chdir( workDir )
   os.chdir( glob.glob( ROOTFS )[ 0 ] )

def copyFiles( workDir ):
   ''' Copies files passed as arguments. '''
   print COPYING
   for triplet in copyFiles.fileList:
      target = os.path.join( '.', triplet.dest.strip( os.sep ) )
      os.system( 'install -v -m %s %s %s' % ( triplet.perm, triplet.file, target ) )

def openBash( workDir ):
   ''' Runs Bash, chroot-ed in the extracted image. '''
   home = os.environ[ 'HOME' ]
   print WELCOME_INTERACTIVE % home

   os.system( 'mkdir -p ws')
   os.system( 'mount --bind %s ws' % home )
   ret = os.system( '''PS1='%s' chroot .''' % SWI_PROMPT )
   os.system( 'umount ws' )
   os.system( 'rm -rf ws' )

   assert os.WIFEXITED( ret ) and 0 == os.WEXITSTATUS( ret )

def bashHandler( args ):
   ''' Handler invoked by 'swi bash' '''
   maybeRerunAsRoot( args )
   
   opts, args, optParser = parseOptions( args )
   filename = validateSwiArg( optParser, args )
   copyFiles.fileList = validateCopyArgs( optParser, args )

   toRun = list()
   if copyFiles.fileList:
      toRun.append( copyFiles )
   if opts.interactive:
      toRun.append( openBash )
   if not toRun:
      raise Exception( NOTHING_TO_DO )

   try:
      Swi.inSwi( filename, [ chdirToImg ] + toRun )
   except AssertionError:
      pass
