# Copyright (c) 2008-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import tempfile, glob, os, sys, subprocess
import zipfile
import Swi.extract, Swi.create
import Swi.sign

class SwiOptions(object):
   def __init__( self, **kargs ):
      self.update( **kargs )
   def update( self, **kargs ):
      self.__dict__.update( **kargs )

def run( argv ):
   p = subprocess.Popen( argv )
   p.communicate()
   assert( p.returncode == 0 )

def runAndReturnOutput( argv, printStdout=True ):
   p = subprocess.Popen( argv, stdout=subprocess.PIPE, stderr=subprocess.PIPE )
   stdoutdata, stderrdata = p.communicate()
   assert( p.returncode == 0 )
   if stdoutdata and printStdout:
      print stdoutdata
   if stderrdata:
      print stderrdata
   return stdoutdata

def inSwi( file, funcs, fast=False, outputfile=None, readOnly=False ):
   if outputfile is None:
      outputfile = file

   # We chdir below, and stay in the rootfs for the SWI. The user specified a
   # filename relative to their current directory, not somewhere in /tmp, so
   # use a sensible absolute path when we've a different current dir.
   outputfile = os.path.abspath(outputfile)

   # first verify permissions on file.
   if not os.access( file, os.R_OK ):
      print "Permission Error: Can't read " + file + ". Does it exist?"
      sys.exit( 1 )
   if os.path.exists( outputfile ) and not os.access( outputfile, os.W_OK ) and \
         not readOnly:
      print "Permission Error: Can't write to " + file + ". Retry with sudo?"
      sys.exit( 1 )
   # Create a temporary directory to work in.
   tmpdir = tempfile.mkdtemp()
   try:
      print "Extracting", file, "to", tmpdir
      
      Swi.extract.extract( file, SwiOptions( dirname=tmpdir, use_existing=True ))
      for func in funcs:
         func( tmpdir )
      if not readOnly:
         print "Creating", outputfile, "from", tmpdir
         options = SwiOptions( dirname=tmpdir, squashfs=False, fast=fast, modules="",
                               installfs_only=False, force=True, trace=True,
                               hdrfile=None )
         if glob.glob( os.path.join( tmpdir, "rootfs-*.sqsh" ) ):
            options.update( squashfs=True )
         Swi.create.create( outputfile, options )
         result = Swi.sign.updateSwiSignature( outputfile )
         print Swi.sign.UPDATE_SIG_MESSAGE[ result ]
   finally:
      print "Cleaning up", tmpdir
      run( ["sudo", "rm", "-rf", tmpdir] )

def getBlessedAndVersion( swiFile ):
   ''' Returns a tuple of ( BLESSED, SWI_VERSION ) from the 'version' 
       file in @swiFile, ex ( '1', '4.20.0F' ) for a blessed image, 
       ( None, '4.20.0F' ) for non-blessed, and ( None, None ) if 
       version and blessed cannot be determined.'''
   try:
      with zipfile.ZipFile( swiFile, 'r' ) as swi:
         if swi.getinfo( 'version' ):
            with swi.open( 'version', 'r' ) as versionFile:
               # Not using SimpleConfigFile here because it's an archive
               # in a zipfile and we would have to pass the file descriptor
               blessed = None
               version = None
               for line in versionFile:
                  line = line.strip().split( '=' )
                  if line[ 0 ] == 'BLESSED':
                     blessed = line[ 1 ]
                  if line[ 0 ] == 'SWI_VERSION':
                     version = line[ 1 ]
               return blessed, version
   except KeyError:
      return None, None

