# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

'''
The WpaSupplicant is responsible for playing the supplicant role in
the Dot1x negotiation. 
'''

import Arnet, Cell, Logging, QuickTrace, ReversibleSecretCli, SuperServer, Tac
import errno, fnmatch, os, signal, subprocess

qv = QuickTrace.Var
qt0 = QuickTrace.trace0
qt1 = QuickTrace.trace1

Logging.logD( id="DOT1X_SUPPLICANT_AUTH_SUCCEEDED",
              severity=Logging.logInfo,
              format="A Dot1x supplicant has successfully authenticated on "
              "interface %s with EAP method %s, identity %s and mac %s.",
              explanation="A Dot1x supplicant successfully authenticated "
              "with an authenticator.",
              recommendedAction=Logging.NO_ACTION_REQUIRED )

Logging.logD( id="DOT1X_SUPPLICANT_AUTH_FAILED",
              severity=Logging.logError,
              format="A Dot1x supplicant failed authentication on "
              "interface %s with EAP method %s, identity %s and mac %s.",
              explanation="A Dot1x supplicant failed to authenticate with "
              "an authenticator.",
              recommendedAction="Verify that the credentials provided for the "
              "supplicant are correct and match what the authenticator expects." )

Logging.logD( id="DOT1X_SUPPLICANT_INCOMPLETE_CONFIGURATION",
              severity=Logging.logError,
              format="A Dot1x supplicant is incompletely configured on "
              "interface %s. Missing configuration is %s.",
              explanation="A Dot1x supplicant failed to start owing to incomplete "
              "configuration information.",
              recommendedAction="Provide the missing configuration information "
              "through CLI." )

def unlinkFile( fileName ):
   try:   
      os.unlink( fileName )
   except OSError, e:
      if e.errno != errno.ENOENT:
         raise

class SupplicantIntfStatusReactor( SuperServer.LinuxService ):
   notifierTypeName = 'Dot1x::Dot1xIntfSupplicantStatus'

   def __init__( self, status, wpaSupplicantStatus ):
      self.status_ = status
      self.wpaSupplicantStatus_ = wpaSupplicantStatus
      self.intfId_ = status.intfId
      qt1( "SupplicantIntfStatusReactor created for %s" % self.intfId_ )
      self.configFileName_ = '/etc/wpa_supplicant-' \
                            + self.status_.deviceIntfId + '.conf'
      self.supplicantLogFile = None
      self.pidfile = None
      self.initialized_ = False
      self.wpaSupplicantProcess_ = None
      self.networkId = None
      self.timeUpdated_ = Tac.beginningOfTime
      self.processName_ = 'wpa_supplicant'
      self.wpaSupplicantPoller = None
      self.wpaSupplicantSuccessPoller = None
      SuperServer.LinuxService.__init__( self, self.processName_, self.processName_,
                             status, self.configFileName_ )
      self.eapSuccessState = "SUCCESS"
      self.eapFailureState = "FAILURE"
      self.eapInvalidState = "INVALID"
      self.eapInProgressState = "INPROGRESS"
      # changing checkServiceHealth to run every 10 seconds to ensure that reauth
      # events are reacted to, quickly on the supplicant side
      self.healthMonitorInterval = 10
      # we will set the below variable just before stopping the service on
      # config removal - this is to ensure that logoff is only called in this case
      # we should not need to logoff if the stop is coming through a restart case
      self.configRemoved = False

   def fastConf( self ):
      conf = ''
      conf += 'key_mgmt=IEEE8021X\n'
      conf += 'eap=FAST\n'
      conf += 'identity="' + self.status_.identity + '"\n'
      conf += 'phase1="fast_provisioning=2"\n'
      return conf

   def conf( self ):
      conf = ''
      conf += '# logging=' + str( self.status_.logging ) + '\n'
      conf += '# passwordUpdate=' + str( self.status_.passwordUpdate ) + '\n'
      conf += 'ctrl_interface=/var/run/wpa_supplicant\n'
      conf += 'ctrl_interface_group=wheel\n'
      conf += 'eapol_version=2\n' 
      conf += 'ap_scan=0\n' 
      conf += 'network={\n'
      
      if( self.status_.eapMethod == 'fast' ):
         conf += self.fastConf()

      conf += '}\n'
      conf += 'cred={\n'
      conf += 'supplicant_mac="' + \
            self.status_.supplicantMacAddr + '"\n'
      conf += '}\n'
      return conf

   # Helper methods for wpa_supplicant cli
   def wpaSupplicantStatus( self ):
      try:
         wpaSupplicantStatus = Tac.run( [ '/usr/local/sbin/wpa_cli', "-i",
                                          self.status_.deviceIntfId, 'status' ],
                                        stdout=Tac.CAPTURE )
         # check for wpa_cli returning a value of "FAIL\n"
         if wpaSupplicantStatus == "FAIL\n":
            return self.eapInvalidState
         statusLines = []
         if wpaSupplicantStatus:
            statusLines = wpaSupplicantStatus.split( "\n" )
        
         if 'EAP state=SUCCESS' in statusLines:
            return self.eapSuccessState
         elif 'EAP state=FAILURE' in statusLines:
            return self.eapFailureState
                                         
         return self.eapInProgressState            
      except:
         qt0( "wpa_supplicant is probably not yet ready for wpa_cli."\
               "Will try again later" )              
         return self.eapInvalidState

   def getAuthMacMskAndSessId( self ):
      try:
         mibStatus = Tac.run( [ "/usr/local/sbin/wpa_cli", "-i",
                                self.status_.deviceIntfId, "mib" ] ,
                              stdout=Tac.CAPTURE )
         # check for wpa_cli returning a value of "FAIL\n"
         if mibStatus == "FAIL\n":
            return ( None, None, None )
         mibLines = mibStatus.split( "\n" )
         authMac = ""
         for entry in mibLines:
            if 'dot1xSuppLastEapolFrameSource' in entry:
               authMac = entry.split( "=" )[ -1 ]

         authMacAddr = Arnet.EthAddr( authMac )

         msk = Tac.run( [ "/usr/local/sbin/wpa_cli", "-i", 
                          self.status_.deviceIntfId, "msk" ],
            stdout=Tac.CAPTURE )
         if msk == "FAIL\n":
            return ( None, None, None )
         sessionid = Tac.run( [ "/usr/local/sbin/wpa_cli", "-i", 
                                self.status_.deviceIntfId,
                                "sessionid" ], stdout=Tac.CAPTURE )
         if sessionid == "FAIL\n":
            return ( None, None, None )
      except:
         qt0( "wpa_supplicant is probably not yet ready for wpa_cli."\
               "Could not retrieve auth mac addr, msk and session id" )
         return ( None, None, None )

      return ( authMacAddr, msk, sessionid ) 

   def getNetworkId( self ):
      try:
         status = Tac.run( [ "/usr/local/sbin/wpa_cli", "-i",
                             self.status_.deviceIntfId, "status" ],
                           stdout=Tac.CAPTURE )
         if status == "FAIL\n":
            return ""

         statusLines = []
         if status:
            statusLines = status.split( "\n" )

         networkId = ""
         for entry in statusLines:
            if entry.startswith( "id=" ):
               networkId = entry.split( "=" )[ -1 ]

         return str( networkId )

      except:
         qt0( "wpa_supplicant is probably not yet ready for wpa_cli."\
               "Will try again later" )
         return ""

   def updateWpaSupplicantStatus( self, authMacAddr, msk, sessionid ):
      if authMacAddr and msk and sessionid:
         self.wpaSupplicantStatus_.connectionStatus = "connecting"
         eapKey = Tac.Value( "Dot1x::EapKey" )
         eapKey.dot1xType = "supplicant"
         eapKey.masterSessionKey = ReversibleSecretCli.encodeKey( msk )
         eapKey.eapSessionId = sessionid
         eapKey.myMacAddr = self.status_.supplicantMacAddr
         eapKey.peerMacAddr = authMacAddr
         self.wpaSupplicantStatus_.eapKey = eapKey
         self.wpaSupplicantStatus_.connectionStatus = "success"
         qt1( "Updated key status for Dot1x supplicant" )
         Logging.log( DOT1X_SUPPLICANT_AUTH_SUCCEEDED, self.status_.intfId,
                      self.status_.eapMethod, self.status_.identity, 
                      self.status_.supplicantMacAddr )
         return True
      else:
         qt0( "Did not retrieve authMac, msk or session id" )
         self.wpaSupplicantStatus_.connectionStatus = "failed"
         Logging.log( DOT1X_SUPPLICANT_AUTH_FAILED, self.status_.intfId,
                      self.status_.eapMethod, self.status_.identity,
                      self.status_.supplicantMacAddr )
         return False

   # Done with helper methods for wpa_supplicant cli

   def getIncompleteSupplicantConfiguration( self ):
      incompleteConfItems = []
      if not self.status_.eapMethod:
         incompleteConfItems.append( "eap-method" )
      if not self.status_.identity:
         incompleteConfItems.append( "identity" )
      if not self.status_.encryptedPassword:
         incompleteConfItems.append( "passphrase" )

      if incompleteConfItems:
         return ", ".join( incompleteConfItems )
      else:
         return ""

   def serviceEnabled( self ):
      incompleteConfItems = self.getIncompleteSupplicantConfiguration()
      if incompleteConfItems != "":
         qt0( "Dot1x supplicant has incomplete configuration. Missing" \
               " item(s): %s" % incompleteConfItems )
         Logging.log( DOT1X_SUPPLICANT_INCOMPLETE_CONFIGURATION,
                      self.status_.intfId, incompleteConfItems )

      return ( self.status_.supplicantCapable == True and
               self.status_.deviceIntfId != None and 
               self.status_.supplicantMacAddr != "00:00:00:00:00:00" and
               incompleteConfItems == "" ) 

   def serviceProcessWarm( self ):
      if not self.wpaSupplicantProcess_ :
         return False

      if self.wpaSupplicantProcess_.poll() != None:
         # The process exited
         self.wpaSupplicantProcess_ = None
         return False

      authenticationStatus = self.wpaSupplicantStatus()
      if authenticationStatus != self.eapSuccessState:
         return False

      return True

   def supplicantWarm( self ):
      return Tac.now() > ( self.timeUpdated_ + self.status_.supplicantWarmupTime ) 

   def _checkServiceHealth( self ):
      if not self.serviceEnabled():
         return

      doRestart = False

      if self.wpaSupplicantProcess_ and \
            self.wpaSupplicantProcess_.poll() != None:
         self.wpaSupplicantProcess_ = None
         doRestart = True
         qt0( "No supplicant process inside _checkServiceHealth. Restarting" )
      else:
         supplicantState = self.wpaSupplicantStatus()
         # TODO: do we need to restart if state is in failure? 
         # this might result in process starting again and again in wrong config
         if self.supplicantWarm() and ( supplicantState != self.eapSuccessState ):
            self.wpaSupplicantStatus_.connectionStatus = "failed"
            qt0( "Supplicant has not reached a success state after warmup time." )
            Logging.log( DOT1X_SUPPLICANT_AUTH_FAILED, self.status_.intfId,
                         self.status_.eapMethod, self.status_.identity,
                         self.status_.supplicantMacAddr )
            doRestart = True
         elif supplicantState == self.eapSuccessState:
            # update the timeUpdated_ here, so that we don't go into the above case
            # of failure after warmup time, in a reauth scenario
            self.timeUpdated_ = Tac.now()
            authMacAddr, msk, sessionid = self.getAuthMacMskAndSessId()
            if authMacAddr and msk and sessionid:
               if ( sessionid != self.wpaSupplicantStatus_.eapKey.eapSessionId or \
                  self.wpaSupplicantStatus_.connectionStatus != "success" ) and \
                  self.wpaSupplicantSuccessPoller == None:
                  # update all required information here
                  qt1( "Supplicant has succeeded. Updating key values in "\
                       "_checkServiceHealth" )
                  supplicantIntfStatus = self.updateWpaSupplicantStatus( 
                                                        authMacAddr, msk, sessionid )
                  if not supplicantIntfStatus:
                     doRestart = True

      if doRestart:
         self.restartService()

      self.healthMonitorActivity_.timeMin = Tac.now() + \
                                               self.healthMonitorInterval

   def startService( self ):
      qt1( "About to startService wpa_supplicant" )
      if not self.initialized_:
         self.initialized_ = True
         # TODO: do we have to return here? it makes things slow to start up

      if self.wpaSupplicantProcess_:
         if self.wpaSupplicantProcess_.poll() is None:
            #The process is still running
            qt1( "Process is still running. Not starting again." )
            return
         else:
            #The process has exited
            if self.wpaSupplicantPoller: 
               self.wpaSupplicantPoller.cancel()
               self.wpaSupplicantPoller = None
            if self.wpaSupplicantSuccessPoller:
               self.wpaSupplicantSuccessPoller.cancel()
               self.wpaSupplicantSuccessPoller = None
            self.wpaSupplicantProcess_ = None
      
      if not self.serviceEnabled():
         qt1( "supplicant service not enabled due to incomplete conf." \
              "Not starting the supplicant" )
         self.stopService()
         return

      self.supplicantLogFile = '/var/log/wpa_supplicant-' + \
            self.status_.deviceIntfId + '.log'
      
      logCmd = []
      if self.status_.logging:
         logCmd = [ '-f', self.supplicantLogFile, '-dd' ]
      self.wpaSupplicantProcess_ = subprocess.Popen( [ 
         '/usr/local/sbin/wpa_supplicant', 
         '-c', self.configFileName_, '-D', 'wired', 
         '-i', self.status_.deviceIntfId ] + logCmd,
         preexec_fn=os.setsid )

      if not self.wpaSupplicantProcess_:
         qt0( "wpa_supplicant process not started" )
         return

      self.wpaSupplicantStatus_.connectionStatus = "connecting"

      decodedPassword = ReversibleSecretCli.decodeKey( 
            self.status_.encryptedPassword )

      def getNetworkCredsFromCli():
         networkId = self.getNetworkId()
         if networkId:
            self.networkId = networkId
            return True
         else:
            return False

      def programInEapPassword( ignore ):
         self.wpaSupplicantPoller = None
         try:
            qt1( "Programming in eap fast password" )
            Tac.run( [ "/usr/local/sbin/wpa_cli", "-i", 
                       self.status_.deviceIntfId,
                       "password", self.networkId, decodedPassword ] )
         except:
            qt0( "Unable to program in password." )
            self.wpaSupplicantPoller = None
            self.restartService()
           
      def timeoutOfNetworkCredsFromCli():
         qt0( "Unable to get networkId and supplicant mac address from cli" )
         self.wpaSupplicantPoller = None
         self.restartService()

      
      self.wpaSupplicantPoller = Tac.Poller( lambda: ( getNetworkCredsFromCli() \
                                                       == True ),
                                                       programInEapPassword, 
                                             timeoutHandler=\
                                                   timeoutOfNetworkCredsFromCli,
                                             description='wpa_cli '\
                                                         'to come up with proper '\
                                                         'n/w creds on %s' % 
                                                         self.intfId_ )
     
      def handleEapSuccessTimeout():
         qt1( "EAP Success was not detected. Some error in starting supplicant" )
         self.wpaSupplicantSuccessPoller = None
         self.restartService()
         return

      def handleEapSuccess( ignore ):
         # need to retrieve SessId, MSK, auth and supplicant mac addresses
         # also need to update the wpa_supplicant status to success
         self.wpaSupplicantSuccessPoller = None
         authMacAddr, msk, sessionid = self.getAuthMacMskAndSessId()
         supplicantIntfStatus = self.updateWpaSupplicantStatus( authMacAddr,
                                                            msk, sessionid )
         if not supplicantIntfStatus:
            self.restartService()

      self.wpaSupplicantSuccessPoller = Tac.Poller( lambda: ( 
         self.wpaSupplicantStatus() == self.eapSuccessState ),
         handleEapSuccess,
         timeoutHandler=handleEapSuccessTimeout,
         description='wpa_supplicant to come up with success status on %s' % \
               self.intfId_ )

      if self.wpaSupplicantProcess_ :
         qt1( 'Got a non-null wpaSupplicantProcess' )
         qt1( 'Started wpa_supplicant' )
         self.timeUpdated_ = Tac.now()
         self.starts += 1
         # write a pid file into /var/run, so that SuperServer can clean up in case 
         # of bad exit
         self.pidfile = '/var/run/wpa_supplicant-' + self.status_.deviceIntfId + \
               '.pid'
         file( self.pidfile, 'w' ).write( str( self.wpaSupplicantProcess_.pid ) )
      else:
         qt1( 'wpaSupplicantProcess is null' )

   def isProcessKilled( self ):
      return self.wpaSupplicantProcess_.poll() != None

   def stopService( self ):
      qt1( "Stopping wpa_supplicant" )
      # send a logoff to auth only if stopping service because of config removal
      if self.configRemoved:
         try:
            qt1( "Sending logoff from supplicant" )
            Tac.run( [ "/usr/local/sbin/wpa_cli", "-i", 
                       self.status_.deviceIntfId, "logoff" ] )
         except:
            qt0( "Unable to run the cli command to logoff supplicant" )

      if self.wpaSupplicantPoller:
         self.wpaSupplicantPoller.cancel()
         self.wpaSupplicantPoller = None
      if self.wpaSupplicantSuccessPoller: 
         self.wpaSupplicantSuccessPoller.cancel()
         self.wpaSupplicantSuccessPoller = None
      if self.wpaSupplicantProcess_:
         os.killpg( self.wpaSupplicantProcess_.pid, signal.SIGKILL )
         try:
            Tac.waitFor( lambda: self.isProcessKilled(),
                         description='wpaSupplicantProcess to be killed',
                         sleep=True )
         except ( Tac.SystemCommandError, Tac.Timeout ):
            qt0( "Error/Timeout while trying to kill wpaSupplicant" )
            return
         self.wpaSupplicantProcess_ = None

      try:
         if os.path.exists( '/var/run/wpa_supplicant/' ) and \
               os.path.exists( '/var/run/wpa_supplicant/%s' % 
                               self.status_.deviceIntfId ):
            Tac.run( [ 'rm', '/var/run/wpa_supplicant/%s' % \
                       self.status_.deviceIntfId ], asRoot=True )
      except:
         qt0( "No /var/run/wpa_supplicant/%s found" % self.status_.deviceIntfId )

      self.stops += 1
      self.wpaSupplicantStatus_.connectionStatus = "down"

   def restartService( self ):
      qt1( "Restarting wpa_supplicant" )
      self.stopService()
      self.startService()
      self.stops -= 1
      self.starts -= 1
      self.restarts += 1

class SupplicantStatusReactor( Tac.Notifiee ):
   notifierTypeName = 'Dot1x::SupplicantStatus'

   def __init__( self, notifier, wpaSupplicantStatus ):
      qt1( "Creating a SupplicantStatusReactor" )
      Tac.Notifiee.__init__( self, notifier )
      self.notifier = notifier
      self.wpaSupplicantStatus = wpaSupplicantStatus
      self.supplicantIntfStatusReactors_ = {}
      # First clean up any running supplicant processes incase of
      # bad exit last time
      # Cleanup pidfiles, logfiles and confFiles as well
      self.cleanup( '/var/run/', 'pid' )
      self.cleanup( '/var/log/', 'log' )
      self.cleanup( '/etc/', 'conf' )

      # Recreate supplicantIntfStatusReactors for all enabled intfs
      for intfId in self.notifier.supplicantIntfStatus.keys():

         # create wpaSupplicantIntfStatus for this intf, if it does not exist
         if intfId not in self.wpaSupplicantStatus.wpaSupplicantIntfStatus.keys():
            self.wpaSupplicantStatus.wpaSupplicantIntfStatus\
                  .newMember( intfId )

         supplicantReactor = SupplicantIntfStatusReactor(
               self.notifier.supplicantIntfStatus[ intfId ], 
               self.wpaSupplicantStatus.wpaSupplicantIntfStatus[ intfId ] )
         self.supplicantIntfStatusReactors_[ intfId ] = supplicantReactor

   @Tac.handler( 'supplicantIntfStatus' )
   def handleStatus( self, intfId ):
      if intfId in self.notifier.supplicantIntfStatus.keys():

         # create wpaSupplicantIntfStatus for this intf, if it does not exist
         if intfId not in self.wpaSupplicantStatus.wpaSupplicantIntfStatus.keys():
            self.wpaSupplicantStatus.wpaSupplicantIntfStatus\
                  .newMember( intfId )

         if intfId not in self.supplicantIntfStatusReactors_.keys():
            supplicantReactor = SupplicantIntfStatusReactor(
                  self.notifier.supplicantIntfStatus[ intfId ],
                  self.wpaSupplicantStatus.wpaSupplicantIntfStatus[ intfId ] )
            self.supplicantIntfStatusReactors_[ intfId ] = supplicantReactor
      else:
         # delete wpaSupplicantIntfStatus
         if intfId in self.wpaSupplicantStatus.wpaSupplicantIntfStatus.keys():
            del self.wpaSupplicantStatus.wpaSupplicantIntfStatus[ intfId ]

         # need to clean up the SupplicantIntfStatusReactor if conf was deleted
         if intfId in self.supplicantIntfStatusReactors_.keys():
            self.supplicantIntfStatusReactors_[ intfId ].configRemoved = True
            self.supplicantIntfStatusReactors_[ intfId ].stopService()
            try:
               unlinkFile( ( self.supplicantIntfStatusReactors_[ intfId ] ).\
                     configFileName_ )
               unlinkFile( ( self.supplicantIntfStatusReactors_[ intfId ] ).\
                     configFileName_ + ".save" )
               unlinkFile( ( self.supplicantIntfStatusReactors_[ intfId ] ).\
                     configFileName_ + ".log" )
               unlinkFile( ( self.supplicantIntfStatusReactors_[ intfId ] ).\
                     supplicantLogFile )
               unlinkFile( ( self.supplicantIntfStatusReactors_[ intfId ] ).\
                     pidfile )
            except:
               qt0( "Unable to unlink some file" )
            del self.supplicantIntfStatusReactors_[ intfId ]

   def cleanup( self, directory, fileExt ):
      for fl in os.listdir( directory ):
         if fnmatch.fnmatch( fl, 'wpa_supplicant-*.%s' % fileExt ):
            if fileExt == 'pid':
               with open( directory + fl, 'r' ) as f:
                  pid = f.readline()
                  try:
                     os.killpg( int( pid ), signal.SIGKILL )
                  except:
                     qt1( "Failed to kill an old process with pid %s" % pid )
            # Delete the file
            try:
               Tac.run( [ 'rm', directory + fl ], asRoot=True )
               if fileExt == 'conf':
                  if os.path.exists( directory + fl + '.save' ):
                     Tac.run( [ 'rm', directory + fl + '.save' ], asRoot=True )
                  if os.path.exists( directory + fl + '.log' ):
                     Tac.run( [ 'rm', directory + fl + '.log' ], asRoot=True )
            except:
               qt1( "Failed to delete file %s" % fl )

class SupplicantManager( SuperServer.SuperServerAgent ):
   def __init__( self, entityManager ):
      qt1( 'Starting the SupplicantManager' )
      SuperServer.SuperServerAgent.__init__( self, entityManager )
      mg = entityManager.mountGroup()
      self.status = mg.mount( Cell.path( 'dot1x/supplicantStatus/dot1x' ), 
                              'Dot1x::SupplicantStatus', 'r' )
      self.wpaSupplicantStatus = \
            mg.mount( Cell.path( 'dot1x/supplicantStatus/superserver' ),
                      'Dot1x::SuperServerSupplicantStatus', 'wf' )

      def _finished():
         # do not start the SupplicantStatusReactor if not on active supe
         if self.active():
            self.notifiee_ = \
                  SupplicantStatusReactor( self.status, self.wpaSupplicantStatus )

      mg.close( _finished )

   def onSwitchover( self, protocol ):
      # when switchover happens from standby to active, start the reactors
      self.notifiee_ = \
            SupplicantStatusReactor( self.status, self.wpaSupplicantStatus )

def Plugin( ctx ):
   ctx.registerService( SupplicantManager( ctx.entityManager ) )
