# Copyright (c) 2008-2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import SuperServer, os, Cell
import Tac, weakref, re, errno
import glob, Logging
from IpLibConsts import DEFAULT_VRF
from SysMgrLib import netnsNameWithUniqueId
from XinetdLib import XinetdService
from PyWrappers.TelnetServer import telnetdPath

import Tracing
t0 = Tracing.trace0
__defaultTraceHandle__ = Tracing.Handle( "Telnet" )

import QuickTrace
qv = QuickTrace.Var
qt0 = QuickTrace.trace0

def includeDir():
   return '/etc/xinetd.d'

def pidFile( vrf ):
   return '/var/run/xinetd.pid'

def vrfNamespace( vrf ):
   if not vrf:
      return 'default'
   else:
      return 'ns-%s' % vrf

def serviceConfFile( service, vrf ):
   if not vrf:
      return '%s/%s' % ( includeDir(), service )
   else:
      return '%s/%s-%s' % ( includeDir(), service, vrf )

def telnetServiceId( vrf ):
   serviceId = 'telnet'
   if vrf:
      serviceId += '-' + vrf
   return serviceId

# swiped heavily from Ssh.py -- probably ought to merge these into a
# library somehow.
class VrfStatusLocalReactor( Tac.Notifiee ):
   notifierTypeName = 'Ip::VrfStatusLocal'

   def __init__( self, vrfStatusLocal, agent, allVrfStatusLocal ):
      t0( 'Initializing VrfStatusLocalReactor' )
      Tac.Notifiee.__init__( self, vrfStatusLocal )
      self.agent_ = weakref.proxy( agent )
      self.allVrfStatusLocal_ = allVrfStatusLocal
      self.service_ = None
      self.activity_ = Tac.ClockNotifiee()
      self.activity_.handler = self.maybeGoActive
      self.activity_.timeMin = Tac.endOfTime
      self.handleState()

   def close( self ):
      self.activity_.timeMin = Tac.endOfTime
      Tac.Notifiee.close( self )

   @Tac.handler( 'state' )
   def handleState( self ):
      vs = self.notifier_
      t0( 'VrfStatusLocalReactor::handleState for vrf %s' % vs.vrfName )
      if vs.state == 'active':
         self.maybeGoActive()
      elif vs.state == 'deleting':
         # The Telnet files must be cleaned before calling the stopService()
         # In this case, service is xinetd and sync will schedule a call to 
         # stopService() to "reload" the xinetd config
         # ---
         # Instead of using stopService directly, sync is being used in order to 
         # avoid a race condition with ssh SuperServer Plugin, when vrf is deleted
         # It makes sure that both ssh and telnet configs are deleted from
         # /etc/xinetd.d before xinetd is reloaded
         cleanupXinetd( vs.vrfName, self.allVrfStatusLocal_ )
         if self.service_:
            XinetdService.scheduleReloadService()
            self.agent_.service_.sync()
            self.service_.cleanupService()
            del self.agent_.vrfServices_[ self.notifier_.vrfName ]
            self.service_ = None

   def maybeGoActive( self ):
      vs = self.notifier_
      t0( 'VrfStatusLocalReactor::maybeGoActive for vrf %s' % vs.vrfName )
      # it's possible that the state went to deleting while we were waiting
      # but we haven't been cleaned up yet.  If so, do nothing.
      if vs.state == 'active':
         self.activity_.timeMin = Tac.endOfTime
         self.service_ = TelnetConfigNotifiee( self.agent_.config_, vs.vrfName, 
                                               self.notifier_ )
         assert self.notifier_.vrfName not in self.agent_.vrfServices_
         self.agent_.vrfServices_[ self.notifier_.vrfName ] = self.service_

class VrfConfigReactor( Tac.Notifiee ):
   notifierTypeName = "Mgmt::Telnet::VrfConfig"
   def __init__( self, notifier, master ):
      self.master_ = master
      Tac.Notifiee.__init__( self, notifier )

   @Tac.handler( 'serverState' )
   def handleVrfConfig( self ):
      if self.notifier_.name == DEFAULT_VRF:
         self.master_.service_.sync()
      elif self.notifier_.name in self.master_.vrfServices_:
         self.master_.vrfServices_[ self.notifier_.name ].sync()

def cleanupXinetd( vrfName, allVrfStatusLocal ):
   assert vrfName
   t0( 'cleanupXinetd for %s' % vrfName )

   # this function is used both when we tear down an Xinetd service
   # normally and also in restart cases when we need to find leftover
   # xinetd service config files that need to be pruned.  
   # As such, it's paranoid about making sure things get deleted.

   def nuke( name, isDir=False ):
      try:
         if isDir:
            os.rmdir( name )
         else:
            os.unlink( name )
      except OSError, e:
         if e.errno != errno.ENOENT:
            raise 

   nuke( '%s/telnet-%s' % ( includeDir(), vrfName ) )
   nuke( telnetdServiceAcl % vrfNamespace( vrfName ) )

class TelnetConfigNotifiee( XinetdService ):
   notifierTypeName = "Mgmt::Telnet::Config"

   def __init__( self, config, vrf=None, vrfStatusLocal=None ):
      t0( 'TelnetConfigNotifiee::__init__ vrf is %s' % vrf )
      self.vrf_ = vrf
      self.vrfStatusLocal_ = vrfStatusLocal
      self.config_ = config
      serviceName = 'Telnet'
      if self.vrf_:
         serviceName += '-' + self.vrf_

      XinetdService.__init__( self, serviceName, config, 
                              serviceConfFile( 'telnet', vrf ) )
                              
   def serviceEnabled( self ):
      vrfActive = True
      enabled = self.config_.serverState == "enabled"
      if self.vrfStatusLocal_:
         vrfActive = ( self.vrfStatusLocal_.state == "active" )
         vrfName = self.vrfStatusLocal_.vrfName
         vrfConfig = self.config_.vrfConfig
         if vrfName in vrfConfig.keys() and \
            vrfConfig[ vrfName ].serverState != "globalDefault":
            enabled = vrfConfig[ vrfName ].serverState == "enabled"
      else:
         vrfConfig = self.config_.vrfConfig 
         if DEFAULT_VRF in vrfConfig and \
            vrfConfig[ DEFAULT_VRF ].serverState != "globalDefault":
            enabled = vrfConfig[ DEFAULT_VRF ].serverState == "enabled"
      return enabled and vrfActive

   def serviceProcessWarm( self ):
      return XinetdService._serviceProcessWarm()

   def conf( self ):
      state = None
      if self.vrfStatusLocal_:
         state = self.vrfStatusLocal_.state
      t0( 'TelnetConfigNotifiee::conf, for vrf %s enabled is %s, vrf state is %s' % 
          (self.vrf_, self.notifier_.serverState, state ) )

      if not self.serviceEnabled():
         t0( 'TelnetConfigNotifiee::conf(): telnet for vrf %s disabled, returning'
             ' empty conf' % self.vrf_ )
         return ""

      t0( 'TelnetConfigNotifiee::conf(): telnet for vrf %s enabled' % self.vrf_ )
      return """
# default: on
# description: The telnet server serves telnet sessions; it uses
# unencrypted username/password pairs for authentication.
service telnet
{
        id              = %s
        disable         = no 
        flags           = REUSE
        socket_type     = stream        
        wait            = no
        user            = root
        server          = %s
        server_args     = -N -V %s
        namespace       = %s
        log_on_failure  += USERID
        instances       = %d
        per_source      = %d
}
""" % ( telnetServiceId( self.vrf_ ), telnetdPath(), 
        vrfNamespace( self.vrf_ ),
        netnsNameWithUniqueId( vrfNamespace( self.vrf_ ) ),
        self.config_.sessionLimit, self.config_.sessionLimitPerHost )

telnetdServiceAcl = '/etc/telnetd_%s.allow'

class Telnet( SuperServer.SuperServerAgent ):
   def __init__( self, entityManager ):
      t0( 'Telnet SuperServerPlugin init' )
      SuperServer.SuperServerAgent.__init__( self, entityManager )
      mg = entityManager.mountGroup()
      self.config_ = mg.mount( 'mgmt/telnet/config', 'Mgmt::Telnet::Config', 'r' )
      self.allVrfStatusLocal = mg.mount( Cell.path( 'ip/vrf/status/local' ),
                                         'Ip::AllVrfStatusLocal', 'r' )
      self.service_ = None
      self.vrfServices_ = {}
      self.allVrfStatusLocalReactor_ = None
      self.vrfConfigCollReactor_ = None

      def _finish():
         t0( 'Telnet::_finish()' )
         self.service_ = TelnetConfigNotifiee( self.config_, vrf=None, 
                                               vrfStatusLocal=None )
         self.cleanupOldXinetds()
         try:
            # if tcp wrapper based service ACLs are disabled,
            # remove the per-service allow files
            for f in glob.glob( telnetdServiceAcl % '*' ):
               os.remove( f )
         except OSError:
            pass
         self.allVrfStatusLocalReactor_ = Tac.collectionChangeReactor(
            self.allVrfStatusLocal.vrf, VrfStatusLocalReactor,
            reactorArgs=( self, self.allVrfStatusLocal ) )
         self.vrfConfigCollReactor_ = Tac.collectionChangeReactor(
            self.config_.vrfConfig, VrfConfigReactor,
            reactorArgs=( self, ) )

      mg.close( _finish )
 
   def cleanupOldXinetds( self ):
      '''Find out which files/processes refer to VRFs that have been removed or are
         inactive and clean them up.
         '''

      t0( 'Telnet::cleanupOldXinetds' )
      def addVrfIfInactive( vrfSet, vrf, allVrfStatusLocal ):
         try:
            if allVrfStatusLocal.vrf[ vrf ].state != 'active':
               vrfSet.add( vrf )
         except KeyError:
            vrfSet.add( vrf ) # VRF has been removed

      removedVrfs = set()
      # init.d scripts
      for fileName in os.listdir( '/etc/init.d' ):
         m = re.search( r'^xinetd-([^\s]+)', fileName )
         if m:
            addVrfIfInactive( removedVrfs, m.group( 1 ), self.allVrfStatusLocal )
      # xinetd.conf files
      for fileName in os.listdir( '/etc' ):
         m = re.search( r'^xinetd-([^\s]+).conf', fileName )
         if m:
            try:
               addVrfIfInactive( removedVrfs, m.group( 1 ), self.allVrfStatusLocal )
            except OSError, e:
               if e.errno == errno.ENOSPC:
                  # pylint: disable-msg=E1101
                  Logging.log( SuperServer.SYS_SERVICE_FILESYSTEM_FULL, fileName,
                               self.service_.serviceName_ )
                  # pylint: enable-msg=E1101
                  return
               else:
                  raise 
      # xinetd binaries
      for fileName in os.listdir( '/usr/sbin' ):
         m = re.search( r'^xinetd-([^\s]+)', fileName )
         if m:
            addVrfIfInactive( removedVrfs, m.group( 1 ), self.allVrfStatusLocal )
      # xinetd pid
      xinetdVrfPidFiles = glob.glob( '/var/run/xinetd-*.pid' )
      for vrfPidFile in xinetdVrfPidFiles:
         # len( '/var/run/xinetd-' ) == 16
         # len( '.pid' ) == 4
         vrfName = vrfPidFile[ 16:-4 ]
         if len( vrfName ):
            addVrfIfInactive( removedVrfs, vrfName, self.allVrfStatusLocal )
      for vrfName in removedVrfs:
         cleanupXinetd( vrfName, self.allVrfStatusLocal )


   def warm( self ):
      # if we have no services, we're not warm
      if not self.service_ and len( self.vrfServices_ ) == 0:
         return False
      # otherwise we're warm if all services that we do have are warm
      w = True
      if self.service_:
         w = w and self.service_.warm()
      for s in self.vrfServices_.values():
         w = w and s.warm()
      return w

def Plugin( ctx ):
   t0( 'Plugin registered' )
   ctx.registerService( Telnet( ctx.entityManager ) )
