#!/usro/bin/env pytho
# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac, SuperServer
import os, signal, time
import Tracing

traceHandle = Tracing.Handle( 'IpsecSswan' )
t9 = traceHandle.trace9

class IkeDaemonConfigNotifiee( SuperServer.LinuxService ):
   """ Reactor to changes in the boolean value of IpsecConfig
   that informs of strongswans running status """

   notifierTypeName = 'Ipsec::DaemonConfig'
   
   def __init__( self, serviceName, ikeDaemonConfig, ikeDaemonStatus,
                 routingHwStatus, redProto=None, active=True ):
      self.started = False
      self.ikeDaemonConfig = ikeDaemonConfig
      self.ikeDaemonStatus = ikeDaemonStatus
      self.routingHwStatus = routingHwStatus
      self.active = active
      self.sswanPidFile = '/etc/strongswan/ipsec.d/run/charon.pid'

      if self.serviceProcessWarm():
         self.started = True
         self.starts = 1
         
      SuperServer.LinuxService.__init__( self, serviceName, 
                                         'strongswan', ikeDaemonConfig,
                                         '/etc/strongswan/strongswan.conf') 

      if self.isRunning():
         # Nothing to do if we are the Alternate Sup on an SSO
         if redProto == 'sso' and not self.active:
            t9( 'SSO-Standby no op' )
            return
         self.ikeDaemonStatus.sswanRunning = True

      t9( 'End init: active=%s started=%s isRunning=%s' % \
            ( self.active, self.started, self.isRunning() ) )


   def serviceEnabled( self ):
      # If there are no mapped policies, strongswan should turn off
      enabled = self.ikeDaemonConfig.enableSswan
      
      t9( 'serviceEnabled: %s' % ( enabled, ) )
      return enabled
   
   def serviceProcessWarm( self ):
      # The agent is warm if it is not enabled
      if not self.serviceEnabled():
         warm = True
      else:
         running = self.isRunning()
         warm = running

      t9( 'serviceProcessWarm: %s' % ( warm, ) )
      return warm

   def conf( self ):
      t9(  'conf() was called' )
      # Set kernel_forward to 'no' in Sfe mode
      # timeout means the milliseconds strongswan will wait for the netlink
      # message response from Ipsec. In Sfe mode, with this, we will not get into 
      # the deadlock when strongswan do GETSA on an already deleted SA, and we have
      # deleted the vrfRoot. Otherwise, strongswan cannot handle "sudo strongswan 
      # stop" request since it's busy in waiting GETSA response. Set timeout to '0'
      # in Sfa mode since it's not needed ( '10000' works fine for stress test with
      # 128 tunnels in Sfe mode )
      if self.routingHwStatus.kernelForwardingSupported:
         kernel_forward = 'yes'
         parallel_xfrm = 'no'
         timeout = '0'
      else:
         kernel_forward = 'no'
         parallel_xfrm = 'yes'
         timeout = '10000'
      # Changing the number of strongswan threads = 24, to process strongswan
      #    up/down.
      # Default value for threads is 16
      # Changing the max_concurrency = 16, to queue multiple strongswan requests.
      # Default value for max_concurrency is 4.
      # Setting inactivity_close_ike closes lingering IKE_SAs if the only
      #    CHILD_SA is closed due to inactivity. IKE_SAs linger if the
      #    ike-policy config matches but the sa-policy config doesn't.
      config = """
charon {
        load_modular = yes
        install_routes = no
        max_concurrent = 64 
        retransmit_tries = 2
        retransmit_timeout = 8.0
        make_before_break = yes
        delete_rekeyed_delay = 10
        threads = 24
        inactivity_close_ike = yes
        filelog {
                /var/log/charon.log {
                     time_format = %%b %%e %%T
                     append = yes
                     default = 1
                     knl = 2
                     ike = 2
                     flush_line = yes
                 }
                 stdout {
                     default = 0
                     ike_name = yes
                 }
        }
        syslog {
               # enable logging to LOG_DAEMON, use defaults
               daemon {
                    default = -1 
               }
               # minimalistic IKE auditing logging to LOG_AUTHPRIV
               auth {
                    default = -1 
               }
        }
        plugins {
                include strongswan.d/charon/*.conf
                stroke {
                  max_concurrent = 16
                }
                kernel-netlink {
                    fwmark = !0x42
                    roam_events = no
                    kernel_forward = %s
                    parallel_xfrm = %s
                    timeout = %s
                }
                socket-default {
                    fwmark = 0x42
                }
                kernel-libipsec {
                    allow_peer_ts = yes
                }
        }
}

include strongswan.d/*.conf
      """ % ( kernel_forward, parallel_xfrm, timeout )
      return config

   def getCharonPid( self ):
      if os.path.exists( self.sswanPidFile ):
         pid = ''
         with open( self.sswanPidFile, 'r' ) as pidFile:
            pid = pidFile.readline()

         if pid != '' and os.path.exists( '/proc/%s' % ( pid[ :-1 ], ) ):
            return int( pid )
      return -1

   def isRunning( self ):
      pid = self.getCharonPid()
      return pid > 0

   def pidExists( self, pid ):
      if pid <= 0:
         return False
      return os.path.exists( '/proc/%s' % ( pid, ) )

   def startService( self ):
      t9( 'Starting Strongswan via _runServiceCmd' )
      #BUG148056
      self._runServiceCmd( 'start' )

   def stopService( self ):
      t9( 'Stopping Strongswan via _runServiceCmd' )
      pid = self.getCharonPid()
      self._runServiceCmd( 'stop' )

      if not self.pidExists( pid ):
         t9( 'stopped strongswan' )
         return
      try:
         os.kill( pid, signal.SIGTERM )
         time.sleep( 1 )
         if self.pidExists( pid ):
            # Kill forcefully
            t9( 'charon still exists, kill forcefully' )
            os.kill( pid, signal.SIGKILL )
      except OSError:
         # Process might have already terminated
         pass

class SswanManager( SuperServer.SuperServerAgent ):
   """ This agent reacts to changes in the boolean value of
   in the IpsecConfig of strongswan initialization, and
   starts or stops strongswan appropriately. """
   
   def __init__( self, entityManager ):
      t9( 'Starting SswanManager...' )
      # Mount configurations
      SuperServer.SuperServerAgent.__init__( self, entityManager )
      mg = entityManager.mountGroup()

      ikeDaemonConfig = mg.mount( 'ipsec/daemonconfig', 'Ipsec::DaemonConfig', 'rO' )
      ikeDaemonStatus = mg.mount( 'ipsec/daemonstatus', 'Ipsec::DaemonStatus', 'w' )
      routingHwStatus = mg.mount( 'routing/hardware/status',
                                  'Routing::Hardware::Status', 'r' )
      self.notifiee = None

      def _finished():
         t9( 'Agent Mounted' )

         self.notifiee = IkeDaemonConfigNotifiee( "SswanManager", ikeDaemonConfig,
                                                  ikeDaemonStatus, routingHwStatus,
                                                  redProto=self.redundancyProtocol(),
                                                  active=self.active() )
      
      mg.close( _finished )

def Plugin( ctx ):
   t9( 'Registering SswanManager SuperServer service' )
   ctx.registerService( SswanManager( ctx.entityManager ) )
