# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

'''
Redis Server
'''
import os

import BothTrace
import ManagedSubprocess
import ReversibleSecretCli
import SuperServer
import Tac

bv = BothTrace.Var
bt0 = BothTrace.trace0

class ConfigReactor( SuperServer.SystemdService ):
   notifierTypeName = "Mcs::CliConfig"

   def __init__( self, config, redisStatus, configFilename ):
      serviceName = "Redis"
      linuxServiceName = "redis"
      self.config = config
      self.redisStatus = redisStatus
      self.redisSubProcess_ = None
      self.enabled = self.config.enabled
      self.redisPassword = self.config.redisPassword
      self.updatedConfig = False
      self.configFilename = configFilename
      SuperServer.SystemdService.__init__( self, serviceName, linuxServiceName,
                                         config, configFilename )
      # Update the enabled flag at the start
      self.redisStatus.enabled = self.serviceEnabled()

   def serviceProcessWarm( self ):
      """ Returns wheather or not redis is fully started with new configuration """
      bt0( "serviceProcessWarm", bv( self.redisStatus.enabled ) )
      return self.redisStatus.enabled

   def serviceEnabled( self ):
      bt0( 'serviceEnabled', bv( len( self.redisPassword ) > 0 and self.enabled ) )
      return len( self.redisPassword ) > 0 and self.enabled

   def redisRunning( self ):
      bt0( 'redisRunning enter' )
      try:
         result = Tac.run( [ "pidof", "redis-server" ],
            timeout=60, stdout=Tac.CAPTURE,
            stderr=Tac.CAPTURE, asRoot=True, ignoreReturnCode=False )
         bt0( 'pidof redis-server result', bv( result ) )
         return int( result.split()[ 0 ] ) > 0
      except Tac.SystemCommandError as error:
         bt0( 'SystemCommandError', bv( error.output ) )
         return False

   def startService( self ):
      bt0( 'startService enter' )
      if os.environ.get( "P4USER" ):
         self.redisSubProcess_ = ManagedSubprocess.Popen(
               [ "redis-server", self.configFilename ],
               stderr=ManagedSubprocess.PIPE )
         bt0( 'Starting redis with pid', bv( self.redisSubProcess_.pid ) )
      else:
         startStatus = SuperServer.LinuxService.serviceCmd( self, 'start' )
         bt0( 'Starting redis as linuxService status', bv( startStatus ) )
         SuperServer.LinuxService.runServiceCommand( startStatus )
      # Updating the start counter
      self.redisStatus.startCount += 1

   def stopService( self ):
      bt0( 'stopService enter' )
      if not self.redisRunning():
         bt0( 'stopService returns - redis-server is not running' )
         return
      if os.environ.get( "P4USER" ):
         if self.redisSubProcess_:
            bt0( 'Redis SubProcess running at', bv( self.redisSubProcess_.pid ) )
            self.redisSubProcess_.kill()
            self.redisSubProcess_.wait()
            self.redisSubProcess_ = None
         else:
            bt0( 'No Redis SubProcess' )
      else:
         stopStatus = SuperServer.LinuxService.serviceCmd( self, 'stop' )
         bt0( 'Stopping redis as linuxService status', bv( stopStatus ) )
         SuperServer.LinuxService.runServiceCommand( stopStatus )
      # Update stop counter
      self.redisStatus.stopCount += 1

   def restartService( self ):
      bt0( 'RestartService' )
      bt0( 'redisStatus.enabled', bv( self.redisStatus.enabled ) )
      bt0( 'updatedConfig', bv( self.updatedConfig ) )
      if self.redisStatus.enabled:
         self.stopService()
         if self.redisStatus.stopCount:
            self.redisStatus.stopCount -= 1
      # Start redis service only when there is a change in the config
      if self.updatedConfig:
         self.startService()
         if self.redisStatus.startCount:
            self.redisStatus.startCount -= 1
      # Update restart counter
      self.redisStatus.restartCount += 1

   def conf( self ):
      bt0( 'Conf enter' )
      self.updatedConfig = True
      configData = """
timeout 0
tcp-keepalive 300
daemonize no
supervised no
pidfile /var/run/redis_6379.pid
loglevel debug
logfile /var/log/redis.log
databases 16
always-show-logo no"""
      if self.redisPassword:
         clearPassword = ReversibleSecretCli.decodeKey( self.redisPassword )
         configData += """
requirepass %s""" % ( clearPassword )
      bt0( 'Conf exit' )
      return configData

   @Tac.handler( 'redisPassword' )
   def handlePassword( self ):
      bt0( 'Handle Redis password' )
      # The password has changed and the following will happen by SuperServerPlugin
      # - Call Conf method
      # - Call restartService because of config file change
      # There is no need to stop/start service in this handler
      self.redisPassword = self.config.redisPassword
      self.redisStatus.enabled = self.serviceEnabled()

   @Tac.handler( 'enabled' )
   def handleEnabled( self ):
      bt0( 'Handling enabled', bv( self.config.enabled ) )
      # The enabled flag in redis config has changed and the following will happen
      # by SuperServerPlugin:
      # - Call Conf method
      # - Call stopService or startService based on return value from
      #   serviceEnabled, False or True, respectively.
      # There is no need to stop/start service in this handler
      self.enabled = self.config.enabled
      self.redisStatus.enabled = self.serviceEnabled()

class Redis( SuperServer.SuperServerAgent ):
   def __init__( self, entityManager ):
      SuperServer.SuperServerAgent.__init__( self, entityManager )
      mg = entityManager.mountGroup()
      config = mg.mount( 'mcs/config/cli', 'Mcs::CliConfig', 'r' )
      redisStatus = mg.mount( 'mcs/status/redis', 'Mcs::RedisStatus', 'w' )
      self.service_ = None
      self.configFilename_ = None
      self.redisPassword_ = None

      def _finish():
         assert config.configFilename, "Config filename is needed for redis to run."
         self.configFilename_ = config.configFilename
         bt0( "ConfigFilename is", bv( self.configFilename_ ) )
         self.service_ = ConfigReactor( config, redisStatus, self.configFilename_ )

      mg.close( _finish )

   def warm( self ):
      return True

def Plugin( ctx ):
   ctx.registerService( Redis( ctx.entityManager ) )
