# Copyright (c) 2007-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from cStringIO import StringIO
from eunuchs.if_h import IFF_RUNNING
import os
import os.path
import re
from socket import AF_INET, AF_INET6
import subprocess
import weakref
from Arnet.NsLib import DEFAULT_NS
import Cell
from IpLibConsts import DEFAULT_VRF
import NtpLib
from NtpLib import serverReachCheckTimeout
from NtpLib import authModeNone, authModeAll, authModeServers
from PyWrappers.NtpUdel import ntpd
from PyWrappers.NtpStat import ntpstatBinary
import QuickTrace
from ReversibleSecretCli import decodeKey
import SharedMem
import Smash
import SuperServer
import Tac
import Toggles.NtpToggleLib

qv = QuickTrace.Var
qt0 = QuickTrace.trace0
qt5 = QuickTrace.trace5 # Ntp wait-for-warmup tracing
featureServeMultiVrfEnabled = Toggles.NtpToggleLib.toggleNtpServeMultiVrfEnabled()

# The list of associations has '+' as the first character for candidate peers
# and * for selected peers.
reachedPeerRe = re.compile( "[*+]" )

ntpdPidFile = "/var/run/ntpd.pid"

class NetStatusReactor( Tac.Notifiee ):
   notifierTypeName = "System::NetStatus"

   def __init__( self, netStatus, ntpService ):
      self.ntpService_ = weakref.proxy( ntpService )
      Tac.Notifiee.__init__( self, netStatus )
      for server in self.notifier_.nameServer:
         self.handleNameServer( server )
      self.handleDomainName()

   @Tac.handler( 'nameServer' )
   def handleNameServer( self, server ):
      qt0( "handleNameServer", qv( self.notifier_.nameServer ) )
      self.ntpService_.sync()

   @Tac.handler( 'domainName' )
   def handleDomainName( self ):
      qt0( "handleDomainName", qv( self.notifier_.domainName ) )
      self.ntpService_.sync()

class KniReactor( Tac.Notifiee ):
   notifierTypeName = "KernelNetInfo::Status"

   def __init__( self, netNs, kniStatus, service ):
      self.netNs_ = netNs
      self.kniStatus_ = kniStatus
      self.service_ = service
      self.interfaces_ = {}
      Tac.Notifiee.__init__( self, self.kniStatus_ )

      for k in self.kniStatus_.interface.keys():
         self.handleKniIntf( k )

      for k in self.kniStatus_.addr.keys():
         self.handleAddr( k )

   @Tac.handler( 'interface' )
   def handleKniIntf( self, key ):
      interface = self.kniStatus_.interface.get( key )
      if not interface:
         if key in self.interfaces_:
            del self.interfaces_[ key ]
         return

      if key not in self.interfaces_ or \
         self.interfaces_[ key ].flags != interface.flags or \
         self.interfaces_[ key ].flagChgCount != interface.flagChgCount:
         self.handleFlags( interface )

      self.interfaces_[ key ] = interface

   def handleFlags( self, interface ):
      qt0( "handleFlags on", qv( interface.deviceName ), qv( interface.flags ) )
      if interface.flags & IFF_RUNNING:
         self.service_.handleIntfChange( interface.deviceName, interface.flags )

   @Tac.handler( 'addr' )
   def handleAddr( self, key ):
      qt0( "handleAddr", qv( key ) )
      if key in self.kniStatus_.addr and self.kniStatus_.addr[ key ].assigned:
         interface = self.kniStatus_.interface.get( key.ifKey )
         if interface:
            self.service_.handleIntfChange( interface.deviceName, key )

class L3StatusReactor( Tac.Notifiee ):
   notifierTypeName = "L3::Intf::Status"
   def __init__( self, l3Status, ipIntfStatusReactor ):
      self.ipIntfStatusReactor_ = ipIntfStatusReactor
      Tac.Notifiee.__init__( self, l3Status )

   @Tac.handler( 'vrf' )
   def handleVrf( self ):
      self.ipIntfStatusReactor_.handleVrf()

class IpIntfStatusReactor( Tac.Notifiee ):
   notifierTypeName = "Ip::IpIntfStatus"

   def __init__( self, notifier, service ):
      self.service_ = service
      Tac.Notifiee.__init__( self, notifier )
      self.l3StatusReactor_ = L3StatusReactor( self.notifier_.l3Status, self )
      self.handleActiveAddrWithMask()
      self.handleVrf()

   @Tac.handler( 'activeAddrWithMask' )
   def handleActiveAddrWithMask( self ):
      addr = self.notifier_.activeAddrWithMask
      qt0( "handleActiveAddrWithMask:", qv( addr.stringValue ), "on",
           qv( self.notifier_.name ) )
      # React only to events that can make servers more reachable: added addresses.
      if addr.stringValue != "0.0.0.0/0":
         self.service_.handleIntfChange( self.notifier_.name, addr )

   def handleVrf( self ):
      qt0( "handleVrf", qv( self.notifier_.vrf ), "for", qv( self.notifier_.name ) )
      self.service_.handleIntfChange( self.notifier_.name, self.notifier_.vrf )

class Ip6IntfStatusReactor( Tac.Notifiee ):
   notifierTypeName = "Ip6::IntfStatus"

   def __init__( self, notifier, service ):
      self.service_ = service
      Tac.Notifiee.__init__( self, notifier )
      for key in notifier.addr:
         self.handleIp6Addr( key )
   
   @Tac.handler( 'addr' )
   def handleIp6Addr( self, key ):
      addr = self.notifier_.addr
      addr6 = str( addr.get( key ) )
      qt0( "handleIp6Addr:", qv( addr6 ), "on", qv( self.notifier_.name ) )
      # React only to events that can make servers more reachable: added addresses.
      if addr6 != "::/0":
         self.service_.handleIntfChange( self.notifier_.name )

class VrfStatusLocalReactor( Tac.Notifiee ):
   notifierTypeName = "Ip::VrfStatusLocal"
   def __init__( self, vrfStatus, ntpService ):
      self.service_ = ntpService
      self.vrfName = vrfStatus.name
      Tac.Notifiee.__init__( self, vrfStatus )

   @Tac.handler( 'state' )
   def handleState( self ):
      qt0( "handleState", qv( self.notifier_.state ), "for",
           qv( self.notifier_.vrfName ) )
      self.service_.handleVrfChange( self.vrfName, self.notifier_.state )

   def close( self ):
      self.service_.handleVrfChange( self.vrfName, "deleted" )
      Tac.Notifiee.close( self )

class SshConfigReactor( Tac.Notifiee ):
   notifierTypeName = "Mgmt::Ssh::Config"
   def __init__( self, sshConfig, ntpService ):
      self.service_ = ntpService
      Tac.Notifiee.__init__( self, sshConfig )

   @Tac.handler( 'fipsRestrictions' )
   def handleFipsRestrictions( self ):
      self.service_.handleKeyFile( None )

   def close( self ):
      Tac.Notifiee.close( self )

class NtpService( SuperServer.LinuxService ):
   """ Manages the ntp service based on configuration mounted at /time/ntp """
   notifierTypeName = "Ntp::Config"
   
   def __init__( self, config, status, netStatus, ipStatus, ip6Status, 
                 sshConfig, allVrfStatusLocal, agent ):
      self.agent_ = weakref.proxy( agent )
      SuperServer.LinuxService.__init__( self, "ntp", ntpd(),
                                         config, '/etc/ntp.conf' )

      self.ntpStatus_ = status
      self.ntpConfig = config
      self.configEnabled_ = False

      # Created to react to changes in FIPS Restrictions which
      # controls usage of MD5 keys
      self.sshConfig_ = sshConfig
      self.sshConfigReactor = SshConfigReactor( self.sshConfig_, 
            weakref.proxy( self ) )

      # Note: Do not use the -x ntpd option: With -x, ntpd tells the kernel
      # that the system clock is unsynchronized, which prevents the kernel
      # from automatically keeping the hw clock in sync
      
      # Flag used by eligible reactors to attempt to force restarts
      self.forceRestartIfEnabled_ = False

      # Get notified when the state ( up/down, or ip address) of the management
      # interfaces change, so we can restart ntp whenever the state changes.
      # This needs to get updated anytime the VRF in use changes.
      self.kniReactor_ = None

      # Create a reactor to the System::NetStatus, so we can restart ntp whenever
      # the nameservers or domainName change
      self.netStatus_ = netStatus
      self.netStatusReactor_ = NetStatusReactor( netStatus, self )

      # Create a reactor to the Ip::IpIntfStatus, to be notified when the ip
      # address of a source interface changes. The reaction is the same as
      # kernel device i.e. restart ntp
      self.ipStatus_ = ipStatus
      self.ipIntfStatusReactor_ = None

      self.ip6Status_ = ip6Status
      self.ip6IntfStatusReactor_ = None

      self.serverReachPoller_ = None

      # Create a reactor to the Ip::AllVrfStatusLocal, to be notified when VRFs are
      # created or deleted. If ntp is configured for a specific VRF, a reaction for
      # the same will cause ntp to start or stop depending on whether the VRF is
      # created or deleted
      self.allVrfStatusLocal_ = allVrfStatusLocal
      self.allVrfStatusLocalReactor_ =  \
          Tac.collectionChangeReactor( allVrfStatusLocal.vrf,
                                       VrfStatusLocalReactor,
                                       reactorArgs=( weakref.proxy( self ), ) )

      # ntpd sysconfig file contents
      self.sysconf_ = ""

      # Until we fix BUG58411 we need to write the keys file
      # on startup.
      self.handleKeyFile( None )

      self.handleEnablingConfig()

   def restartsOnIntfChangesEnabled( self ):
      return self.ntpStatus_.restartsOnIntfChangesEnabled or \
             self.ntpConfig.restartsForcedOnIntfChanges

   def updateIpIntfInfoReactor( self, vrfName ):
      # Delete the reactor
      if not vrfName or \
         ( vrfName != DEFAULT_VRF and vrfName not in self.allVrfStatusLocal_.vrf ):
         qt0( "Deleting kni reactor" )
         netNs = self.kniReactor_.netNs_
         self.kniReactor_ = None
         # Don't unmount the default network namespace
         if netNs == DEFAULT_NS:
            return
         shmemEm = SharedMem.entityManager( sysdbEm=self.agent_.entityManager )
         shmemEm.doUnmount( "kni/ns/%s/status" % netNs )
         return
      netNs = DEFAULT_NS
      if vrfName != DEFAULT_VRF:
         netNs = self.allVrfStatusLocal_.vrf[ vrfName ].networkNamespace
      # Update
      if not self.kniReactor_ or self.kniReactor_.netNs_ != netNs:
         qt0( "Updating kni reactor to netNs", netNs )
         shmemEm = SharedMem.entityManager( sysdbEm=self.agent_.entityManager )
         kniStatus = shmemEm.doMount( "kni/ns/%s/status" % netNs,
                                      "KernelNetInfo::Status",
                                      Smash.mountInfo( 'keyshadow' ) )
         self.kniReactor_ = KniReactor( netNs, kniStatus, self )

   @Tac.handler( 'symmetricKey' )
   def handleKeyFile( self, keyId ):
      # We don't care what the change was, just rewrite the whole key
      # file and have ntpd restart
      qt0( "handleKeyFile: Updating key file passwords" )
      keyConf = ""
      for key in self.ntpConfig.symmetricKey.itervalues():
         keyType = key.keyType.upper()
         if keyType == 'MD5':
            if self.sshConfig_.fipsRestrictions:
               keyConf += "# Key %d of type %s is " % ( key.keyId, keyType ) + \
                          "omitted because it is not FIPS-compatible.\n"
               continue
         elif keyType == 'SHA1':
            pass
         else:
            assert False, "Unknown KeyType choosen"
         cmd = "%s %s %s\n" % ( key.keyId, keyType, decodeKey( key.secret ) )
         keyConf += cmd
      # No need to check the return code here as we're calling sync() anyways below
      self.writeConfigFile( '/etc/ntp/keys', keyConf )
      self.forceRestartIfEnabled_ = True
      self.sync()

   def handleIntfChange( self, *args ):
      if self.restartsOnIntfChangesEnabled():
         self.forceRestartIfEnabled_ = True
      qt0( "handleIntfChange: forceRestartIfEnabled_ is",
           qv( self.forceRestartIfEnabled_ ) )
      self.sync()
         
   def handleVrfChange( self, *args ):
      self.handleEnablingConfig( self )
      self.sync()

   @Tac.handler( 'server' )
   def handleEnablingConfig( self, *_args ):
      vrfName = self._enabledVrf()
      qt0( "handleEnablingConfig: vrfName is", qv( vrfName ) )
      if vrfName is None:
         if self.configEnabled_:
            self.configEnabled_ = False
            self.updateIpIntfInfoReactor( None )
            self.ipIntfStatusReactor_ = None
            self.ip6IntfStatusReactor_ = None
      else:
         # Written as a separate statement instead of elif
         # to more clearly parallel the [vrfName is None] case
         if not self.configEnabled_:
            self.configEnabled_ = True
            self.updateIpIntfInfoReactor( vrfName )
            self.ipIntfStatusReactor_ = Tac.collectionChangeReactor(
               self.ipStatus_.ipIntfStatus, IpIntfStatusReactor,
               reactorArgs=( self, ) )
            self.ip6IntfStatusReactor_ = Tac.collectionChangeReactor(
               self.ip6Status_.intf, Ip6IntfStatusReactor,
               reactorArgs=( self, ) )

   def _enabledVrf( self ):
      if not self.ntpConfig.server:
         return None
      # If a server is configured, run only if it is for a valid and active VRF.
      allVrf = self.allVrfStatusLocal_.vrf
      vrfName = None
      for server in self.ntpConfig.server.itervalues():
         if vrfName and vrfName != server.vrf:
            qt0( "Ntp is configured across multiple VRFs, disabling it" )
            return None
         if server.vrf:
            vrfName = server.vrf
         else:
            vrfName = DEFAULT_VRF
      # Disable Ntp if there's at least one server and it doesn't correspond to an
      # active VRF. The default VRF is assumed to be active.
      if vrfName and vrfName != DEFAULT_VRF:
         if vrfName not in allVrf:
            qt0( "VRF", vrfName, "is not a valid VRF: disabling Ntp" )
            return None
         elif allVrf[ vrfName ].state != 'active':
            qt0( "VRF", vrfName, "is in state", allVrf[ vrfName ].state,
                 ": disabling Ntp" )
            return None
      return vrfName

   def serviceEnabled( self ):
      # Don't run Ntp unless this is the active supervisor.
      if not self.agent_.active():
         return False
      return self.configEnabled_

   def restartService( self ):
      SuperServer.LinuxService.restartService( self )
      # Just in case the poller somehow wasn't already started
      self._startServerReachPoller()

   def _findReachedPeer( self ):
      if not os.path.exists( ntpdPidFile ):
         return False
      try:
         pid = open( ntpdPidFile ).read()
      except IOError:
         return False
      if not os.path.exists( "/proc/" + pid ):
         return False

      # Disable host name resolution (-n) to ensure that ntpq completes in reasonably
      # less time than the agent heartbeat timeout.
      try:
         output = NtpLib.runMaybeInNetNs( 
            self.ntpConfig, self.allVrfStatusLocal_,
            [ "/usr/sbin/ntpq", "-np", "127.0.0.1" ], ignoreReturnCode=True,
            stdout=Tac.CAPTURE, stderr=Tac.CAPTURE, timeout=serverReachCheckTimeout )
      except NtpLib.NetNsError:
         return False
      except Tac.Timeout:
         return False
      reachCol = None
      for line in StringIO( output ):
         words = line.split()
         if reachCol is None:
            for i, w in enumerate( line.split() ):
               if w == "reach":
                  reachCol = i
                  break
         elif reachedPeerRe.match( line ):
            # If we ever add a local clock source back into the conf file, it will
            # appear as "LOCAL(1)" in the peer list.
            if "LOCAL(1)" in line:
               continue
            if reachCol < len( words ) and words[ reachCol ] != "0":
               qt5( "Reached peer found" )
               return True
      return False

   def _startServerReachPoller( self ):
      if self.restartsOnIntfChangesEnabled() and not self.serverReachPoller_:
         self.serverReachPoller_ = Tac.Poller(
            self._findReachedPeer,
            timeout=serverReachCheckTimeout,
            handler=lambda _: self._disableRestartsOnIntfChanges(),
            timeoutHandler=self._disableRestartsOnIntfChanges,
            description="NTP server to be reached",
            warnAfter=Tac.endOfTime
         )

   def startService( self ):
      SuperServer.LinuxService.startService( self )
      self._startServerReachPoller()

   def _disableRestartsOnIntfChanges( self ):
      qt0( "Disabling forced restarts on interface changes" )
      self.ntpStatus_.restartsOnIntfChangesEnabled = False
      self.serverReachPoller_ = None

   def _stopServerReachPoller( self ):
      if self.serverReachPoller_:
         self.serverReachPoller_.cancel()
         self.serverReachPoller_ = None

   def stopService( self ):
      self._stopServerReachPoller()
      SuperServer.LinuxService.stopService( self )

   def _validVrf( self, vrfNm ):
      if vrfNm == DEFAULT_VRF or vrfNm in self.allVrfStatusLocal_.vrf:
         return True
      return False

   def _allIntfVrfs( self ):
      """
      Return all the vrfs for server enabled interfaces that are not
      server enabled by default or the primary vrf.
      """
      vrfList = []
      exceptionEnabledIntf = self.ntpConfig.serverModeEnabledIntf
      for intfName in exceptionEnabledIntf:
         ipIntfStatus = self.ipStatus_.ipIntfStatus.get( intfName )
         if ipIntfStatus:
            intfVrf = ipIntfStatus.vrf if ipIntfStatus.vrf else DEFAULT_VRF
            if intfVrf not in self.ntpConfig.serveVrfName \
               and intfVrf != NtpLib.vrfInUse( self.ntpConfig ):
               vrfList.append( intfVrf )
      return vrfList

   def _allIpAddrs( self, intfName, expectedIntfVrf, masked=True ):
      """
      Return all the IP addresses associated with an interface.
      Sometimes we want to apply some configuration option
      to all of an interface's IP addresses.
      """
      # IPv4
      ipIntfStatus = self.ipStatus_.ipIntfStatus.get( intfName )
      addrList = []
      intfVrf = DEFAULT_VRF
      if ipIntfStatus:
         intfVrf = ipIntfStatus.vrf if ipIntfStatus.vrf else DEFAULT_VRF
         if expectedIntfVrf == intfVrf or featureServeMultiVrfEnabled:
            if ipIntfStatus.activeAddrWithMask:
               addrList.append( str( ipIntfStatus.activeAddrWithMask ) )
            for addr in ipIntfStatus.activeSecondaryWithMask:
               addrList.append( str( addr ) )

      # IPv6
      ip6IntfStatus = self.ip6Status_.intf.get( intfName )
      if ip6IntfStatus:
         if expectedIntfVrf == intfVrf or featureServeMultiVrfEnabled:
            addrList += [ str( a ) for a in ip6IntfStatus.addr if
                          not a.address.isLinkLocal ]

      if not masked:
         addrList = [ ip.split( "/" )[ 0 ] for ip in addrList ]
      return addrList

   def _ipv4Addr( self, intfName, expectedIntfVrf, ntpVrf ):
      ipIntfStatus = self.ipStatus_.ipIntfStatus.get( intfName )
      if ipIntfStatus:
         addr = ipIntfStatus.activeAddrWithMask.address
         intfVrf = ipIntfStatus.vrf
         if not intfVrf:
            intfVrf = DEFAULT_VRF
         if addr != '0.0.0.0':
            if expectedIntfVrf == intfVrf == ntpVrf:
               return addr
            else:
               qt0( "Skipping {intfName} (v4) as it's in VRF {intfVrf} (in use: "
                    "{ntpVrf}, in command: {expectedIntfVrf})".format( **locals() ) )
         else:
            qt0( "Skipping %s (v4) as it doesn't have an IPv4 address" % intfName )
      return None

   def _ipv6Addr( self, intfName, expectedIntfVrf, ntpVrf ):
      ip6IntfStatus = self.ip6Status_.intf.get( intfName )
      if ip6IntfStatus:
         for addr in sorted( ip6IntfStatus.addr ):
            if addr.address.isLinkLocal:
               continue
            addr6 = str( addr ).split( "/" )[ 0 ] if addr else None
            intfVrf = ip6IntfStatus.vrf or DEFAULT_VRF
            if addr6 and addr6 != "::":
               if expectedIntfVrf == intfVrf == ntpVrf:
                  return addr6
               else:
                  qt0( "Skipping {intfName} (v6) as it's in VRF {intfVrf} "
                       "(in use: {ntpVrf}, in command: "
                       "{expectedIntfVrf})".format( **locals() ) )
            else:
               qt0( "Skipping %s (v6) as it doesn't have an IPv6"
                    " address" % intfName )
      return None

   def _ipAddr( self, intfName, expectedIntfVrf, ntpVrf, addrFamily=None ):
      """ Return the IP address associated with an interface. Return None in the
      following cases:
      - the interface does not exist
      - the interface is not configured with an IP address
      - the interface is not configured with an IP address of the specified
        address family
      - the VRF configured for the Ntp Source Interface is different than the VRF 
        the Source Interface is in
      - Ntp is using a different VRF
      """
      addr = None
      if not intfName:
         qt0( "Skipping NULL interface" )
         return None

      # IPv4
      if not addrFamily or addrFamily == AF_INET:
         addr = self._ipv4Addr( intfName, expectedIntfVrf, ntpVrf )

      # IPv6
      if not addr and ( not addrFamily or addrFamily == AF_INET6 ):
         addr = self._ipv6Addr( intfName, expectedIntfVrf, ntpVrf )
      return addr

   def _addressFamily( self, addr ):
      if re.search( r"\d+\.\d+\.\d+\.\d+", addr ) != None:
         return AF_INET
      if ":" in addr:
         return AF_INET6
      return None

   def _serverSourceIntfAddr( self, server ):
      sourceIntf = None
      sourceAddr = None
      expectedVrf = server.vrf
      serverAddrFamily = self._addressFamily( server.ipOrHost )
      if server.sourceIntf:
         sourceIntf = server.sourceIntf
         expectedVrf = server.vrf
      elif self.notifier_.defaultSourceIntf:
         sourceIntf = self.notifier_.defaultSourceIntf.intf
         expectedVrf = self.notifier_.defaultSourceIntf.vrf
      sourceAddr = self._ipAddr( sourceIntf, expectedVrf,
                                 server.vrf, serverAddrFamily )
      return ( sourceIntf, sourceAddr )

   def _vrfServerModeEnabled( self, intfName, expectedIntfVrf ):
      """
      Based on the interface, determine whether ServerEnabled is true
      for the corresponding VRF.  Return True or False, and the name of the VRF.
      """
      if not featureServeMultiVrfEnabled:
         return self.ntpConfig.serverModeEnabledDefault, None
      ipIntfStatus = self.ipStatus_.ipIntfStatus.get( intfName )
      if ipIntfStatus:
         intfVrf = ipIntfStatus.vrf if ipIntfStatus.vrf else DEFAULT_VRF
      else:
         intfVrf = DEFAULT_VRF
      qt0( "vrfServerModeEnabled intfVrf ", qv( intfVrf ) )
      if intfVrf == expectedIntfVrf:
         qt0( "vrfServerModeEnabled expectedintfVrf ", qv( expectedIntfVrf ) )
         return self.ntpConfig.serverModeEnabledDefault, None
      return intfVrf in self.ntpConfig.serveVrfName, intfVrf

   def _interfaceErrorMsg( self, blockOrAccept, intfName, vrfName ):
      if featureServeMultiVrfEnabled:
         msg = ( "# The switch is configured to %s NTP requests on %s but\n"
                    "# the interface has no IP address,\n"
                    "# so a \"restrict\" command for its address cannot be\n"
                    "# included here.\n"
                        % ( blockOrAccept, intfName ) )
         if vrfName:
            msg += "# Vrf: %s" % vrfName
      else:
         msg = ( "# The switch is configured to %s NTP requests on %s but\n"
                    "# the interface has no IP address or is in a different VRF,\n"
                    "# so a \"restrict\" command for its address cannot be\n"
                    "# included here.\n"
                        % ( blockOrAccept, intfName ) )
      return msg

   def writeNtpdSysconfig( self, conf ):
      # Write static configuration files for the ntp server.
      qt0( "Writing ntpd sysconfig file" )
      return self.writeConfigFile( "/etc/sysconfig/ntpd", conf )

   def ntpdSysconfig( self ):
      qt5( "computing ntpd sysconfig" )
      conf = """# Drop root to id 'ntp:ntp'
OPTIONS="-u ntp:ntp -g -p {}"
""".format( ntpdPidFile )
      vrfName = NtpLib.vrfInUse( self.ntpConfig )
      if vrfName != DEFAULT_VRF:
         try:
            netNs = self.allVrfStatusLocal_.vrf[ vrfName ].networkNamespace
            conf += '\nNETNS_EXEC="ip netns exec ' + netNs + '"'
         except KeyError:
            # there's no guarantee that the VRF that triggered the reaction is still
            # there
            pass
      return conf

   def conf( self ):
      """ Calculate the new ntp conf """
      # Either way we reset the flag
      qt0( "conf: forceRestartIfEnabled_ is", qv( self.forceRestartIfEnabled_ ) )
      VrfName = Tac.Type( "L3::VrfName" )
      forceRestartIfEnabled = self.forceRestartIfEnabled_
      self.forceRestartIfEnabled_ = False
      if self.serviceEnabled():
         if forceRestartIfEnabled:
            self.forceRestart_ = forceRestartIfEnabled
         sysconf = self.ntpdSysconfig()
         if self.sysconf_ != sysconf:
            if self.writeNtpdSysconfig( sysconf ):
               self.sysconf_ = sysconf
            else:
               self.sync()

         # Basic configuration
         conf = """
# Encode the nameservers, domain name, and VRF into the config, so
# that changes show up in the config and result in an ntp restart.
# NameServers: %s
# DomainName: %s
# VRF: %s

driftfile /persist/local/ntp.drift
logconfig +events +status
""" % ( ", ".join( sorted( self.netStatus_.nameServer.values() ) ),
        self.netStatus_.domainName,
        NtpLib.vrfInUse( self.ntpConfig ) )

         conf += "interface listen wildcard\n"

         listenVrfs = sorted( set( self.ntpConfig.serveVrfName.keys() ) |
                              set( self._allIntfVrfs() ) )
         for serveVrfName in listenVrfs:
            if self._validVrf( serveVrfName ):
               conf += "interface listen netns %s wildcard\n" \
                       % VrfName( serveVrfName ).nsName()
            else:
               conf += "# Ntp configured invalid netns %s\n" % \
                       VrfName( serveVrfName ).nsName()

         restrictions = "nomodify notrap noquery nopeer"
         if self.notifier_.authMode == authModeAll:
            restrictions += " notrust"

         for serveVrfName in listenVrfs:
            if not self._validVrf( serveVrfName ):
               continue
            if serveVrfName in self.ntpConfig.serveVrfName:
               conf += "restrict netns %s default %s\n" % \
                       ( VrfName( serveVrfName ).nsName(), restrictions )
            else:
               conf += "restrict netns %s default %s\n" % \
                       ( VrfName( serveVrfName ).nsName(),
                         restrictions + " noclient" )

         serverModeEnabledDefault = self.ntpConfig.serverModeEnabledDefault
         if not serverModeEnabledDefault:
            restrictions += " noclient"

         conf += ( "restrict default %s\n" % restrictions )

         if self.notifier_.authMode != authModeNone and self.notifier_.trustedKeys:
            keys = self.notifier_.trustedKeys.split( "," )
            cmd = "trustedkey"
            for key in keys:
               if "-" in key:
                  keyRange = key.split( "-" )
                  cmd += " (%s ... %s)" % tuple( keyRange )
               else:
                  cmd += " " + str( key )
            cmd += "\n"
            conf += cmd
            cmd = "keys /etc/ntp/keys\n"
            conf += cmd

         conf += "restrict 127.0.0.1\n"
         conf += "restrict ::1\n"

         vrfInUse = NtpLib.vrfInUse( self.ntpConfig, defaultToNone=True )

         # Restrictions to apply to interfaces with server-mode
         # configuration.
         serverModeIntfRestrictions = restrictions
         if "noclient" not in restrictions:
            serverModeIntfRestrictions += " noclient"

         for intfName in sorted( self.ntpConfig.serverModeDisabledIntf ):
            qt0( "serverModeDisabledIntf", qv( intfName ) )
            serverModeEnabled, nonPrimaryVrf = self._vrfServerModeEnabled( intfName,
                                                                           vrfInUse )
            if not serverModeEnabled:
               continue
            if nonPrimaryVrf and not self._validVrf( nonPrimaryVrf ):
               continue
            qt0( "serverModeDisabledIntf serverModeEnabled ", qv( intfName ) )
            addrList = self._allIpAddrs( intfName, vrfInUse )
            validAddrFound = False
            for maskedAddr in addrList:
               if maskedAddr != "0.0.0.0/0" and maskedAddr != "::/0":
                  addr = maskedAddr.split( "/" )[ 0 ]
                  if nonPrimaryVrf:
                     conf += "restrict netns %s dst %s %s\n" % \
                             ( VrfName( nonPrimaryVrf ).nsName(), addr,
                               serverModeIntfRestrictions )
                  else:
                     conf += "restrict dst %s %s\n" % \
                             ( addr, serverModeIntfRestrictions )
                  validAddrFound = True
            if not validAddrFound:
               conf += self._interfaceErrorMsg( "block", intfName, nonPrimaryVrf )

         serverModeIntfRestrictions = \
                           serverModeIntfRestrictions.replace( " noclient", "" )

         for intfName in sorted( self.ntpConfig.serverModeEnabledIntf ):
            qt0( "serverModeEnabledIntf", qv( intfName ) )
            serverModeEnabled, nonPrimaryVrf = self._vrfServerModeEnabled( intfName,
                                                                           vrfInUse )
            if serverModeEnabled:
               continue
            if nonPrimaryVrf and not self._validVrf( nonPrimaryVrf ):
               conf += "# Ntp configured interface %s in invalid netns %s\n" % \
                       ( intfName, VrfName( serveVrfName ).nsName() )
               continue
            addrList = self._allIpAddrs( intfName, vrfInUse )
            validAddrFound = False
            for maskedAddr in addrList:
               if maskedAddr != "0.0.0.0/0" and maskedAddr != "::/0":
                  addr = maskedAddr.split( "/" )[ 0 ]
                  if nonPrimaryVrf:
                     conf += "restrict netns %s dst %s %s\n" % \
                             ( VrfName( nonPrimaryVrf ).nsName(), addr,
                               serverModeIntfRestrictions )
                  else:
                     conf += "restrict dst %s %s\n" % \
                             ( addr, serverModeIntfRestrictions )
                  validAddrFound = True
            if not validAddrFound:
               conf += self._interfaceErrorMsg( "accept", intfName, nonPrimaryVrf )

         # Add a line for each server configuration
         # NOTE - the order matters! ntp seems to go through the servers
         # in the order they appear in the ntp.conf file. This means that
         # preferred servers MUST appear first, otherwise they will not
         # necessarily be chosen even if they are available (this happens
         # if ntp has already synchronized to a server of a lower stratum
         # than the preferred server)
         def _cmpServers( serverA, serverB ):
            if serverA.prefer and not serverB.prefer:
               return -1
            elif serverB.prefer and not serverA.prefer:
               return 1
            else:
               return NtpLib.compareVrfAndHost( serverA.vrfAndHost,
                                                serverB.vrfAndHost )
         
         for server in sorted( self.notifier_.server.values(), cmp=_cmpServers ):
            # server source interface overrides the default source interface 
            # i.e if the source interface is specified as part of the ntp server 
            # config use it, else use the default ntp source interface config
            ( sourceIntf, sourceAddr ) = self._serverSourceIntfAddr( server )
            if sourceIntf and not sourceAddr:
               # List the source interface in a comment in case they don't expect
               # the resulting behavior and check the conf file.
               
               conf += ( "# %s was specified as the source address for\n"
                         "# server %s, but it has no IP address or is in\n"
                         "# a different VRF.\n" ) % ( sourceIntf, server.ipOrHost )
               continue

            conf += "server "
            if sourceAddr:
               if self._addressFamily( sourceAddr ) == AF_INET:
                  conf += "-4 "
               else:
                  conf += "-6 "
            conf += ( server.ipOrHost )
            for option in [ "prefer", "burst", "iburst" ]:
               if getattr( server, option ):
                  conf += " " + option
            for option in [ "version", "minpoll", "maxpoll" ]:
               value = getattr( server, option )
               if value:
                  conf += " {option} {value}".format( **locals() )
            if sourceAddr:
               conf += " source %s" % sourceAddr
            if self.notifier_.authMode != authModeNone and server.keyId != 0:
               conf += " key %s" % str( server.keyId )
            conf += "\n"
            if self.notifier_.authMode == authModeServers:
               conf += "restrict %s notrust\n" % server.ipOrHost

      else:
         conf = "# Ntp service not enabled\n"
      return conf
   
   def serviceProcessWarm( self ):
      if not self.serviceEnabled():
         qt5( "Ntp is disabled, and thus warm" )
         return True
      rc = 0
      try:
         NtpLib.runMaybeInNetNs( self.ntpConfig, self.allVrfStatusLocal_,
                                 [ ntpstatBinary() ],
                                 stdout=open( "/dev/null", "w" ),
                                 stderr=subprocess.STDOUT )
      except NtpLib.NetNsError:
         # This means the VRF doesn't exist, but there's no guarantee it will ever
         # be created. So we declare ourselves warm regardless.
         pass
      except Tac.SystemCommandError, e:
         rc = e.error
      # 0 = synchronized; 1 = unsynchronized; 2 = indeterminant
      if rc in [ 0, 1 ]:
         qt5( "Ntp is warm" )
         return True
      else:
         qt5( "Ntp is not yet warm" )
         return False
      
class Ntp( SuperServer.SuperServerAgent ):

   def __init__( self, entityManager ):
      SuperServer.SuperServerAgent.__init__( self, entityManager )
      mg = entityManager.mountGroup()
      self.ntpConfig = mg.mount( 'sys/time/ntp/config', 'Ntp::Config', 'r' )
      self.ntpStatus = mg.mount( 'sys/time/ntp/status', 'Ntp::Status', 'w' )
      netStatus = mg.mount( Cell.path( 'sys/net/status' ), 
                            'System::NetStatus', 'r' )
      Tac.Type( "Ira::IraIpStatusMounter" ).doMountEntities( mg.cMg_, True, True )
      ipStatus = mg.mount( 'ip/status', 'Ip::Status', 'r' )
      ip6Status = mg.mount( 'ip6/status', 'Ip6::Status', 'r' )
      sshConfig = mg.mount( 'mgmt/ssh/config', 'Mgmt::Ssh::Config', 'r' )
      avsl = mg.mount( Cell.path( 'ip/vrf/status/local' ),
                       'Ip::AllVrfStatusLocal', 'r' )
      self.service_ = None

      def _finished():
         self.service_ = NtpService( self.ntpConfig, self.ntpStatus, netStatus,
                                     ipStatus, ip6Status, sshConfig,
                                     avsl, self )

      mg.close( _finished )

   def warm( self ):
      if self.service_:
         return self.service_.warm()
      else:
         return False

   def onSwitchover( self, protocol ):
      self.service_.sync()

def Plugin( ctx ):
   ctx.registerService( Ntp( ctx.entityManager ) )
