# Copyright (c) 2007-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
import Tac
import SuperServer
import QuickTrace
import Arnet
import sys
import Cell
import Arnet.NsLib as NsLib
from NetConfigLib import netnsNameWithUniqueId
import Tracing
from IpLibConsts import DEFAULT_VRF
import PyWrappers.Dnsmasq as dnsmasq
import weakref

t0 = Tracing.trace0
__defaultTraceHandle__ = Tracing.Handle( 'NetworkManager' )

qv = QuickTrace.Var
qt0 = QuickTrace.trace0
qt1 = QuickTrace.trace1

def name():
   return 'NetworkManager'

def addrMapping( line, hostname='' ):
   line = line.strip()
   segments = line.split()
   condition = ( len( segments ) < 2 or line[ 0 ] == '#' )
   if hostname != '':
      condition = condition or ( segments[ 1 ] != hostname )
   if not condition:
      try:
         Arnet.IpAddress( segments[ 0 ] )
         return ( segments[ 0 ], segments[ 1 ] )
      except Exception: # pylint: disable-msg=W0703
         try:
            Arnet.Ip6Addr( segments[ 0 ] )
            return ( segments[ 0 ], segments[ 1 ] )
         except Exception: # pylint: disable-msg=W0703
            pass
   return None

# the reactor class for sysdb object sys/net/config/hostAddr
class HostAddrNotifiee( Tac.Notifiee ):
   """Reactor to the 'sys/net/config/hostAddr' object """
   notifierTypeName = 'System::HostAddr'

   def __init__( self, obj, key, netConfigNotifiee, reservedHosts ):
      Tac.Notifiee.__init__( self, obj )
      self.master_ = netConfigNotifiee
      self.name_ = key
      self.reserved_ = ( self.name_ in reservedHosts )

   def close( self ):
      self.master_.handleHostnameToIpMapping( self.name_, None, deleteAll=True )
      Tac.Notifiee.close( self )

   def handleAddressChange( self, address, ipv6=False ):
      if self.reserved_:
         # if the host is a reserved one ( present in /etc/hosts.Eos)
         qt1( "attempt to add a mapping to the reserve host: ", qv( self.name_ ) )
         return
      self.master_.handleHostnameToIpMapping( self.name_, address, ipv6 )

   @Tac.handler( 'ipAddr' )
   def handleIpAddr( self, address ):
      self.handleAddressChange( address, False )

   @Tac.handler( 'ip6Addr' )
   def handleIp6Addr( self, address ):
      self.handleAddressChange( address, True )

class VrfStatusLocalReactor( Tac.Notifiee ):
   notifierTypeName = 'Ip::VrfStatusLocal'

   def __init__( self, vrfStatusLocal, allVrfStatusLocalReactor ):
      Tac.Notifiee.__init__( self, vrfStatusLocal )
      self.allVrfStatusLocalReactor_ = allVrfStatusLocalReactor
      self.vrfStatusLocal_ = vrfStatusLocal
      self.handleState()

   @Tac.handler( 'state' )
   def handleState( self ):
      qt0( 'VrfStatusLocalReactor.handleVrfState on vrf',
           qv( self.vrfStatusLocal_.vrfName ) )
      t0( 'VrfStatusLocalReactor.handleVrfState on vrf %s' %
           self.vrfStatusLocal_.vrfName )
      self.allVrfStatusLocalReactor_.handleState( self.vrfStatusLocal_.vrfName )

class AllVrfStatusLocalReactor( Tac.Notifiee ):
   notifierTypeName = 'Ip::AllVrfStatusLocal'

   def __init__( self, allVrfStatusLocal, master ):
      Tac.Notifiee.__init__( self, allVrfStatusLocal )
      self.vrfStatusLocalReactors_ = dict()
      self.allVrfStatusLocal_ = allVrfStatusLocal
      self.master_ = master

      for v in allVrfStatusLocal.vrf:
         self.handleVrf( v )

   def handleState( self, vrfName ):
      qt0( 'AllVrfStatusLocalReactor.handleVrfState on vrf ', qv( vrfName ) )
      t0( 'AllVrfStatusLocalReactor.handleVrfState on vrf %s' % vrfName )
      self.master_.handleVrfState( vrfName )

   @Tac.handler( 'vrf' )
   def handleVrf( self, vrf ):
      qt0( 'AllVrfStatusLocalReactor.handleVrf on vrf ', qv( vrf ) )
      t0( 'AllVrfStatusLocalReactor.handleVrf on vrf %s' % vrf )
      if self.allVrfStatusLocal_.vrf.get( vrf ):
         t0( 'vrf %s exists' % vrf )
         if not vrf in self.vrfStatusLocalReactors_:
            self.vrfStatusLocalReactors_[ vrf ] = \
                VrfStatusLocalReactor( self.allVrfStatusLocal_.vrf[ vrf ], self )
            self.master_.handleVrfState( vrf )
      else:
         t0( 'vrf does not exist' )
         if vrf in self.vrfStatusLocalReactors_:
            self.master_.handleVrfState( vrf )
            del self.vrfStatusLocalReactors_[ vrf ]

class NslReactor( Tac.Notifiee ):
   notifierTypeName = 'Arnet::NamespaceStatusLocal'

   def __init__( self, nsl, master ):
      qt0( 'NslReactor::__init__' )
      assert nsl.nsName == NsLib.DEFAULT_NS
      Tac.Notifiee.__init__( self, nsl )
      self.master_ = master
      self.handleState()

   @Tac.handler( 'state' )
   def handleState( self ):
      qt0( 'NslReactor::handleState' )
      self.master_.handleDefaultNs()

class IslReactor( Tac.Notifiee ):
   notifierTypeName = 'Interface::IntfStatusLocal'

   def __init__( self, isl, master ):
      qt0( 'IslReactor::__init__' )
      self.master_ = master
      self.isl_ = isl
      Tac.Notifiee.__init__( self, isl )
      self.handleNetNsName()

   @Tac.handler( 'netNsName' )
   def handleNetNsName( self ):
      qt0( 'IslReactor::handleNetNsName for ', qv( self.isl_.intfId ) )
      self.master_.handleIslNetNsName( self.isl_.intfId )

class IpIntfStatusReactor( Tac.Notifiee ):
   notifierTypeName = 'Ip::IpIntfStatus'

   def __init__( self, iis, master ):
      qt0( 'IpIntfStatusReactor::__init__' )
      self.iis_ = iis
      self.master_ = master
      Tac.Notifiee.__init__( self, iis )
      self.handleActiveAddrWithMask()

   @Tac.handler( 'activeAddrWithMask' )
   def handleActiveAddrWithMask( self ):
      qt0( 'IpIntfStatusReactor::handleActiveAddrWithMask for ',
           qv( self.iis_.intfId ) )
      self.master_.handleActiveAddrWithMask( self.iis_.intfId )

class DnsMasqService( SuperServer.LinuxService ):
   notifierTypeName = 'Ip::AllVrfStatusLocal'

   def __init__( self, allVrfStatusLocal, netStatus, netConfig, master, ipStatus,
                 ip6Status, namespaceStatusLocalDir, allIntfStatusLocalDir ):
      self.master_ = master
      self.allVrfStatusLocal_ = allVrfStatusLocal
      self.netStatus_ = netStatus
      self.netConfig_ = netConfig
      self.ipStatus_ = ipStatus
      self.ip6Status_ = ip6Status
      self.namespaceStatusLocalDir_ = namespaceStatusLocalDir
      self.allIntfStatusLocalDir_ = allIntfStatusLocalDir
      self.started_ = False

      SuperServer.LinuxService.__init__( self, dnsmasq.service(),
                                         dnsmasq.service(),
                                         self.allVrfStatusLocal_,
                                         "/etc/dnsmasq.conf", sync=False )

   def serviceProcessWarm( self ):
      return True

   def defaultNsInitComplete( self ):
      nsld = self.namespaceStatusLocalDir_
      qt0( 'defaultNsInitComplete in ld is ', qv( NsLib.DEFAULT_NS in nsld.ns ) )
      if NsLib.DEFAULT_NS in nsld.ns:
         qt0( 'state is ', qv( nsld.ns[ NsLib.DEFAULT_NS ].state ) )
      return ( NsLib.DEFAULT_NS in nsld.ns and
               nsld.ns[ NsLib.DEFAULT_NS ].state == 'active' )

   def serviceEnabled( self ):
      enabled = ( ( len( self.netStatus_.nameServer ) > 0 or
                    len( self.netStatus_.v6NameServer ) > 0 ) and
                  self.defaultNsInitComplete() )

      t0( 'dnsmasq serviceEnabled is %s (defaultNsInitComplete is %s)' %
          ( enabled, self.defaultNsInitComplete() ) )
      qt0( 'dnsmasq serviceEnabled is ', qv( enabled ),
           ' defaultNsInitComplete is ', qv( self.defaultNsInitComplete() ) )
      self.netStatus_.dnsmasqEnabled = enabled
      return enabled

   def conf( self ):
      # there is a race condition here because we validate that the
      # VRF namespaces exists when we write the config file, but there
      # is no way to lock them to force them to stick around until
      # dnsmasq has started and is trying to use them.  When a VRF is
      # torn down, it sometimes churns sysdb enough that we restart
      # dnsmasq right before we delete the namespace, thus causing it
      # to print a few error messages about being unable to access the
      # files in /var/run/netns.  This is hard to fix because we don't
      # really want to put in an active lock mechanism to stop Ira
      # from doing this, so we just live with it.  It's harmless,
      # except for noise in the log messages.

      t0( 'making conf file for dnsmasq' )
      qt0( 'making conf file for dnsmasq' )
      cfg = "\nno-resolv\n"

      # this is where the DNS servers are, and where we proxy the requests to
      serverNs = NsLib.DEFAULT_NS
      # This is the placeholder for listen namespaces that we proxy rquests from.
      listenNs = []

      sourceIpAddr = None

      assert self.netStatus_.vrfName != ''
      if ( self.netStatus_.vrfName != DEFAULT_VRF and
           self.master_.vrfExists( self.netStatus_.vrfName ) ):
         # this means we're proxying from vrf to main
         vrfStatus = self.allVrfStatusLocal_.vrf[ self.netStatus_.vrfName ]
         serverNs = vrfStatus.networkNamespace
         # we have to add a listen socket for this namespace too, to listen
         # for dns requests
         t0( 'dnsmasq::conf adding %s to listen ns set' % serverNs )

      # now identify the set of all namespaces -- we're listening in
      # all of them now because of source address support
      for ( vrf, vrfStatusLocal ) in self.allVrfStatusLocal_.vrf.items():
         t0( 'dnsmasq::conf  checking %s to see if we need to listen' % vrf )
         if self.master_.vrfExists( vrf ):
            t0( 'dnsmasq::conf vrf %s exists' % vrf )
            netNs = vrfStatusLocal.networkNamespace
            assert netNs != ""
            t0( 'dnsmasq::conf  adding %s to listen namespace list' % netNs )
            listenNs.append( netNs )
         else:
            t0( 'dnsmasq::conf vrf %s does not exist' % vrf )

      def findSourceIpAddr( vrf, serverNs, proto, destAddr=None ):
         # source address is valid if:
         #
         # - DNS is configured to use source intf in main VRF (Non-Default
         #   VRF not supported yet)
         # - intf is actually configured into the main VRF (namespace must
         #   match)
         # - intf has a non-zero IP address
         #
         # We have reactors to IntfStatusLocal and IpIntfStatus to
         # cover changes in the last two, in addition to the
         # NetConfigNotifiee.  Note that we do not bother to check the
         # interface's OperStatus because in the event that it gets
         # shutdown the IP address will be converted to 0.0.0.0 by
         # Ira, so this is already covered
         #
         if vrf not in self.netStatus_.sourceIntf:
            t0( 'no source interface configured for vrf %s' % vrf )
            return None

         intfId = self.netStatus_.sourceIntf[ vrf ]
         t0( 'Source interface for vrf %s is %s' % ( vrf, intfId ) )
         if not intfId:
            return None

         aisl = self.allIntfStatusLocalDir_.intfStatusLocal
         intfNs = None
         if intfId not in aisl:
            # if there is no IntfStatusLocal, then that means the intf
            # is in the default namespace
            intfNs = NsLib.DEFAULT_NS
            t0( 'intf %s not in aisl, defaulting to %s' %
                ( intfId, NsLib.DEFAULT_NS ) )
         else:
            intfNs = aisl[ intfId ].netNsName
            t0( 'intf %s is in namespace %s' % ( intfId, intfNs ) )

         if intfNs != serverNs:
            t0( 'intf namespace %s is not server ns %s, cannot use as source' %
                ( intfNs, serverNs ) )
            return None

         if proto == 4:
            if intfId not in self.ipStatus_.ipIntfStatus:
               t0( 'intf %s not in ipIntfStatus, cannot use as source' % intfId )
               return None

            sourceIpAddr = self.ipStatus_.ipIntfStatus[ intfId ].\
                activeAddrWithMask.address
            if sourceIpAddr == '0.0.0.0':
               t0( 'intf %s active address is 0.0.0.0, ignoring' % intfId )
               return None

         else:
            assert proto == 6
            assert destAddr
            ip6is = self.ip6Status_.intf.get( intfId )
            if not ip6is:
               t0( 'intf %s not in ip6 intf status, cannot use as source' % intfId )
               return None

            sourceIpAddr = ip6is.sourceAddressForDestination( destAddr )
            if sourceIpAddr == '::':
               t0( 'sourceAddressForDestination returned ::, cannot use' )
               return None

         t0( 'intf %s active address is %s, using as source for dest %s' %
             ( intfId, sourceIpAddr, destAddr ) )
         return sourceIpAddr

      # when we support having servers in multiple VRFs at once we'll
      # need to iterate through these -- for now we only have one
      sourceIpAddr = findSourceIpAddr( self.netStatus_.vrfName, serverNs, proto=4 )

      cfg += "server-namespace=%s\n" % serverNs
      # we want dnsmasq to always listen in the default namespace
      # and hence keep the listen-namespace entry for the default namespace
      # ahead of listen-namespace entries for other namespaces
      cfg += "listen-namespace=%s\n" % NsLib.DEFAULT_NS
      for ns in listenNs:
         if ns != NsLib.DEFAULT_NS:
            cfg += "listen-namespace=%s\n" % netnsNameWithUniqueId( ns )

      t0( 'number of nameservers is v4: %s, v6: %s' %
          ( len( self.netStatus_.nameServer ),
            len( self.netStatus_.v6NameServer ) ) )

      maxIDs = Tac.Type( "System::NameServerId" ).max
      for i in range( 1, maxIDs + 1 ):
         if i not in self.netStatus_.nameServer:
            continue
         v4NameServer = self.netStatus_.nameServer[ i ]

         srcIpAddrStr = ''
         if sourceIpAddr:
            srcIpAddrStr = '@%s' % sourceIpAddr

         t0( 'adding server %s with sa %s to dnsmasq.conf' %
             ( v4NameServer, sourceIpAddr ) )
         cfg += "server=%s%s\n" % ( v4NameServer, srcIpAddrStr )

      for i in range( 1, maxIDs + 1 ):
         if i not in self.netStatus_.v6NameServer:
            continue
         v6NameServer = self.netStatus_.v6NameServer[ i ]

         srcIpAddrStr = ''
         sourceIp6Addr = findSourceIpAddr( self.netStatus_.vrfName, serverNs,
                                           proto=6, destAddr=v6NameServer )
         if sourceIp6Addr:
            srcIpAddrStr = '@%s' % sourceIp6Addr

         t0( 'adding server %s with sa %s to dnsmasq.conf' %
             ( v6NameServer, sourceIp6Addr ) )
         cfg += "server=%s%s\n" % ( v6NameServer, srcIpAddrStr )

      if self.netStatus_.externalDnsProxy:
         cfg += 'external-proxy=1'
      else:
         cfg += 'external-proxy=0'

      t0( 'dnsmasq.conf is %s' % cfg )

      return cfg

   def startService( self ):
      SuperServer.LinuxService.startService( self )
      self.started_ = True

   def stopService( self ):
      SuperServer.LinuxService.stopService( self )
      self.started_ = False

   # don't need to override restart because we don't SIGHUP dnsmasq

class NetConfigNotifiee( SuperServer.GenericService ): # pylint: disable-msg=W0223
   """ Reactor to the 'sys/net/config' object that sets up the hostname, dns
   appropriately based on system configuration. """

   notifierTypeName = 'System::NetConfig'

   def __init__( self, serviceName, config, status, allVrfStatusLocal, ipStatus,
                 ip6Status, nsStatusLocalDir, allIntfStatusLocalDir ):
      self.config_ = config
      self.status_ = status
      self.allVrfStatusLocal_ = allVrfStatusLocal
      self.ipStatus_ = ipStatus
      self.ip6Status_ = ip6Status
      self.nsStatusLocalDir_ = nsStatusLocalDir
      self.allIntfStatusLocalDir_ = allIntfStatusLocalDir
      self.activity_ = None
      self.activityInterval_ = 60
      self.intfStatusReactor_ = None
      self.ipIntfStatusReactor_ = None

      SuperServer.GenericService.__init__( self, serviceName, self.config_,
                                           sync=False )

      reservedHosts = self.getReservedHostnames( )

      # Store the contents of hosts.Eos which should be copied as is
      # to /etc/hosts file first and then user configured host mappings
      # are added.
      self.hostsEosData = ''
      try:
         desc = open( '/etc/hosts.Eos', 'r' )
         self.hostsEosData = desc.read()
         desc.close()
      except OSError, e:
         qt0( "Error occurred opening Eos host file:", qv( e ) )
         t0( "Error occurred opening Eos host file: %s" % e )
      except:
         e = sys.exc_info()[ 1 ]
         qt0( "Error occurred opening Eos host file:", ( e ) )
         t0( "Error occurred opening Eos host file: %s" % e )

      # init this to none so that we can check in handleDefaultNs/etc.
      # Otherwise we wind up with a 'no attribute' error when we get
      # the callback from the __init__() function in NslReactor/etc
      self.dnsMasqService_ = None

      # Set of tuple of (IpAddress, hostname). Note that same hostname can have
      # multiple ip address mapping and same ip address can be mapped to
      # multiple hostnames.
      self.hostMappings = set( [] )
      self.syncHostAddrStatus( self.config_, status, reservedHosts )
      self.hostAddrReactor_ = Tac.collectionChangeReactor(
                              self.notifier_.hostAddr, HostAddrNotifiee,
                              reactorArgs=( self, reservedHosts ),
                              reactorTakesKeyArg=True )
      self.nslReactor_ = Tac.collectionChangeReactor(
         self.nsStatusLocalDir_.ns, NslReactor,
         reactorArgs=( weakref.proxy( self ), ),
         reactorFilter=lambda nsName: nsName == NsLib.DEFAULT_NS )

      self.allVrfStatusLocalReactor_ = \
          AllVrfStatusLocalReactor( self.allVrfStatusLocal_, self )
      # Do this last as it depends on NetStatus to be in sync with NetConfig
      self.dnsMasqService_ = DnsMasqService( self.allVrfStatusLocal_, self.status_,
                                             self.config_, self, self.ipStatus_,
                                             self.ip6Status_,
                                             self.nsStatusLocalDir_,
                                             self.allIntfStatusLocalDir_ )

      for k in self.notifier_.sourceIntf:
         self.handleSourceIntf( k )
      self.handleHostname()
      self.handleVrfName()
      self.handleExternalDnsProxy()

   def vrfExists( self, vrf ):
      assert vrf != None and vrf != ''
      if vrf == DEFAULT_VRF:
         return True

      if not self.allVrfStatusLocal_.vrf.get( vrf ):
         return False
      if not self.allVrfStatusLocal_.vrf[ vrf ].state == 'active':
         return False
      return True

   # return number of non-default VRFs that exist and are active
   def numVrfs( self ):
      n = 0
      for vrf in self.allVrfStatusLocal_.vrf.keys():
         if self.vrfExists( vrf ):
            n = n + 1
      return n

   def handleVrfState( self, vrf ):
      assert vrf and vrf != ''
      t0( 'NetConfigNotifiee.handleVrfState for %s' % vrf )
      if self.dnsMasqService_:
         self.sync()

   def handleDefaultNs( self ):
      qt0( 'NetConfigNotifiee.handleDefaultNs' )
      t0( 'NetConfigNotifiee.handleDefaultNs' )
      if self.dnsMasqService_:
         self.dnsMasqService_.sync()

   def handleIslNetNsName( self, intfId ):
      t0( 'NetConfigNotifiee.handleIslNetNsName for %s' % intfId )
      if self.dnsMasqService_:
         self.dnsMasqService_.sync()

   def handleActiveAddrWithMask( self, intfId ):
      t0( 'NetConfigNotifiee.handleActiveAddrWithMask for %s' % intfId )
      if self.dnsMasqService_:
         self.dnsMasqService_.sync()

   def sync( self ):
      t0( 'syncing on NetConfigNotifiee' )
      qt0( 'syncing on NetConfigNotifiee' )
      # note that we have to sync the netStatus first, because the
      # rest of the sync code uses the results of this so that we can
      # avoid replicating checks
      self.syncNetStatusToSysdb()
      # The hostname config is update with an explicit handler
      self.syncResolvConf()
      self.dnsMasqService_.sync()

   def syncResolvConf( self ):
      t0( 'NetConfigNotifiee.syncResolvConf' )
      qt0( 'NetConfigNotifiee.syncResolvConf' )
      return self.writeConfigFile( '/etc/resolv.conf', self.resolvConfConfig(),
                                   updateInPlace=True )

   def legalDomainList( self ):
      '''Returns a list of domains that should be printed to the search path.
      Assumes that the domain and character limits on the search path were
      enforced by the Cli.
      '''
      domains = self.notifier_.domainList.values()
      if self.notifier_.domainName:
         if self.notifier_.domainName in domains:
            domains.remove( self.notifier_.domainName )
         domains.insert( 0, self.notifier_.domainName )
      return domains

   def resolvConfConfig( self ):
      t0( 'generating resolv.conf' )
      config = ""

      def _domainSearchPath():
         '''Returns the resolv.conf domain search path configuration option.
         Returns the empty string if there are no domains to search.
         '''
         searchPath = ' '.join( self.legalDomainList() )
         if searchPath:
            return "search %s\n" % searchPath
         else:
            return ''

      config += "\n"
      config += "#\n"
      config += "# As of EOS-4.11.0, all DNS requestes are now proxied through\n"
      config += "# dnsmasq, so that it can implement the DNS source interface\n"
      config += "# feature.  This means that the nameservers in use are now listed\n"
      config += "# in /etc/dnsmasq.conf.\n"
      config += "#\n"
      config += "\n"

      searchPath = _domainSearchPath()
      t0( 'domain search path is %s' % searchPath )
      config += searchPath
      if( len( self.status_.nameServer ) > 0 and
          self.dnsMasqService_.defaultNsInitComplete() ):
         t0( 'adding 127.0.0.1 to resolv.conf' )
         config += "nameserver 127.0.0.1\n"
      else:
         t0( 'skipping 127.0.0.1, num v4 servers is %d, ns init is %s' %
             ( len( self.status_.nameServer ),
               self.dnsMasqService_.defaultNsInitComplete() ) )

      if( len( self.status_.v6NameServer ) > 0 and
          self.dnsMasqService_.defaultNsInitComplete() ):
         t0( 'adding ::1 to resolv.conf' )
         config += "nameserver ::1\n"
      else:
         t0( 'skipping ::1, num v6 servers is %d, ns init is %s' %
             ( len( self.status_.v6NameServer ),
               self.dnsMasqService_.defaultNsInitComplete() ) )

      t0( 'config is %s' % config )

      return config

   def syncNetStatusToSysdb( self ):
      t0( 'syncNetStatusToSysdb for %s' % self.notifier_.vrfName )
      qt0( 'syncNetStatusToSysdb for ', qv( self.notifier_.vrfName ) )
      for nsid in self.status_.nameServer:
         del self.status_.nameServer[ nsid ]
      for nsid in self.status_.v6NameServer:
         del self.status_.v6NameServer[ nsid ]

      if self.vrfExists( self.notifier_.vrfName ):
         for ( nsid, v4NameServer ) in self.notifier_.nameServer.items():
            t0( 'syncing %s as %d' % ( v4NameServer, nsid ) )
            qt0( 'syncing ', qv( v4NameServer ), ' as ', qv( nsid ) )
            self.status_.nameServer[ nsid ] = v4NameServer
         for ( nsid, v6NameServer ) in self.notifier_.v6NameServer.items():
            t0( 'syncing %s as %d' % ( v6NameServer, nsid ) )
            qt0( 'syncing ', qv( v6NameServer ), ' as ', qv( nsid ) )
            self.status_.v6NameServer[ nsid ] = v6NameServer

         self.status_.vrfName = self.notifier_.vrfName
      else:
         self.status_.vrfName = DEFAULT_VRF

      self.status_.domainName = self.notifier_.domainName

   @Tac.handler( 'sourceIntf' )
   def handleSourceIntf( self, vrf ):
      t0( 'NetConfigNotifiee.handleSourceIntf for %s' % vrf )
      qt0( 'NetConfigNotifiee.handleSourceIntf for ', qv( vrf ) )
      # create reactor for this interface's intfStatus and ipIntfStatus
      if self.intfStatusReactor_:
         self.intfStatusReactor_.close()
         self.intfStatusReactor_ = None
      if self.ipIntfStatusReactor_:
         self.ipIntfStatusReactor_.close()
         self.ipIntfStatusReactor_ = None

      assert vrf is not None
      if vrf in self.notifier_.sourceIntf:
         # we just copy this straight over into the status, the
         # dnsmasq generation code handles figuring out whether or not
         # we should actually use it based on the rest of the state in
         # the system
         self.status_.sourceIntf[ vrf ] = self.notifier_.sourceIntf[ vrf ]

         def _intfFilter( intfId ):
            return intfId == self.notifier_.sourceIntf[ vrf ]

         self.intfStatusReactor_ = Tac.collectionChangeReactor(
            self.allIntfStatusLocalDir_.intfStatusLocal, IslReactor,
            reactorArgs=( weakref.proxy( self ), ), reactorFilter=_intfFilter )

         self.ipIntfStatusReactor_ = Tac.collectionChangeReactor(
            self.ipStatus_.ipIntfStatus, IpIntfStatusReactor,
            reactorArgs=( weakref.proxy( self ), ), reactorFilter=_intfFilter )
      else:
         del self.status_.sourceIntf[ vrf ]

      self.sync()

   @Tac.handler( 'vrfName' )
   def handleVrfName( self ):
      t0( 'NetConfigNotifiee.handleVrfName' )
      assert self.notifier_.vrfName != ''
      self.sync()

   @Tac.handler( 'externalDnsProxy' )
   def handleExternalDnsProxy( self ):
      t0( 'NetConfigNotifiee.handleExternalDnsProxy' )
      self.status_.externalDnsProxy = self.config_.externalDnsProxy
      self.sync()

   @Tac.handler( 'hostname' )
   def handleHostname( self ):
      hostname = self.notifier_.hostname or 'localhost'

      # If the hostname contains '_', we replace it with '-' and program it.
      # Even though it's not a valid hostname, some customers do want to use it.
      sysHostname = hostname.replace( '_', '-' )
      curHostname = Tac.run( [ 'hostname' ], stdout=Tac.CAPTURE,
                             timeout=int( self.config_.hostnameTimeout ) ).strip()
      if sysHostname != curHostname:
         qt0( "Setting hostname to", qv( sysHostname ) )
         t0( "Setting hostname to %s" % sysHostname )
         try:
            Tac.run( [ 'hostname', sysHostname ], stdout=Tac.CAPTURE,
                     timeout=int( self.config_.hostnameTimeout ) )
         except Tac.SystemCommandError, e:
            # XXX_APECH What to do in this case?
            qt0( "Failed to set hostname. Output:", qv( e.output ) )
            t0( "Failed to set hostname. Output: %", e.output )
            return
      else:
         qt0( "Linux hostname is already set to", qv( sysHostname ) )
         t0( "Linux hostname is already set to %s", sysHostname )

      # Update the System::NetStatus object
      self.status_.hostname = hostname

   def writeToHostsFile( self ):
      mapData = self.hostsEosData + '\n'
      for mapping in self.hostMappings:
         mapData = mapData + '%s    %s' % ( mapping[ 0 ], mapping[ 1 ] ) + '\n'
      if not self.writeConfigFile( '/etc/hosts', mapData ):
         if self.activity_:
            self.activity_.timeMin = min( self.activity_.timeMin,
                                          Tac.now() + self.activityInterval_ )
         else:
            self.activity_ = Tac.ClockNotifiee()
            self.activity_.handler = self.writeToHostsFile
            self.activity_.timeMin = Tac.now() + self.activityInterval_

   def syncHostAddrStatus( self, config, status, reservedHosts ):
      # Syncs the hostAddr in status with what is present in config
      # Also populates the hostMappings used for writing to /etc/hosts
      for _name, hostObject in config.hostAddr.items():
         if not _name in reservedHosts and not _name in status.hostAddr:
            newEntry = status.newHostAddr( _name )
            for ipAddr in hostObject.ipAddr:
               newEntry.ipAddr[ ipAddr ] = True
            for ip6Addr in hostObject.ip6Addr:
               newEntry.ip6Addr[ ip6Addr ] = True

      for _name, hostObject in status.hostAddr.items():
         if not _name in config.hostAddr:
            del status.hostAddr[ _name ]
            continue
         for ipAddr in hostObject.ipAddr:
            self.hostMappings.add( ( ipAddr, _name ) )
         for ip6Addr in hostObject.ip6Addr:
            self.hostMappings.add( ( ip6Addr.stringValue, _name ) )
      self.writeToHostsFile()

   # Returns a list of reserved hostnames by reading /etc/hosts.Eos
   # These are always present in /etc/hosts.
   def getReservedHostnames( self ):
      reservedHosts = []
      try:
         hostsEosRead = open( '/etc/hosts.Eos', 'r' )
         line = hostsEosRead.readline()
         while line != '':
            if addrMapping( line ):
               segments = line.split()
               segments.remove( segments[ 0 ] )
               for _name in segments:
                  reservedHosts.append( _name )
            line = hostsEosRead.readline()
         hostsEosRead.close()
      except OSError, e:
         qt0( "Error occurred opening Eos host file:", qv( e ) )
         t0( "Error occurred opening Eos host file: %s" % e )
      except:
         e = sys.exc_info()[ 1 ]
         qt0( "Error occurred opening Eos host file:", ( e ) )
         t0( "Error occurred opening Eos host file: %s" % e )
      return reservedHosts

   def handleHostnameToIpMapping( self, hostname, address, ipv6=False,
                                 deleteAll=False ):
      if deleteAll:
         # hostAddr object for given name must not be present in netConfig
         # delete hostname from netStatus and remove all mappings for it
         qt0( "removing all the mapping for host: ", qv( hostname ) )
         t0( "removing all the mapping for host: %s" % hostname )
         del self.status_.hostAddr[ hostname ]
         rewriteEtcHosts = False
         removeMappings = [ mapping for mapping in self.hostMappings if
                     mapping[ 1 ] == hostname ]
         for mapping in removeMappings:
            self.hostMappings.remove( mapping )
            rewriteEtcHosts = True

         if rewriteEtcHosts:
            self.writeToHostsFile()
         return

      if not hostname in self.status_.hostAddr:
         # if a new hostAddr object is added to netConfig
         # create a new one in netStatus also
         self.status_.newHostAddr( hostname )
         qt0( "adding a new HostAddr object '", qv( hostname ), "' to netStatus" )
         t0( "adding a new HostAddr object '%s' to netStatus" % hostname )

      if ipv6:
         configIpList = self.notifier_.hostAddr[ hostname ].ip6Addr
         statusIpList = self.status_.hostAddr[ hostname ].ip6Addr
      else:
         configIpList = self.notifier_.hostAddr[ hostname ].ipAddr
         statusIpList = self.status_.hostAddr[ hostname ].ipAddr

      rewriteEtcHosts = False

      if address is not None:
         addrString = address.stringValue if ipv6 else str( address )
         if address in configIpList:
            # new mapping is added to netConfig
            qt0( "adding a new mapping: ", qv( hostname ), "->", qv( addrString ) )
            t0( "adding a new mapping: %s -> %s" % ( hostname, addrString ) )
            statusIpList[ address ] = True
            self.hostMappings.add( ( addrString, hostname ) )
            rewriteEtcHosts = True
         else:
            # a mapping is deleted from netConfig
            del statusIpList[ address ]
            qt0( "removing the mapping: ", qv( hostname ), "->", qv( addrString ) )
            t0( "removing the mapping: %s -> %s" % ( hostname, addrString ) )
            self.hostMappings.discard( ( addrString, hostname ) )
            rewriteEtcHosts = True
            # deleting the HostAddr object if it has no more mappings
            if not ( self.status_.hostAddr[ hostname ].ipAddr or
                     self.status_.hostAddr[ hostname ].ip6Addr ):
               del self.status_.hostAddr[ hostname ]
      else:
         # synchronising all the addresses when None is passed as address/key
         qt0( "synchronising all ", qv( "ipv6" if ipv6 else "ipv4" ),
               " mappings for host: ", qv( hostname ) )
         t0( "synchronising all %s mappings for host %s" %
             ( "ipv6" if ipv6 else "ipv4", hostname ) )
         for address in statusIpList.keys():
            addrString = address.stringValue if ipv6 else str( address )
            if not address in configIpList:
               del statusIpList[ address ]
               self.hostMappings.discard( ( addrString, hostname ) )
               rewriteEtcHosts = True
         for address in configIpList.keys():
            addrString = address.stringValue if ipv6 else str( address )
            if not address in statusIpList:
               statusIpList[ address ] = True
               self.hostMappings.add( ( addrString, hostname ) )
               rewriteEtcHosts = True

      if rewriteEtcHosts:
         self.writeToHostsFile()

class NetStatusNotifiee( Tac.Notifiee ):
   """ Reactor to the 'sys/net/status' object that sets up the fqdn
   appropriately based on system configuration. """

   notifierTypeName = 'System::NetStatus'

   def __init__( self, notifier ):
      Tac.Notifiee.__init__( self, notifier )
      self.handleHostname()

   @Tac.handler( 'hostname' )
   def handleHostname( self ):
      hostname = self.notifier_.hostname
      domainName = self.notifier_.domainName
      if domainName and not hostname.endswith( domainName ):
         self.notifier_.fqdn = hostname + '.' + domainName
      else:
         self.notifier_.fqdn = hostname

   @Tac.handler( 'domainName' )
   def handleDomainName( self ):
      hostname = self.notifier_.hostname
      domainName = self.notifier_.domainName
      if not hostname.endswith( domainName ) and domainName:
         self.notifier_.fqdn = hostname + '.' + domainName
      else:
         self.notifier_.fqdn = hostname

class NetworkManager( SuperServer.SuperServerAgent ):
   """ This agent reacts to the 'system/net/config' object and configures
   hostname, dns, etc on the host """
   def __init__( self, entityManager ):
      SuperServer.SuperServerAgent.__init__( self, entityManager )
      mg = entityManager.mountGroup()
      networkConfig = mg.mount( 'sys/net/config', 'System::NetConfig', 'r' )
      networkStatus = mg.mount( Cell.path( 'sys/net/status' ),
                                'System::NetStatus', 'wf' )
      avsl = mg.mount( Cell.path( 'ip/vrf/status/local' ),
                       'Ip::AllVrfStatusLocal', 'r' )
      Tac.Type( "Ira::IraIpStatusMounter" ).doMountEntities( mg.cMg_, True, True )
      ipStatus = mg.mount( 'ip/status', 'Ip::Status', 'r' )
      ip6Status = mg.mount( 'ip6/status', 'Ip6::Status', 'r' )
      nsld = mg.mount( Cell.path( 'namespace/status' ),
                       'Arnet::NamespaceStatusLocalDir', 'r' )
      # Get local interface entity, created by SuperServer
      aisld = self.intfStatusLocal()
      self.fqdnNotifiee_ = None
      self.service_ = None
      self.status_ = None

      def _finished():
         qt0( 'NetworkManager::_finished' )
         self.status_ = networkStatus
         self.fqdnNotifiee_ = NetStatusNotifiee( networkStatus )
         self.service_ = NetConfigNotifiee( "NetConfig", networkConfig,
                                            networkStatus, avsl, ipStatus,
                                            ip6Status, nsld, aisld )
      mg.close( _finished )

def Plugin( ctx ):
   ctx.registerService( NetworkManager( ctx.entityManager ) )
