# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tracing
import Logging
import SuperServer
import Tac
import ArchiveLib
import SpaceMgmtLib
import CEosHelper
import collections
import os

# pylint: disable-msg=W1401
traceHandle = Tracing.Handle( "LogArchiver" )
t0 = traceHandle.trace0

EOS_ARCHIVE_NO_CONFIG_ERROR = None
EOS_ARCHIVE_INV_CONFIG_ERROR = None
EOS_ARCHIVE_QUOTACMD_ERROR = None
EOS_ARCHIVE_FS_ERROR = None

Logging.logD( 'EOS_ARCHIVE_NO_CONFIG_ERROR',
              severity=Logging.logError,
              format='Unable to get the currently configured archive',
              explanation=( 'File %s does not exist.'
                            % ArchiveLib.Archive.configFilePath ),
              recommendedAction=Logging.CALL_SUPPORT_IF_PERSISTS )

Logging.logD( 'EOS_ARCHIVE_INV_CONFIG_ERROR',
              severity=Logging.logError,
              format='Unable to use the currently configured archive',
              explanation=( 'Unable to read the configuration file %s'
                            ' or its content is invalid.'
                            % ArchiveLib.Archive.configFilePath ),
              recommendedAction=Logging.CALL_SUPPORT_IF_PERSISTS )

Logging.logD( 'EOS_ARCHIVE_QUOTACMD_ERROR',
              severity=Logging.logError,
              format='An error occured managing linux quota: %s',
              explanation=( 'Updating quota limit failed. It could mean that the'
                            ' filesystem is mounted without quota options or that'
                            ' the quota database file aquota.user is missing or'
                            ' corrupted.' ),
              recommendedAction=Logging.CALL_SUPPORT_IF_PERSISTS )

Logging.logD( 'EOS_ARCHIVE_FS_ERROR',
              severity=Logging.logError,
              format=( 'An error occurred getting filesystem information'
                       ' for the archive destination path: %s' ),
              explanation=( 'We are unable to get filesystem information for the'
                            ' destination path or the mountpoint of the filesystem.'
                            ' This means the path does not exist or the filesystem'
                            ' is not mounted.' ),
              recommendedAction=Logging.CALL_SUPPORT_IF_PERSISTS )

class LogArchiverConfigReactor( Tac.Notifiee ):
   """
   Reactor to process the change in quota pct or the change in archive status.
   The handler will invoke archive code to handle these.
   """

   notifierTypeName = 'Mgmt::Archive::Config'
   superServerArchiveLockTimeout = 30

   def __init__( self, archiveConfig ):
      """The __init__ initialization method for LogArchiverConfigReactor."""

      Tac.Notifiee.__init__( self, archiveConfig )
      self._config = archiveConfig
      self.actionQueue = collections.deque( [] )
      self.clockNotifie = Tac.ClockNotifiee( timeMin=Tac.endOfTime,
                                             handler=self.flushActionQueue )
      os.environ[ ArchiveLib.Archive.archiveLockTimeoutEnvVar ] = (
         str( LogArchiverConfigReactor.superServerArchiveLockTimeout ) )

   def currentArchive( self ):
      try:
         t0( 'retrieve current archive' )
         archive = ArchiveLib.Archive.currentArchive()
      except AssertionError as e:
         Logging.log( EOS_ARCHIVE_FS_ERROR, e )
         return None

      if archive is None:
         Logging.log( EOS_ARCHIVE_INV_CONFIG_ERROR )
      elif archive is False:
         Logging.log( EOS_ARCHIVE_NO_CONFIG_ERROR )

      return archive

   def flushActionQueue( self ):
      reschedulingTime = 10
      startTime = Tac.now()
      execTime = lambda: Tac.now() - startTime

      try:
         # If we stay in this handler too long, SuperServer is going to miss a
         # heartbeat and get killed. See BUG289813.
         while self.actionQueue and execTime() < 10:
            action = self.actionQueue.popleft()
            t0( 'exec %s (try number %d)'
                % ( action[ 'func' ].__name__, action[ 'try' ] ) )
            action[ 'func' ]( action[ 'archive' ] )
      except Tac.Timeout:
         t0( 'timeout executing %s (try number %d)'
              % ( action[ 'func' ].__name__, action[ 'try' ] ) )
         action[ 'try' ] += 1
         self.actionQueue.appendleft( action )
         t0( 'rescheduling clock notifie in %d secs' % reschedulingTime )
         self.clockNotifie.timeMin = Tac.now() + reschedulingTime
      else:
         if self.actionQueue:
            t0( 'rescheduling clock notifie in 1 secs for heartbeat' )
            self.clockNotifie.timeMin = Tac.now() + 1

   def enqueueAction( self, archive, func ):
      t0( 'enqueue action for %s' % func.__name__ )
      if not self.actionQueue:
         self.clockNotifie.timeMin = 0
      self.actionQueue.append( { 'func': func, 'archive': archive, 'try': 1 } )

   def doShutdown( self, archive ):
      if self._config.shutdown:
         t0( 'disable archive' )
         archive.disable()
      else:
         t0( 'enable archive' )
         archive.enable()

   @Tac.handler( 'shutdown' )
   def handleArchiveShutdown( self ):
      """Handle update to management archive shutdown/noshutdown."""

      t0( 'archive config shutdown state changed to', self._config.shutdown )

      archive = self.currentArchive()
      if not archive:
         t0( 'unable to get current archive' )
         return

      self.enqueueAction( archive, self.doShutdown )

   def doQuotaPct( self, archive ):
      try:
         t0( 'set archive quota pct to', self._config.quotapct )
         archive.setQuotaPct( self._config.quotapct )
         t0( 'done set archive quota pct to', self._config.quotapct )
      except SpaceMgmtLib.Quota.QuotaCmdException as e:
         t0( 'updating quota failed because of a QUOTACMD_ERROR' )
         Logging.log( EOS_ARCHIVE_QUOTACMD_ERROR, e )
      except AssertionError as e:
         t0( 'updating quota failed because of an FS_ERROR' )
         Logging.log( EOS_ARCHIVE_FS_ERROR, e )

   @Tac.handler( 'quotapct' )
   def handleArchiveQuotaPct( self ):
      """Handle update to management archive quotapct [0-100]."""

      t0( 'archive config quotapct changed to', self._config.quotapct )

      archive = self.currentArchive()
      if not archive:
         t0( 'unable to get current archive' )
         return

      if CEosHelper.isCeos():
         t0( 'can not set quota percentage in ceos' )
         return

      self.enqueueAction( archive, self.doQuotaPct )

   def _overrideArchiveConfigFile( self, destName, destPath, enabled ):
      t0( 'overriding archive config file' )
      Tac.run( [ 'rm', '-f', ArchiveLib.Archive.configFilePath ],
               asRoot=True,
               ignoreReturnCode=True,
               stdout=Tac.DISCARD,
               stderr=Tac.DISCARD )
      try:
         ArchiveLib.Archive.writeConfig( destName, destPath, None, enabled )
      except ( IOError, OSError, ValueError ) as e:
         t0( 'failed to write archive config file:', e )
         return False
      return True

   def doDestSetup( self, archive ):
      try:
         if archive.enabled:
            t0( 'rotate any existing archive directory and setup archive' )
            archive.setup( rotate=True, ignoreConfig=True )
         else:
            t0( 'rotate any existing archive directory' )
            archive.rotateArchiveDir()
      except ( SpaceMgmtLib.Quota.QuotaCmdException,
               Tac.SystemCommandError ) as e:
         Logging.log( EOS_ARCHIVE_QUOTACMD_ERROR, e )
      except ( IOError, OSError, AssertionError ) as e:
         Logging.log( EOS_ARCHIVE_FS_ERROR, e )

   @Tac.handler( 'dest' )
   def handleArchiveDest( self ):
      """Handle update to management archive destination."""

      t0( "archive config destination changed to '%s'" % self._config.dest )

      if self._config.dest == '':
         ArchiveLib.Archive.updateConfig( name='', path='', quotaPct=None )
      else:
         try:
            destName, destPath = self._config.dest.split( ':' )
         except ValueError as e:
            t0( 'invalid archive config dest format:', self._config.dest )
            return

         try:
            t0( 'update archive config file' )
            ArchiveLib.Archive.updateConfig( name=destName,
                                             path=destPath,
                                             quotaPct=None )
         except ( IOError, OSError, ValueError ) as e:
            t0( 'failed to update archive config file:', e )
            if not self._overrideArchiveConfigFile( destName,
                                                    destPath,
                                                    not self._config.shutdown ):
               return

         archive = self.currentArchive()
         if not archive:
            t0( 'unable to use configured archive' )
            return

         self.enqueueAction( archive, self.doDestSetup )

   def close( self ):
      """Handle management archive close."""

      Tac.Notifiee.close( self )

class LogArchiver( SuperServer.SuperServerAgent ):
   def __init__( self, entityManager ):
      SuperServer.SuperServerAgent.__init__( self, entityManager )
      mg = entityManager.mountGroup()

      # SSD File Archive config and status object
      archiveConfig = mg.mount( 'mgmt/archive/config', 'Mgmt::Archive::Config', 'r' )

      self.archiveConfigReactors_ = None

      def _finish():

         # If we're the Alternate Sup on an SSO we can't update config
         if self.redundancyProtocol() == 'sso' and not self.active():
            t0( 'LogMgrArchive: SSO-Standby no op' )
            return

         # SSD File Archive Reactors
         self.archiveConfigReactors_ = LogArchiverConfigReactor( archiveConfig )

      mg.close( _finish )

   def warm( self ):
      return True

def Plugin( ctx ):
   ctx.registerService( LogArchiver( ctx.entityManager ) )
