# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
import re
import weakref
import Tac
import SuperServer
import Tracing

t0  = Tracing.trace0
__defaultTraceHandle__ = Tracing.Handle( "FileSystemMount" )

_nfsBashCmdTimeout_ = 5 #seconds
_nfsRetryCmdInterval_ = 10 #seconds
_nfsGetStatusInterval_ = 60 #seconds

class NfsConfigReactor( Tac.Notifiee ):
   notifierTypeName = 'Mgmt::FileSystems::NfsConfig'

   def __init__( self, notifier, configReactor, nfsStatus ):
      t0( 'In NfsConfigReactor:: Init for key: %s' % notifier.localPath )

      self.nfsConfig = notifier
      self.nfsStatus = nfsStatus
      self.configReactor_ = weakref.proxy( configReactor )
      Tac.Notifiee.__init__( self, notifier )

      self.retryActivity_ = Tac.ClockNotifiee()
      self.retryActivity_.handler = self.reTryMount
      self.retryActivity_.timeMin = Tac.now() + Tac.endOfTime

      # this is needed for getting nfs to mount profiles from the saved
      # running config. When startup config is parsed, no agents are up
      # so we wont get any callbacks on handleEnabled. Therefore, when the
      # agent gets up eventually, we process the nfs profiles by calling
      # handleEnabled deliberately
      self.handleEnabled()

   # called through handleCollectionChange, when the key is deleted from it
   def close( self ):
      t0( 'NfsConfigReactor:: Close -key: %s' % self.nfsStatus.localPath )
      self.unMountConfig()
      self.configReactor_.handleNfsConfigChange(self.notifier_.name)
      Tac.Notifiee.close( self )

   def handleEnabled( self ):
      t0( 'Handling nfs config enable to %s for key: %s' % \
            ( self.nfsStatus.enabled, self.nfsConfig.localPath ) )
      if self.nfsStatus.enabled == True:
         self.reMount()
   
   @Tac.handler( 'genId' )
   def handleGenId( self ):
      t0( 'Handling nfs config change for key: %s' % self.nfsConfig.localPath )
      if self.nfsStatus.enabled:
         self.reMount()
      else:
         self.nfsStatus.enabled = True
         self.handleEnabled()

   def reMount( self ):
      if self.nfsStatus.enabled == True:
         t0( 'remounting nfs config for key: %s' % ( self.nfsConfig.localPath ) )

         if self.nfsStatus.active == True:
            self.unMountConfig()
         self.mounConfig()

   def mounConfig( self ):
      t0( 'NfsConfigReactor: Mounting config for key: %s' % \
                                          self.nfsConfig.localPath )
      
      self.stopRetryTimer()
      # if remote address is not configured
      if not self.nfsConfig.remoteAddr or not self.nfsConfig.remotePath:
         t0( 'Remote address and remote path not configured for mount at: %s' % \
               ( self.nfsConfig.localPath ) )
         return      

      try:
         # create the directory if does not exist
         cmdRan = Tac.run( [ "mkdir", "-p", "%s" % self.nfsConfig.localPath ],
                  stdout=Tac.CAPTURE, asRoot=True, asUser=None, env=None )
         t0( 'mkdir command ran: ' + cmdRan )

         # soft option is necessary, if hard option is used then mount 
         # infinitely tries with nfs server, and if nfs server is shut
         # then leads to blocking. We ignore the return code because of 
         # http://reviewboard/r/70749/
         cmdRan = Tac.run(
                     [ "mount",
                        "-t", "nfs",
                        "-o", "nolock,%s,timeo=%d,soft,retrans=%d" % (
                                       self.nfsConfig.access,
                                       self.nfsConfig.timeout,
                                       self.nfsConfig.retransmit ),
                        "%s:%s" % ( self.nfsConfig.remoteAddr,
                                    self.nfsConfig.remotePath ),
                        self.nfsConfig.localPath ],
                     stdout=Tac.CAPTURE, asRoot=True, ignoreReturnCode=True,
                     timeout=_nfsBashCmdTimeout_ )
         t0( 'Mount command ran : %s' % cmdRan )

         cmdRan = Tac.run( [ "grep", "-qs", "%s" % self.nfsConfig.localPath,
                             "/proc/mounts" ], stdout=Tac.CAPTURE, asRoot=True )
         t0( 'check mount command ran : %s' % cmdRan )

         self.mountSuccess()
      except Tac.SystemCommandError as e:
         t0( "Failed to mount %s:%s with error: %s" % ( 
                  self.nfsConfig.remoteAddr, self.nfsConfig.remotePath, str( e ) ) )
         self.mountFailure()
      except Tac.Timeout as t:
         t0( "Failed to mount %s:%s with error: %s" % ( 
                  self.nfsConfig.remoteAddr, self.nfsConfig.remotePath, str( t ) ) )
         self.mountFailure()

   def reTryMount( self ):
      t0( 'ReTrying to mount config at %s' % self.nfsConfig.localPath )

      self.mounConfig()

   def mountSuccess( self ):
      t0( 'Handling Mount Success for key: %s' % self.nfsConfig.localPath )
      self.nfsStatus.active = True
      self.stopRetryTimer()

   def mountFailure( self ):
      # retry after some time
      t0( 'Handling Mount Failure for key: %s' % self.nfsConfig.localPath )

      self.startRetryTimer()
      self.nfsStatus.active = False

   def unMountSuccess( self ):
      t0( 'Handling UnMount Success for key: %s' % self.nfsConfig.localPath )
      self.nfsStatus.active = False
      self.nfsStatus.enabled = False
      self.handleEnabled()

   def unMountFailure( self ):
      # TODO: retry after some time
      t0( 'Handling UnMount Failure for key: %s' % self.nfsConfig.localPath )
      self.nfsStatus.enabled = False
      self.handleEnabled()

   def unMountConfig( self ):
      t0( 'handle Unmount config for key: %s' % self.nfsConfig.localPath )

      if self.nfsStatus and self.nfsStatus.active == False:
         t0( '%s not mounted' % self.nfsStatus.localPath )
         return

      try:
         t0( 'Unmount config, key : %s' % self.nfsStatus.localPath )
         cmdRan = Tac.run( [ "umount", "-l", "%s" % self.nfsStatus.localPath ], 
                  asRoot=True, stdout=Tac.CAPTURE, timeout=_nfsBashCmdTimeout_ )
         t0( 'Unmount command ran : %s' % cmdRan )
         self.unMountSuccess()
      except (Tac.SystemCommandError) as e:
         t0( "Failed to unmount %s with error: %s" % ( 
                  self.nfsStatus.localPath, e.output ) )
         self.unMountFailure()
      except (Tac.Timeout) as t:
         t0( "Failed to unmount %s with error: %s" % ( 
                        self.nfsStatus.localPath, t.output() ) )
         self.unMountFailure()

   def stopRetryTimer( self ):
      self.retryActivity_.timeMin = Tac.now() + Tac.endOfTime

   def startRetryTimer( self ):
      timeout = Tac.now() + _nfsRetryCmdInterval_
      self.retryActivity_.timeMin = timeout

class ConfigReactor( Tac.Notifiee ):
   notifierTypeName = 'Mgmt::FileSystems::Config'

   def __init__( self, notifier, status ):
      t0( 'ConfigReactor:: init called' )

      self.nfsConfigReactor_ = {}
      self.status = status
      self.config = notifier

      self.statusActivity_ = Tac.ClockNotifiee()
      self.statusActivity_.handler = self.getStatus
      self.statusActivity_.timeMin = Tac.now() + _nfsGetStatusInterval_

      Tac.Notifiee.__init__( self, notifier )

      for key in self.notifier_.nfsConfig.keys():
         self.handleNfsConfig( key )

   @Tac.handler( 'nfsConfig' )
   def handleNfsConfig( self, key ):
      t0( 'ConfigReactor:: handleNfsConfig for key: %s' % key )

      fsStatus = self.status
      nfsConfig = self.notifier_.nfsConfig.get( key )
      nfsStatus = None

      # adds the config, deletion of config is taken care by 
      # 'handleNfsConfigChange func
      if nfsConfig:
         t0( 'ConfigReactor:: handleNfsConfig creating status object for\
                                                         key: %s' % key )
         nfsStatus = fsStatus.newNfsStatus( key )

      t0( 'handleNfsConfig key:%s' % key )

      Tac.handleCollectionChange( NfsConfigReactor, key,
                                 self.nfsConfigReactor_,
                                 self.notifier_.nfsConfig,
                                 reactorArgs=(self, nfsStatus) )

   # handles the deletion of the config
   def handleNfsConfigChange( self, key ):
      t0( 'ConfigReactor:: handleNfsConfigChange for key: %s' % key )

      # delete nfsstatus
      nfsStatus = self.status.nfsStatus.get( key )
      if nfsStatus:
         del self.status.nfsStatus[ key ]
         t0( 'Deleted nfsStatus from collection - %s' % key )

   # gets the nfs host/server status to know whether nfs server
   # is up or not, this is needed as 'mount' command does not
   # provide the status of each mount profile, esp when server is down
   def isNfsHostActive( self, nfsConfig ):
      nfsHostName = nfsConfig.remoteAddr
      t0( 'Checking NFS host status for : %s' % nfsHostName )
      
      try:
         # showmount shows the directory that are mounted, but we just
         # parse it to know whether any error was given or not.
         output = Tac.run( [ "showmount", "-d", "%s" % nfsHostName ], 
               stdout=Tac.CAPTURE, asRoot=True, asUser=None, env=None, 
               timeout=_nfsBashCmdTimeout_ )

         matchStr = '%s\s+on\s+%s' % ( 'Directories', nfsHostName )

         m = re.search( matchStr, output )
         if m:
            return True

      except (Tac.SystemCommandError) as e:
         t0( "SystemCommandError error: %s" %( str( e ) ) )
      except (Tac.Timeout) as t:
         t0( "Failed to execute command: %s" % str( t ) )

      return False

   # gets the status of the mounts that were active
   def getStatus( self ):
      t0( 'Checking status' )
      # run 'mount' command and parse it
      try:
         output = Tac.run( [ "mount" ], stdout=Tac.CAPTURE, asRoot=True, 
                     asUser=None, env=None, timeout=_nfsBashCmdTimeout_ )

         #parse in lines
         mountLines = []
         mountLines = output.split('\n')

         for localPath in self.config.nfsConfig.keys():
            nfsConfig = self.config.nfsConfig[ localPath ]
            nfsStatus = self.status.nfsStatus[ localPath ]
         
            if nfsStatus.active == False:
               continue

            remoteAddr = nfsConfig.remoteAddr
            remotePath = nfsConfig.remotePath

            # no remote address or path configured
            if not remoteAddr or not remotePath:
               continue

            if localPath[ len(localPath) - 1 ] == '/':
               localPath = localPath[ 0 : len(localPath) - 1 ]

            bFoundMount = False
            for line in mountLines:
               t0( "line: %s" % line )
               if re.match( remoteAddr + ':' + remotePath + '\s*.on\s*.' + 
                                                            localPath, line ):
                  bFoundMount = True
                  break

            bNfsHostActive = False
            if bFoundMount:
               t0( "%s:%s actively mounted over %s" % 
                                          (remoteAddr, remotePath, localPath) )
               # check nfs server status
               bNfsHostActive = self.isNfsHostActive( nfsConfig )
            else:
               t0( "%s:%s not mounted over %s" % 
                                          (remoteAddr, remotePath, localPath) )

            # if both 'nfs server' and mount shows active, then we cont
            if bFoundMount and bNfsHostActive:
               continue
            
            # else mount profile is not active
            t0( "(%s:%s)- server status:%s, mount status: %s" % \
               (remoteAddr, remotePath, str( bNfsHostActive ), str( bFoundMount ) ) )
            # remount the directory
            nfsStatus.enabled = True
            self.nfsConfigReactor_[ localPath ].reMount()

      except (Tac.SystemCommandError) as e:
         t0( "SystemCommandError error: %s" %( str( e ) ) )
      except (Tac.Timeout) as t:
         t0( "Failed to execute command 'Mount': %s" % str( t ) )
      
      self.statusActivity_.timeMin = Tac.now() + _nfsGetStatusInterval_

class FileSystemMounter( SuperServer.SuperServerAgent ):
   def __init__(self, entityManager):
      t0( 'FileSystemMounter:: Init' )
      SuperServer.SuperServerAgent.__init__( self, entityManager )
      mg = entityManager.mountGroup()
      self.config = mg.mount( "mgmt/filesystems/config", 
                              "Mgmt::FileSystems::Config", "r" )
      self.status = mg.mount( "mgmt/filesystems/status", 
                              "Mgmt::FileSystems::Status", "w" )

      self.configReactor_ = None

      def _finish():
         if self.redundancyProtocol() == 'sso' and not self.active():
            t0( 'FileSystemMount: SSO-Standby no op' )
            return
         
         self.configReactor_ = ConfigReactor( self.config, self.status )
      mg.close( _finish )

   def onSwitchover( self, protocol ):
      t0( 'FileSystemMount: onSwitchover() ' + protocol )
      if not self.configReactor_:
         t0( 'Creating ConfigReactor' )
         self.configReactor_ = ConfigReactor( self.config, self.status )
          
def Plugin( ctx ):
   t0( 'Mount Plugin registered' )
   ctx.registerService( FileSystemMounter( ctx.entityManager ) )
