#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pkgdeps: rpm kea-dhcp
# pkgdeps: rpm kea-dhcp-hooks

from __future__ import absolute_import, division, print_function
import SuperServer
import Tac
import DhcpServerReactor
import json
import QuickTrace
import os
import socket
import errno
import ArPyUtils
from DeviceNameLib import eosIntfToKernelIntf
from EosDhcpServerLib import DhcpIntfStatusMessages as DISM
from EosDhcpServerLib import Dhcp6DirectRelayMessages as D6DRM
from EosDhcpServerLib import overrideLockfileDir
from EosDhcpServerLib import tmpKeaConfFile
from EosDhcpServerLib import tmpToCheckKeaConfFile
from EosDhcpServerLib import keaLeasePath
from EosDhcpServerLib import keaConfigPath
from EosDhcpServerLib import keaControlConfigPath
from EosDhcpServerLib import keaServiceName
from EosDhcpServerLib import keaPidPath
from EosDhcpServerLib import keaControlSock
from EosDhcpServerLib import runKeaDhcpCmd
from EosDhcpServerLib import configReloadCmdData
from EosDhcpServerLib import KeaDhcpCommandError
from EosDhcpServerLib import featureOption43Status
from EosDhcpServerLib import defaultVendorId
from EosDhcpServerLib import getSubOptionData
from EosDhcpServerLib import vendorSubOptionType
from EosDhcpServerLib import featureStaticIPv4AddressStatus
from eunuchs.if_h import IFF_RUNNING
import ManagedSubprocess
import re
import glob
from IpLibConsts import DEFAULT_VRF
import SharedMem
import Smash
import Tracing
import weakref
import Arnet
from collections import defaultdict
import Toggles.DhcpServerToggleLib

__defaultTraceHandle__ = Tracing.Handle( "DhcpServer" )
t0 = __defaultTraceHandle__.trace0
t1 = __defaultTraceHandle__.trace1
t2 = __defaultTraceHandle__.trace2
t3 = __defaultTraceHandle__.trace3
t4 = __defaultTraceHandle__.trace4
t5 = __defaultTraceHandle__.trace5

qv = QuickTrace.Var
qt0 = QuickTrace.trace0

addrFamilyEnum = Tac.Type( "Arnet::AddressFamily" )

maxCheckConfigInternalRecursionLevel = 5

def configTest():
   return "DHCPSERVER_CONFIG_BREADTH_TEST" in os.environ

def getSubnetFailureReason( reason, isPreExistingFailure=False ):
   '''
   We have several different subnet error messages that can be put into 2 categories:
   - config related errors (starts with 'subnet configuration failed'):
   we will use `configPattern` regex
   - other errors (e.g. trying to add the same subnet with different subnet-id):
   we will use `alreadyExistPattern`

   An example error message is:
   Error encountered: subnet configuration failed: a pool of type V4, with the
   following address range: 192.168.1.110-192.168.1.254 overlaps with an existing
   pool in the subnet: 192.168.1.0/24 to which it is being added.

   The match regex would be:
   match.group('pool') : 192.168.1.110-192.168.1.254
   match.group('overlap') : overlaps with an existing pool
   match.group('nomatch') : None
   match.group('configPattern') : overlaps with an existing pool
   match.group('alreadyExistPattern') : None
   '''
   heading = '(?P<heading>address range: )'
   poolPattern = r'(?P<pool>[0-9a-f:\-\.]+) '
   overlapPattern = '(?P<overlap>overlaps with an existing pool)'
   notMatchPattern = '(?P<notmatch>does not match)'
   configPattern = '(?P<configPattern>({}|{}))'.format(
      overlapPattern, notMatchPattern )
   preExistingSubnetPattern = '(?P<preExisting> already exists)'

   pattern = heading + poolPattern + configPattern
   pattern += ( '|' + preExistingSubnetPattern ) if isPreExistingFailure else ''

   match = re.search( pattern, reason )
   if not match:
      return 'Unknown failure'

   if match.group( 'configPattern' ):
      return 'range {} {}'.format( match.group( 'pool' ),
                                    match.group( 'configPattern' ) )
   return match.group( 'preExisting' )

# reservations mac-address
def reservationsMacAddrFailureReason( reason ):
   '''
   Example Error Message:
   Error encountered: failed to add new host using the HW address
   'd6:b4:20:d8:78:94 and DUID '(null)' to the IPv4 subnet id '123' for the address
   1.0.0.12: There's already a reservation for this address
   '''
   # match "1.0.0.12"
   ipv4AddrPattern = r'(?P<ipv4Addr>\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3})'
   errorPattern = 'for the address {}'.format( ipv4AddrPattern )

   errorRe = re.compile( errorPattern )
   match = errorRe.search( reason )
   if not match:
      t0( 'Unknown failure for mac-address reservations' )
      return 'Unknown failure'

   reason = "ipv4-address {} is reserved more than once"
   return reason.format( match.group( 'ipv4Addr' ) )

def configErrorSubnetIdRe():
   # Example: "to the IPv4 subnet id '123'"
   # match "'123'"
   lookBehind = "(?<=subnet id )"
   subnetIdPattern = r'{}(?P<subnetId>\'\d+\')'.format( lookBehind )
   return re.compile( subnetIdPattern )

def configErrorSubnetRe():
   lookahead = '(?= to which it is being added| already exists)'
   reason = '(?P<subnet>[0-9a-f.:/]+){}'.format( lookahead )
   return re.compile( reason )

def configErrorRe():
   errorHeading = '(?P<heading>Error encountered: )'

   subnetConfigFailure = '(?P<subnetConfigFailure>subnet configuration failed)'
   subnetExistingFailure = '(?P<subnetExistingFailure>subnet with the prefix of)'
   serverConfigFailure = '(?P<serverConfigFailure>[a-z A-Z]+)'
   # reservations mac-address
   reservationsConfigFailure = '(?P<reservationsConfigFailure>failed to add new'\
                                ' host using the HW address)'
   fails = [ subnetConfigFailure, subnetExistingFailure, reservationsConfigFailure,
             serverConfigFailure ]
   failRegex = '(?P<failRegex>{}|{}|{}|{}):?\n*'.format( *fails )

   reasonRegex = '(?P<reason> ?.*)'

   errorRegex = errorHeading + failRegex + reasonRegex

   return re.compile( errorRegex )

def subOptionTypeToKeaType( type_ ):
   if type_ == vendorSubOptionType.string:
      return 'string'
   elif type_ == vendorSubOptionType.ipAddress:
      return 'ipv4-address'
   else:
      assert False, "Unknown sub-option type"
      return None

class DhcpServerVrfService( SuperServer.LinuxService ):
   notifierTypeName = '*'

   def __init__( self, config, status, vrf, ipVersion ):
      self.ipVersion_ = ipVersion
      self.vrf_ = vrf
      self.config_ = config
      self.status_ = status
      # intfCfg_ contains the config for kea
      self.intfCfg_ = set()
      # activeInterfaces is a list of tuples ( intfName, ip addresses )
      self.activeInterfaces_ = []
      self.serviceName_ = keaServiceName.format( version=ipVersion,
                                                 vrf=vrf.nsName() )
      self.leasePath_ = keaLeasePath.format( version=ipVersion, vrf=vrf.nsName() )
      self.configPath_ = keaConfigPath.format( version=ipVersion, vrf=vrf.nsName() )
      self.configCtrlPath_ = keaControlConfigPath.format( version=ipVersion,
                                                          vrf=vrf.nsName() )
      self.pidPath_ = keaPidPath.format( version=ipVersion, vrf=vrf.nsName() )
      self.controlSock_ = keaControlSock.format( version=ipVersion,
                                                 vrf=vrf.nsName() )
      # Tmp file, used to save "ideal" config from the user
      self.tmpKeaConfFile_ = tmpKeaConfFile.format( version=ipVersion,
                                                    vrf=vrf.nsName() )
      # Tmp file, used to check malformed configs if any
      self.tmpToCheckKeaConfFile_ = tmpToCheckKeaConfFile.format( version=ipVersion,
                                                                  vrf=vrf.nsName() )
      SuperServer.LinuxService.__init__( self, self.serviceName_, 'keactrl',
                                         self.config_, self.configPath_,
                                         configFileHeaderEnabled=False )
      self.writeConfigFile( self.configCtrlPath_, self.controlConfConfig(),
                            updateInPlace=True )

   def controlConfConfig( self ):
      config = ''
      config += '# Location of Kea configuration files.\n'
      config += 'kea_dhcp{ver}_config_file={path}\n'.format(
         ver=self.ipVersion_, path=self.configPath_ )

      config += '# Location of Kea binaries.\n'
      config += 'dhcp4_srv=/usr/sbin/kea-dhcp4\n'
      config += 'dhcp6_srv=/usr/sbin/kea-dhcp6\n'
      config += 'dhcp_ddns_srv=/usr/sbin/kea-dhcp-ddns\n'
      config += 'ctrl_agent_srv=/usr/sbin/kea-ctrl-agent\n'
      config += 'kea_netconf_config_file=/usr/sbin/kea-netconf\n'
      config += 'kea_ctrl_agent_config_file=/etc/kea/kea-ctrl-agent.conf\n'
      config += 'kea_dhcp_ddns_config_file=/etc/kea/kea-dhcp-ddns.conf\n'

      if self.ipVersion_ == "4":
         config += 'dhcp4=yes\n'
         config += 'dhcp6=no\n'
         config += 'kea_dhcp6_config_file=/etc/kea/kea-dhcp6.conf\n'
      else:
         assert self.ipVersion_ == "6"
         config += 'dhcp4=no\n'
         config += 'dhcp6=yes\n'
         config += 'kea_dhcp4_config_file=/etc/kea/kea-dhcp4.conf\n'

      config += 'dhcp_ddns=no\n'
      config += 'ctrl_agent=no\n'
      config += 'netconf=no\n'
      config += 'kea_verbose=no\n'
      return config

   def serviceCmd( self, cmd ):
      return [ 'keactrl', cmd, '-s', 'dhcp{}'.format( self.ipVersion_ ),
               '-c', self.configCtrlPath_ ]

   def checkServiceCmdOutput( self, cmd, output ):
      statusStr = "DHCPv{} server: active".format( self.ipVersion_ )
      # The calling code assumes Tac.SystemCommandError will be thrown if the process
      # isn't started.
      if output and not any( statusStr in line for line in output ):
         raise Tac.SystemCommandError( output )
      return True

   def _runKeaCtrlCmd( self, cmd ):
      traceStatement = 'Kea cmd:'
      qt0( traceStatement, qv( cmd ), qv( self.ipVersion_ ) )
      traceStatement += ' {} {}'.format( cmd, self.ipVersion_ )
      t3( traceStatement )
      timeout = self.serviceCmdTimeout( cmd )
      cmd = self.serviceCmd( cmd )
      timeout = self.serviceCmdTimeout( ' '.join( cmd ) )
      # Its a real shame that we are using the env variable to set this dir. I tried
      # changing the local status dir in the build to be /var, but that failed
      # miserably because of how kea is structured.
      env = { "KEA_LOCKFILE_DIR": overrideLockfileDir }
      try:
         os.makedirs( overrideLockfileDir )
      except OSError as e:
         if e.errno != errno.EEXIST:
            raise

      env.update( os.environ )
      oldmask = os.umask( 0 )
      Tac.run( cmd, stdout=Tac.DISCARD, stderr=Tac.DISCARD, timeout=timeout,
               asRoot=True, env=env )
      oldmask = os.umask( oldmask )

   def getPid( self ):
      try:
         with open( self.pidPath_ ) as f:
            keaPid = int( f.read() )
            qt0( "kea PID:", qv( keaPid ), qv( self.ipVersion_ ) )
            return keaPid
      except IOError as e:
         if e.errno != errno.ENOENT:
            raise e
      except ValueError:
         pass
      qt0( "no current kea PID", qv( self.ipVersion_ ) )
      return -1

   def started( self ):
      pid = self.getPid()
      if pid <= 0:
         return False
      # check if pid is running
      try:
         os.kill( pid, 0 )
      except OSError as e:
         if e.errno == errno.ESRCH:
            return False
         elif e.errno == errno.EPERM:
            return True
         else:
            raise
      else:
         return True

   def conf( self ):
      '''Return the content to write to DHCP server service's kea-dhcp4.conf file'''
      t0( 'Creating kea-dhcpX.conf file content' )
      out = {}

      # Authorative
      authoritative = self._authoritative()
      if authoritative:
         out[ 'authoritative' ] = authoritative

      # Lease file
      out[ 'lease-database' ] = {
         'type': 'memfile',
         'name': self.leasePath_
         }

      # Hooks for lease commands
      out[ 'hooks-libraries' ] = [ {
         'library': '/usr/lib{}/kea-dhcp/hooks/libdhcp_lease_cmds.so'.format(
            '64' if ArPyUtils.arch() != 32 else "" ) } ]

      out[ 'control-socket' ] = {
         'socket-type': 'unix',
         'socket-name': self.controlSock_ }

      out[ 'sanity-checks' ] = {
         'lease-checks': 'fix-del'
         }

      # Custom option definition for subnet name
      # Kea DHCP does not have an option for subnet name
      # thus we will have to create our own option data with
      # name "name". 222 is just a random unused number for type code.
      out[ 'option-def' ] = [ {
         'name': 'name',
         'code': 222,
         'type': 'string' } ]

      optionsDef = self._optionsDef()
      out[ 'option-def' ].extend( optionsDef )

      out[ 'option-data' ] = []
      if featureOption43Status():
         # add client classes
         clientClasses, defaultOptionsData = self._clientClasses( self.config_ )
         if clientClasses:
            out[ 'client-classes' ] = clientClasses

         # add default vendorId option-data
         if defaultOptionsData:
            out[ 'option-data' ].extend( defaultOptionsData )

      if not self.serviceEnabled():
         return json.dumps( out )

      lifetime = self._leaseTime()
      if lifetime:
         out[ 'valid-lifetime' ] = int( lifetime )
         if self.ipVersion_ == '6':
            # IPv6 DHCP requires preferred-lifetime to be no less than valid-lifetime
            out[ 'preferred-lifetime' ] = int( lifetime )

      intfConfig = list( self.intfCfg_ )
      if intfConfig:
         out[ 'interfaces-config' ] = { 'interfaces': intfConfig }

      subnetConfig, hostReservationIDs = self._subnetConfig()
      if subnetConfig:
         out[ 'subnet{}'.format( self.ipVersion_ ) ] = subnetConfig

      if featureStaticIPv4AddressStatus() and hostReservationIDs:
         out[ 'host-reservation-identifiers' ] = hostReservationIDs

      optionsData = self._optionsData( self.config_ )
      if optionsData:
         out[ 'option-data' ].extend( optionsData )

      ddnsData = self._ddnsData()
      if ddnsData:
         out[ 'dhcp-ddns' ] = ddnsData

      allConfig = { 'Dhcp{}'.format( self.ipVersion_ ): out }

      loggingConfig = self._loggingConfig()
      if loggingConfig:
         allConfig[ 'Logging' ] = loggingConfig

      # need to clear disabled messages/reasons since it will be updated by
      # checkConfig
      self._resetStaleDisabledMessages()
      self.checkConfig( allConfig )
      return json.dumps( allConfig, indent=4, sort_keys=True )

   def checkConfig( self, allConfig, level=0 ):
      # since tmpToCheckKeaConfFile will keep removing invalid subnets recursively
      # until we reach a "valid" config. at the end it will
      # be the same as the KeaConfFile, therefore save a copy beforehand
      # for troubleshooting (since this should match the dhcpserver::config)
      t0( 'Current checkConfig level:', level )
      if level == 0:
         with open( self.tmpKeaConfFile_, "w" ) as tmpFile:
            tmpFile.write( json.dumps( allConfig, indent=4 ) )

      # save working config
      with open( self.tmpToCheckKeaConfFile_, "w" ) as tmpFile:
         tmpFile.write( json.dumps( allConfig, indent=4 ) )

      if level > maxCheckConfigInternalRecursionLevel:
         t0( 'Avoiding infinite recursion. Reached maximum recursion of:', level )
         os.remove( self.tmpToCheckKeaConfFile_ )
         # removing invalid config
         allConfig.pop( 'Dhcp{}'.format( self.ipVersion_ ) )
         if self.ipVersion_ == "4":
            self.status_.ipv4ServerDisabled = 'Server is disabled'
         else:
            self.status_.ipv6ServerDisabled = 'Server is disabled'

      else:
         cmd = 'kea-dhcp{} -t {}'.format(
            self.ipVersion_, self.tmpToCheckKeaConfFile_ ).split( ' ' )
         process = ManagedSubprocess.Popen( cmd, stdout=ManagedSubprocess.DISCARD,
                                            stderr=ManagedSubprocess.PIPE )

         self._checkConfigInternal( allConfig, process, level=level )

   def startService( self ):
      if configTest():
         return

      qt0( 'start service' )
      if not self.serviceEnabled():
         self._runKeaCtrlCmd( "stop" )
         return

      if self.started():
         # startService is called from _maybeRestartService but there's
         # no change in the config file. If there's already a process running
         # we just reload the config.
         qt0( 'reload config' )
         cmdData = configReloadCmdData()
         try:
            runKeaDhcpCmd( self.ipVersion_, self.vrf_.nsName(), cmdData )
            return
         except KeaDhcpCommandError:
            # Happens after reboot when some interfaces are not up/present yet
            # or when socket is not present.
            qt0( 'Kea command error running:', qv( cmdData ), qv( self.ipVersion_ ) )
            return
         except socket.error as e:
            qt0( 'socket error:', qv( os.strerror( e.errno ) ) )
         # call sync() to reschedule _maybeRestartService()
         t0( 'Start Sync' )
         self.sync()
      else:
         self._runKeaCtrlCmd( "start" )

   def stopService( self ):
      if configTest():
         return
      pid = self.getPid()
      if pid <= 0:
         return
      qt0( 'stop service kea PID:', qv( pid ), qv( self.ipVersion_ ) )
      self._runKeaCtrlCmd( "stop" )
      timeout = self.serviceCmdTimeout( "stop" )

      # wait for pid to be deleted
      try:
         Tac.waitFor( lambda: not self.started(), description='kea process to stop',
                      timeout=timeout,
                      sleep=True )
      except ( Tac.SystemCommandError, Tac.Timeout ):
         qt0( 'Error/Timeout when trying to stop kea service' )
         return

      # delete unix domain socket
      try:
         os.unlink( self.controlSock_ )
      except OSError:
         qt0( 'failed to unlink socket file' )

   def restartService( self ):
      qt0( 'restart service' )

      # Start the service if no kea process is running
      if not self.started():
         self.startService()
         return

      cmdData = configReloadCmdData()
      try:
         qt0( 'reload config' )
         runKeaDhcpCmd( self.ipVersion_, self.vrf_.nsName(), cmdData )
         return
      except KeaDhcpCommandError:
         # Happens during ConfigReplace when some interfaces are not
         # up/present/ready yet.
         # Also can happen when trying to do a reload and some interface flaps.
         qt0( 'Kea command error running:', qv( cmdData ), qv( self.ipVersion_ ) )
         return
      except socket.error as e:
         qt0( 'socket error:', qv( os.strerror( e.errno ) ) )
      # call sync() to reschedule _maybeRestartService()
      t0( 'Restart Sync' )
      self.sync()

   def verboseLoggingPath( self ):
      if "DHCPSERVER_VERBOSE_LOGGING" in os.environ:
         return "/tmp/kea-{}-debug.log".format( self.vrf_.nsName() )
      if self.config_.debugLogPath:
         return self.config_.debugLogPath
      return None

   def _loggingConfig( self ):
      logPath = self.verboseLoggingPath()
      if logPath:
         return {
            "loggers": [
               { "name": "kea-dhcp{}".format( self.ipVersion_ ),
                 "severity": "DEBUG",
                 "debuglevel": 99,
                 "output_options": [
                    {
                       "output": logPath
                    }
                 ],
               }
            ]
         }
      return {
         "loggers": [
            { "name": "kea-dhcp{}".format( self.ipVersion_ ),
              "severity": "WARN",
              "output_options": [
                 {
                    "output": "syslog",
                 }
              ]
            }
         ]
      }

   def _leaseTime( self ):
      raise NotImplementedError()

   def _subnetConfig( self ):
      raise NotImplementedError()

   def _vendorOptionsDef( self, config ):
      return []

   def _clientClasses( self, config ):
      return [], []

   def _optionsDef( self ):
      return []

   def _optionsData( self, config, isSubnetConfig=False ):
      raise NotImplementedError()

   def _resetStaleDisabledMessages( self ):
      raise NotImplementedError()

   def _checkOverlappingSubnet( self, out ):
      raise NotImplementedError()

   def _checkConfigInternal( self, allConfig, process, level=0 ):
      raise NotImplementedError()

   def _removeSubnet( self, subnets, subnetPrefix ):
      raise NotImplementedError()

   def _containsHostReservations( self, subnets ):
      return False

   def configuredInterfaces( self ):
      raise NotImplementedError()

   def handleDisable( self, disabled ):
      raise NotImplementedError()

   def handleDebugLog( self ):
      self.sync()

   def _authoritative( self ):
      raise NotImplementedError()

   def _ddnsData( self ):
      raise NotImplementedError()

   def dhcpInterfaceStatus( self ):
      raise NotImplementedError()

   def genKeaIntfCfgCmds( self, linuxIntfName, ipAddrs ):
      raise NotImplementedError()

   def getOrCreateSubnetOverlapped( self, subnetId ):
      raise NotImplementedError()

   def subnetGenPrefix( self, subnetPrefix ):
      """
      Cast subnet prefix to IpGenPrefix

      @Input
         subnetPrefix (Arnet::Prefix) or (Arnet::Ip6Prefix)

      @Returns
         genPrefix (Arnet::IpGenPrefix)
      """

      genPrefix = Arnet.IpGenPrefix( str( subnetPrefix ) )
      return genPrefix

   def handleActiveIntfs( self, intfCfg, activeIntfs ):
      t1( "IPv%s: Old active: %s  --  New active: %s" %
          ( self.ipVersion_, self.intfCfg_, intfCfg ) )
      if intfCfg == self.intfCfg_ and activeIntfs == self.activeInterfaces_:
         return
      self.intfCfg_ = intfCfg
      self.activeInterfaces_ = activeIntfs

      # If no more active interfaces, reset the broadcast status
      if not self.intfCfg_ and self.ipVersion_ == "6":
         for subnetId in self.status_.ipv6SubnetToId:
            self.status_.subnetBroadcastStatus[ subnetId ] = (
               D6DRM.NO_IP6_ADDRESS_MATCH )

      self.sync()

   def serviceEnabled( self ):
      raise NotImplementedError()

class DhcpServerIpv4Service( DhcpServerVrfService ):
   notifierTypeName = 'DhcpServer::Config'

   def __init__( self, config, status, vrf ):
      DhcpServerVrfService.__init__( self, config, status, vrf, "4" )

      self.status_.ipv4SubnetsDisabled.clear()
      self.status_.ipv4IdToSubnet.clear()
      self.status_.ipv4SubnetToId.clear()
      self.status_.ipv4SubnetLastUsedId = 0
      self.status_.ipv4SpaceIdToVendorId.clear()
      self.status_.ipv4VendorIdToSpaceId.clear()
      self.status_.ipv4VendorIdSpaceLastUsedId = 0
      self.status_.lastClearIdIpv4 = 0
      self.status_.subnetOverlapped.clear()
      self.status_.interfaceIpv4Disabled.clear()

      self.configReactor_ = DhcpServerReactor.ServiceV4Reactor( self.config_, self )

   ##############################
   # Implemeting Derived Methods
   ##############################
   def serviceProcessWarm( self ):
      ''' Check if the DHCP server service has fully restsarted '''
      return bool( self.config_.interfacesIpv4 )

   def _authoritative( self ):
      return True

   # Options Def
   def _optionsDef( self ):
      """
      IPv4 Options Def

      @Returns
         optionsDef (list of dicts) : options-def to be added in the kea config json
      """
      optionsDef = []

      # Custom option definition for DHCP code 150 (TFTP server addresses).
      # Kea DHCP does not support code 150 because it has multiple definitions
      # https://www.iana.org/assignments/bootp-dhcp-parameters/
      #       bootp-dhcp-parameters.xhtml#options
      # We will have to define our own option 150 in compliance with RFC5859.
      optionsDef.append( {
         'name': 'tftp-server-address',
         'code': 150,
         'type': 'ipv4-address',
         'array': True } )

      featureOption43Enabled = Toggles.DhcpServerToggleLib.\
                                       toggleFeatureOption43Enabled()
      if featureOption43Enabled:
         # example to verify if the feature toggle is properly enabled/disabled
         optionsDef.append( {
            'name': 'testingFeatureToggleOption43',
            'code': 1,
            'space': 'featureToggle',
            'type': 'ipv4-address',
            } )

      if featureStaticIPv4AddressStatus():
         # example to verify if the feature toggle is properly enabled/disabled
         optionsDef.append( {
            'name': 'testingFeatureToggleStaticIpv4AddressStatus',
            'code': 2,
            'space': 'featureToggle',
            'type': 'ipv4-address',
            } )

      if featureOption43Status():
         optionsDef.extend( self._vendorOptionsDef( self.config_ ) )

      return optionsDef

   # Options data
   def _optionsData( self, config, isSubnetConfig=False ):
      options = []
      # subnet config options
      if isSubnetConfig:
         servers = config.dnsServers
         if config.defaultGateway != '0.0.0.0':
            options.append( {
               'name': 'routers',
               'data': config.defaultGateway } )
         if config.subnetName:
            options.append( {
               'name': 'name',
               'data': config.subnetName } )
         if config.tftpServerOption66:
            options.append( {
               'name': 'tftp-server-name',
               'data': config.tftpServerOption66 } )
         if config.tftpServerOption150:
            options.append( {
               'name': 'tftp-server-address',
               'data': ','.join( config.tftpServerOption150.values() ) } )
         if config.tftpBootFileName:
            bootFileName = config.tftpBootFileName.replace( ",", r"\," )
            options.append( {
               'name': 'boot-file-name',
               'data': bootFileName } )
      # global config options
      else:
         servers = config.dnsServersIpv4
         if config.domainNameIpv4:
            options.append( { 'name': 'domain-name',
                              'data': config.domainNameIpv4 } )
         if config.tftpServerOption66Ipv4:
            # option 66
            options.append( { 'name': 'tftp-server-name',
                              'data': config.tftpServerOption66Ipv4 } )
         if config.tftpServerOption150Ipv4:
            options.append( { 'name': 'tftp-server-address',
                              'data': ','.join(
                                 config.tftpServerOption150Ipv4.values() ) } )
         if config.tftpBootFileNameIpv4:
            # option 67
            bootFileName = config.tftpBootFileNameIpv4.replace( ",", r"\," )
            options.append( { 'name': 'boot-file-name',
                              'data': bootFileName } )

      # common for both subnet and global config
      # if no DNS server is configured for subnet, kea will use the
      # global DNS server configuration.
      if servers:
         options.append( {
            'name': 'domain-name-servers',
            'data': ', '.join( servers.values() ) } )
      return options

   # Subnet
   def _subnetConfig( self ):
      output = []
      hostReservationIDs = set()
      for keaSubnetId, subnetId in self.status_.ipv4IdToSubnet.iteritems():
         subnetConfig = {}
         subnet = self.config_.subnetConfigIpv4[ subnetId ]
         subnetConfig[ 'subnet' ] = str( subnet.subnetId )
         subnetConfig[ 'id' ] = keaSubnetId

         # subnet range
         if subnet.ranges:
            pools = []
            for _range in subnet.ranges.keys():
               pools.append( { 'pool': '{}-{}'.format( _range.start, _range.end ) } )
            subnetConfig[ 'pools' ] = pools

         # valid-lifetime
         if subnet.leaseTime:
            subnetConfig[ 'valid-lifetime' ] = int( subnet.leaseTime )

         # options data
         options = self._optionsData( subnet, isSubnetConfig=True )
         if options:
            subnetConfig[ 'option-data' ] = options

         # host reservations mac address
         if featureStaticIPv4AddressStatus() and subnet.reservationsMacAddr:
            reservations = []
            for hostReservation in subnet.reservationsMacAddr.values():
               # a valid reservation must have at least 1 attribute configured
               ipAddr = hostReservation.ipAddr
               if ipAddr != '0.0.0.0':
                  hostReservationIDs.add( "hw-address" )
                  reservation = {
                        'hw-address': hostReservation.hostId,
                        'ip-address': ipAddr,
                        }
                  reservations.append( reservation )

            if reservations:
               subnetConfig[ 'reservations' ] = reservations

         output.append( subnetConfig )

      return output, list( hostReservationIDs )

   def _vendorOptionsDef( self, config ):
      optionsDef = []
      # add all subOption definitions
      for vendorId, spaceId in self.status_.ipv4VendorIdToSpaceId.iteritems():
         for subOption in config.vendorOptionIpv4[ vendorId ].\
                                 subOptionConfig.values():
            keaType = subOptionTypeToKeaType( subOption.type )
            optionsDef.append( {
               'name': 'subOption{}'.format( subOption.code ),
               'code': subOption.code,
               'type': keaType,
               'array': subOption.isArray,
               'space': '{}-space'.format( spaceId )
            } )

      # add default Option 43 definition if set
      if config.vendorOptionIpv4.get( defaultVendorId ):
         spaceId = self.status_.ipv4VendorIdToSpaceId[ defaultVendorId ]
         optionsDef.append( {
            'name': 'vendor-encapsulated-options',
            'type': 'empty',
            'code': 43,
            'encapsulate': '{}-space'.format( spaceId )
         } )

      return optionsDef

   def _clientClasses( self, config ):
      clientClasses = []
      defaultOptionsData = []
      for vendorId, spaceId in self.status_.ipv4VendorIdToSpaceId.iteritems():
         clientClass = {
            'name': 'VENDOR_CLASS_{}'.format( vendorId ),
            }

         # add all subOption options-data
         optionsData = []
         for subOption in config.vendorOptionIpv4[ vendorId ].\
                                 subOptionConfig.values():
            data = getSubOptionData( subOption )
            optionsData.append( {
               'name': 'subOption{}'.format( subOption.code ),
               'code': subOption.code,
               'space': '{}-space'.format( spaceId ),
               'data': data
               } )

         # add option43 data and definition
         optionsData.append( {
            'name': 'vendor-encapsulated-options'
            } )

         optionsDef = [ {
            'name': 'vendor-encapsulated-options',
            'type': 'empty',
            'code': 43,
            'encapsulate': '{}-space'.format( spaceId ),
            } ]

         if vendorId == defaultVendorId:
            # There is no client-class for default vendorId, instead it needs
            # to be in Dhcp4 option-data
            defaultOptionsData.extend( optionsData )
         else:
            clientClass[ 'option-data' ] = optionsData
            clientClass[ 'option-def' ] = optionsDef
            clientClasses.append( clientClass )

      return clientClasses, defaultOptionsData

   def _resetStaleDisabledMessages( self ):
      """ clear disabled messages/reasons """
      self.status_.ipv4SubnetsDisabled.clear()

   def _checkOverlappingSubnet( self, out ):
      """
      Update overlapping subnet error message and remove overlapping subnets

      @Input
         out (Dict) : kea config as a dict

      @Returns
         None
      """
      for subnetGenPrefix in self.status_.subnetOverlapped.keys():
         qt0( 'updating overlapped disable mesage' )
         msg = 'Subnet is disabled'
         subnetId = subnetGenPrefix.v4Prefix
         # we don't want to override the disable msg because the current one is
         # the most descriptive, which comes from kea.
         if subnetId not in self.status_.ipv4SubnetsDisabled:
            self.status_.ipv4SubnetsDisabled[ subnetId ] = msg

         # remove disabled subnet from potential config file
         subnets = out[ 'subnet4' ]
         self._removeSubnet( subnets, subnetId )

   def _checkConfigInternal( self, allConfig, process, level=0 ):
      # we only need to check this once
      out = allConfig[ 'Dhcp4' ]
      if level == 0:
         if self.config_.dhcpServerMode != self.config_.dhcpServerModeDefault and \
            self.config_.interfacesIpv4:
            self.status_.ipv4ServerDisabled = ''
         else:
            self.status_.ipv4ServerDisabled = self.status_.ipv4ServerDisabledDefault

      if process.wait():
         output = process.stderr.read()
         t0( "Malformed config", output )
         match = configErrorRe().search( output )

         # Unknown error message
         if not match or match.group( 'serverConfigFailure' ):
            self.status_.ipv4ServerDisabled = 'Server is disabled'

         # reservations mac-address error message
         elif featureStaticIPv4AddressStatus() and match.\
                                             group( 'reservationsConfigFailure' ):
            subnetId = configErrorSubnetIdRe().search( match.group( 'reason' ) ).\
                                               group( 'subnetId' ).strip( "\'" )
            subnetPrefix = self.status_.ipv4IdToSubnet[ int( subnetId ) ]

            # get failure reason
            reason = reservationsMacAddrFailureReason( match.group( 'reason' ) )
            msg = 'Subnet is disabled - {}'.format( reason )
            self.status_.ipv4SubnetsDisabled[ subnetPrefix ] = msg

            # remove the subnet configuration from potential config file
            subnets = out[ 'subnet4' ]
            self._removeSubnet( subnets, subnetPrefix )

            # check if there is another subnet with host reservations
            if not self._containsHostReservations( subnets ):
               del out[ 'host-reservation-identifiers' ]

         else:
            isPreExistingFailure = match.group( 'subnetExistingFailure' ) \
                                   is not None
            # get subnetID
            subnet = configErrorSubnetRe().search( match.group( 'reason' ) ).\
                     group( 'subnet' ).strip( "\'" )
            subnetId = Arnet.Prefix( subnet )

            # get failure reason
            reason = getSubnetFailureReason( match.group( 'reason' ),
                                             isPreExistingFailure )

            if isPreExistingFailure:
               reason = subnet + reason
            msg = 'Subnet is disabled - {}'.format( reason )
            self.status_.ipv4SubnetsDisabled[ subnetId ] = msg

            # Remove that subnet configuration from potential config file
            subnets = out[ 'subnet4' ]
            self._removeSubnet( subnets, subnetId )

         # since we can only catch ONE error at a time, remove the disabled subnet
         # and recursively check for another invalid subnet if any
         self.checkConfig( allConfig, level=level + 1 )
         return

      # we want to do this at the end to catch the most descriptive
      # disable messages from kea first
      self._checkOverlappingSubnet( out )

      # if there is any invalid subnet, we should NOT remove the tmpKeaConfFile
      # for troubleshooting/debugging purposes
      if not self.status_.ipv4SubnetsDisabled:
         t0( 'Removing the tmp file' )
         os.remove( self.tmpKeaConfFile_ )

      # only needed to gather malformed configs
      os.remove( self.tmpToCheckKeaConfFile_ )
      return

   def _removeSubnet( self, subnets, subnetPrefix ):
      for item in subnets:
         if item[ 'subnet' ] == str( subnetPrefix ):
            t0( 'Removing subnet:', subnetPrefix )
            subnets.remove( item )

   def _containsHostReservations( self, subnets ):
      for item in subnets:
         if item.get( "reservations" ):
            return True
      return False

   def getOrCreateSubnetOverlapped( self, subnetId ):
      """
      Get or create the overlapping subnets for a given subnetId

      @Input
         subnetId (Arnet::Prefix)

      @Returns
         subnetOverlapped (DhcpServer::SubnetOverlapped)
      """

      subnetGenPref = self.subnetGenPrefix( subnetId )
      subnetOverlapped = self.status_.subnetOverlapped.get( subnetGenPref )
      if not subnetOverlapped:
         subnetOverlapped = self.status_.subnetOverlapped.newMember( subnetGenPref )
      return subnetOverlapped

   def _updateSubnetOverlap( self, subnetId, add=True ):
      """
      Update the overlapping subnet(s) based on the newly added/deleted subnetId

      @Input
         subnetId (Arnet::Prefix) : subnet to be added/deleted
         add (Bool) : True if we are adding subnetId

      @Returns
         None
      """

      qt0( 'Updating overlapping subnets for:', qv( subnetId ) )
      subnetsOverlapped = []
      # get all the subnets that overlaps subnetId
      for subnet in self.status_.ipv4SubnetToId.keys():
         if subnetId == subnet:
            # Skip ourselves, happens during SSO
            continue
         if subnet.overlaps( subnetId ):
            qt0( 'overlapping subnet:', qv( subnet ) )
            subnetsOverlapped.append( subnet )

      subnetIdGenPref = self.subnetGenPrefix( subnetId )
      if add:
         if subnetsOverlapped:
            qt0( 'Adding overlapping subnet(s)' )
            for subnet in subnetsOverlapped:
               # set the overlapping flag to both subnets
               subnetOverlapped = self.getOrCreateSubnetOverlapped( subnetId )
               subnetGenPref = self.subnetGenPrefix( subnet )
               subnetOverlapped.overlappingSubnet[ subnetGenPref ] = True

               subnetOverlapped2 = self.getOrCreateSubnetOverlapped( subnet )
               subnetOverlapped2.overlappingSubnet[ subnetIdGenPref ] = True
      else:
         # update the status of the overlapped subnet
         qt0( 'Removing overlapping subnet(s)' )
         del self.status_.subnetOverlapped[ subnetIdGenPref ]
         for subnet in subnetsOverlapped:
            subnetOverlapped = self.getOrCreateSubnetOverlapped( subnet )
            del subnetOverlapped.overlappingSubnet[ subnetIdGenPref ]

            # remove if there are no more subnets overlapping it
            if not subnetOverlapped.overlappingSubnet:
               subnetGenPref = self.subnetGenPrefix( subnet )
               del self.status_.subnetOverlapped[ subnetGenPref ]

   def handleSubnetAdded( self, subnetId ):
      qt0( 'adding subnet:', qv( subnetId ) )
      self._updateSubnetOverlap( subnetId )
      subnetKeaId = self.status_.ipv4SubnetToId.get( subnetId, 0 )
      if not subnetKeaId:
         self.status_.ipv4SubnetLastUsedId = self.status_.ipv4SubnetLastUsedId + 1
         subnetKeaId = self.status_.ipv4SubnetLastUsedId
      self.status_.ipv4IdToSubnet[ subnetKeaId ] = subnetId
      self.status_.ipv4SubnetToId[ subnetId ] = subnetKeaId

   def handleSubnetRemoved( self, subnetId ):
      qt0( 'removing subnet:', qv( subnetId ) )
      _id = self.status_.ipv4SubnetToId.get( subnetId )
      if _id:
         del self.status_.ipv4IdToSubnet[ _id ]
      del self.status_.ipv4SubnetToId[ subnetId ]
      self._updateSubnetOverlap( subnetId, add=False )

   def handleVendorOptionAdded( self, vendorId ):
      qt0( 'adding vendor option:', qv( vendorId ) )
      spaceId = self.status_.ipv4VendorIdToSpaceId.get( vendorId )
      if not spaceId:
         self.status_.ipv4VendorIdSpaceLastUsedId += 1
         spaceId = self.status_.ipv4VendorIdSpaceLastUsedId
      self.status_.ipv4SpaceIdToVendorId[ spaceId ] = vendorId
      self.status_.ipv4VendorIdToSpaceId[ vendorId ] = spaceId

   def handleVendorOptionRemoved( self, vendorId ):
      qt0( 'removing vendor option:', qv( vendorId ) )
      spaceId = self.status_.ipv4VendorIdToSpaceId.get( vendorId )
      if spaceId:
         del self.status_.ipv4SpaceIdToVendorId[ spaceId ]
      del self.status_.ipv4VendorIdToSpaceId[ vendorId ]

   def configuredInterfaces( self ):
      return self.config_.interfacesIpv4

   def _leaseTime( self ):
      return int( self.config_.leaseTimeIpv4 )

   def _ddnsData( self ):
      return {}

   def genKeaIntfCfgCmds( self, linuxIntfName, ipAddrs ):
      return [ linuxIntfName ]

   def handleDisable( self, disabled ):
      if disabled:
         self.status_.ipv4ServerDisabled = 'Server is disabled'
      elif self.serviceEnabled():
         self.status_.ipv4ServerDisabled = ''
      else:
         self.status_.ipv4ServerDisabled = self.status_.ipv4ServerDisabledDefault

   def handleClearLease( self, clearId, start=True ):
      t1( "V4: handle clear lease" )
      self.stopService()
      # remove lease file
      for lf in glob.glob( self.leasePath_ + '*' ):
         os.remove( lf )
      self.status_.lastClearIdIpv4 = clearId
      if start:
         self.startService()

   def shutdown( self ):
      t0( 'server shutting down' )
      # Stop the server and delete all leases
      self.handleClearLease( self.status_.lastClearIdIpv4 + 1, start=False )
      self.status_.ipv4ServerDisabled = self.status_.ipv4ServerDisabledDefault

   def dhcpInterfaceStatus( self ):
      return self.status_.interfaceIpv4Disabled

   def serviceEnabled( self ):
      ''' Check if the DHCP server service is enabled  or if the configuration is
          adequate to start the service '''
      serverDisabled = self.status_.ipv4ServerDisabled == 'Server is disabled'
      return bool( self.intfCfg_ ) and not self.config_.disabled \
         and self.config_.dhcpServerMode and not serverDisabled

class DhcpServerIpv6Service( DhcpServerVrfService ):
   notifierTypeName = 'DhcpServer::Config'

   def __init__( self, config, status, vrf ):
      DhcpServerVrfService.__init__( self, config, status, vrf, "6" )

      self.status_.ipv6SubnetsDisabled.clear()
      self.status_.ipv6IdToSubnet.clear()
      self.status_.ipv6SubnetToId.clear()
      self.status_.ipv6SubnetLastUsedId = 0
      self.status_.lastClearIdIpv6 = 0
      self.status_.subnet6Overlapped.clear()
      self.status_.interfaceIpv6Disabled.clear()
      self.status_.subnetBroadcastStatus.clear()

      self.configReactor_ = DhcpServerReactor.ServiceV6Reactor( self.config_, self )

   def serviceProcessWarm( self ):
      ''' Check if the DHCP server service has fully restsarted '''
      return bool( self.config_.interfacesIpv6 )

   def _authoritative( self ):
      return False

   def _optionsData( self, config, isSubnetConfig=False ):
      options = []
      # subnet config options
      if isSubnetConfig:
         servers = config.dnsServers
         if config.subnetName:
            options.append( {
               'name': 'name',
               'data': config.subnetName } )
      # global config options
      else:
         servers = config.dnsServersIpv6
      # common for both subnet and global config
      if servers:
         options.append( {
            'name': 'dns-servers',
            'data': ', '.join( [ str( v ) for v in servers.values() ] ) } )

      return options

   def _ddnsData( self ):
      if self.config_.domainNameIpv6:
         return { "enable-updates": False,
                  'qualifying-suffix': self.config_.domainNameIpv6 }
      return {}

   def _allIntfMatchSubnet( self, subnetId, intfsToSubnets ):
      allIntfMatches = set()
      # Only look at active ones, since they are guarenteed to have addresses
      for intf, addrkeys in self.activeInterfaces_:
         for addrkey in addrkeys:
            # If the subnet contains the ip of the interface, and the subnet mask
            # length is <= to the mask length for the interface, then this subnet
            # can reply to broadcast requests on this interface
            if ( subnetId.contains( addrkey.addr.v6Addr ) and
                 subnetId.len <= addrkey.mask ):
               intfsToSubnets[ intf ].add( subnetId )
               allIntfMatches.add( intf )

      t5( "Subnet", str( subnetId ), "Interfaces", allIntfMatches )
      return allIntfMatches

   def _subnetConfig( self ):
      output = []
      intfsToSubnets = defaultdict( set )
      subnetConfigs = {}
      maybeSubnetValid = set()
      # host reservations are not supported for IPv6
      hostReservationIDs = set()
      self.status_.subnetBroadcastStatus.clear()
      for keaSubnetId, subnetId in self.status_.ipv6IdToSubnet.iteritems():
         subnetConfig = {}
         subnet = self.config_.subnetConfigIpv6[ subnetId ]
         subnetConfig[ 'subnet' ] = str( subnet.subnetId )
         subnetConfig[ 'id' ] = keaSubnetId

         allIntfMatches = self._allIntfMatchSubnet( subnetId, intfsToSubnets )
         self.status_.subnetBroadcastStatus[ subnetId ] = D6DRM.NO_IP6_ADDRESS_MATCH

         if len( allIntfMatches ) > 1:
            t4( "Subnet", str( subnetId ), "too many interface matches" )
            self.status_.subnetBroadcastStatus[ subnetId ] = (
               D6DRM.MULTIPLE_INTERFACES_MATCH_SUBNET.format(
                  " ".join( sorted( allIntfMatches ) ) ) )
         else:
            t4( "subnet", str( subnetId ), "Might be valid" )
            maybeSubnetValid.add( subnetId )

         # valid-lifetime
         # preferred-lifetime
         if subnet.leaseTime:
            subnetConfig[ 'valid-lifetime' ] = int( subnet.leaseTime )
            subnetConfig[ 'preferred-lifetime' ] = int( subnet.leaseTime )

         # subnet range
         if subnet.ranges:
            pools = []
            for _range in subnet.ranges.keys():
               pools.append( { 'pool': '{}-{}'.format( _range.start, _range.end ) } )
            subnetConfig[ 'pools' ] = pools

         # options data
         options = self._optionsData( subnet, isSubnetConfig=True )
         if options:
            subnetConfig[ 'option-data' ] = options

         subnetConfigs[ subnetId ] = subnetConfig
         output.append( subnetConfig )

      # Now check that each interface only matches a single subnet. If it does,
      # update the subnets config, otherwise update the subnet broadcast status.
      # Sort the items so the same config always generates the same status messages.
      for intf, subnets in sorted( intfsToSubnets.iteritems() ):
         if len( subnets ) == 1:
            subnetId = subnets.pop()
            if subnetId not in maybeSubnetValid:
               # More that 1 interface matched this subnet
               continue
            kniIntf = eosIntfToKernelIntf( intf )
            subnetConfigs[ subnetId ][ 'interface' ] = kniIntf
            del self.status_.subnetBroadcastStatus[ subnetId ]
         else:
            msg = D6DRM.MULTIPLE_SUBNETS_MATCH_INTERFACE.format( intf )
            t4( "Subnet", str( subnets ), "too many subnets match interface", intf )
            for subnet in subnets:
               self.status_.subnetBroadcastStatus[ subnet ] = msg

      return output, list( hostReservationIDs )

   def _resetStaleDisabledMessages( self ):
      """ clear disabled messages/reasons """
      self.status_.ipv6SubnetsDisabled.clear()

   def _checkOverlappingSubnet( self, out ):
      """
      Update overlapping subnet error message and remove overlapping subnets

      @Input
         out (Dict) : kea config as a dict

      @Returns
         None
      """
      for subnetGenPrefix in self.status_.subnet6Overlapped.keys():
         qt0( 'updating overlapped disable mesage' )
         msg = 'Subnet is disabled'
         subnetId = subnetGenPrefix.v6Prefix
         # we don't want to override the disable msg because the current one is
         # the most descriptive, which comes from kea.
         if subnetId not in self.status_.ipv6SubnetsDisabled.keys():
            self.status_.ipv6SubnetsDisabled[ subnetId ] = msg

         # remove disabled subnet from potential config file
         subnets = out[ 'subnet6' ]
         self._removeSubnet( subnets, subnetId )

   def _checkConfigInternal( self, allConfig, process, level=0 ):
      # we only need to check this once
      out = allConfig[ 'Dhcp6' ]
      if level == 0:
         if self.config_.dhcpServerMode != self.config_.dhcpServerModeDefault and \
            self.config_.interfacesIpv6:
            self.status_.ipv6ServerDisabled = ''
         else:
            self.status_.ipv6ServerDisabled = self.status_.ipv6ServerDisabledDefault

      if process.wait():
         output = process.stderr.read()
         t0( "Malformed config", output )
         match = configErrorRe().search( output )

         # Unknown error message
         if not match or match.group( 'serverConfigFailure' ):
            self.status_.ipv6ServerDisabled = 'Server is disabled'
         else:
            isPreExistingFailure = match.group( 'subnetExistingFailure' ) \
                                   is not None

            match2 = configErrorSubnetRe().search( match.group( 'reason' ) )
            if not match2:
               self.status_.ipv6ServerDisabled = 'Server is disabled'
               self.checkConfig( allConfig, level=level + 1 )
               return

            subnet = match2.group( 'subnet' ).strip( "'" )
            subnetId = Arnet.Ip6Prefix( subnet )

            # get failure reason
            reason = getSubnetFailureReason( match.group( 'reason' ),
                                             isPreExistingFailure )

            if isPreExistingFailure:
               reason = subnet + reason
            msg = 'Subnet is disabled - {}'.format( reason )
            self.status_.ipv6SubnetsDisabled[ subnetId ] = msg

            # Remove that subnet configuration from potential config file
            subnets = out[ 'subnet6' ]
            self._removeSubnet( subnets, subnetId )

         # since we can only catch ONE error at a time, remove the disabled subnet
         # and recursively check for another invalid subnet if any
         self.checkConfig( allConfig, level=level + 1 )
         return

      # we want to do this at the end to catch the most descriptive
      # disable messages from kea first
      self._checkOverlappingSubnet( out )

      # if there is any invalid subnet, we should NOT remove the tmpKeaConfFile
      # for troubleshooting/debugging purposes
      if not self.status_.ipv6SubnetsDisabled:
         t0( 'Removing the tmp file' )
         os.remove( self.tmpKeaConfFile_ )

      # only needed to gather malformed configs
      os.remove( self.tmpToCheckKeaConfFile_ )
      return

   def _removeSubnet( self, subnets, subnetPrefix ):
      for item in subnets:
         if item[ 'subnet' ] == str( subnetPrefix ):
            t0( 'Removing subnet:', subnetPrefix )
            subnets.remove( item )

   def getOrCreateSubnetOverlapped( self, subnetId ):
      """
      Get or create the overlapping subnets for a given subnetId

      @Input
         subnetId (Arnet::Ip6Prefix)

      @Returns
         subnetOverlapped (DhcpServer::SubnetOverlapped)
      """

      subnetGenPref = self.subnetGenPrefix( subnetId )
      subnet6Overlapped = self.status_.subnet6Overlapped.get( subnetGenPref )
      if not subnet6Overlapped:
         subnet6Overlapped = (
                   self.status_.subnet6Overlapped.newMember( subnetGenPref ) )
      return subnet6Overlapped

   def _updateSubnetOverlap( self, subnetId, add=True ):
      """
      Update the overlapping subnet(s) based on the newly added/deleted subnetId

      @Input
         subnetId (Arnet::Ip6Prefix) : subnet to be added/deleted
         add (Bool) : True if we are adding subnetId

      @Returns
         None
      """

      qt0( 'Updating overlapping subnets for:', qv( subnetId ) )
      subnets6Overlapped = []
      # get all the subnets that overlaps subnetId
      for subnet in self.status_.ipv6SubnetToId.keys():
         if subnetId == subnet:
            # Skip ourselves, happens during SSO
            continue
         if subnet.overlaps( subnetId ):
            qt0( 'Overlapping subnet:', qv( subnet ) )
            subnets6Overlapped.append( subnet )

      subnetIdGenPref = self.subnetGenPrefix( subnetId )
      if add:
         if subnets6Overlapped:
            qt0( 'Adding overlapping subnet(s)' )
            for subnet in subnets6Overlapped:
               # set the overlapping flag to both subnets
               subnet6Overlapped = self.getOrCreateSubnetOverlapped( subnetId )
               subnetGenPref = self.subnetGenPrefix( subnet )
               subnet6Overlapped.overlappingSubnet[ subnetGenPref ] = True

               subnet6Overlapped2 = self.getOrCreateSubnetOverlapped( subnet )
               subnet6Overlapped2.overlappingSubnet[ subnetIdGenPref ] = True
      else:
         # update the status of the overlapped subnet
         qt0( 'Removing overlapping subnet(s)' )
         del self.status_.subnet6Overlapped[ subnetIdGenPref ]
         for subnet in subnets6Overlapped:
            subnet6Overlapped = self.getOrCreateSubnetOverlapped( subnet )
            del subnet6Overlapped.overlappingSubnet[ subnetIdGenPref ]

            # remove if there are no more subnets overlapping it
            if not subnet6Overlapped.overlappingSubnet:
               subnetGenPref = self.subnetGenPrefix( subnet )
               del self.status_.subnet6Overlapped[ subnetGenPref ]

   def handleSubnetAdded( self, subnetId ):
      qt0( 'adding subnet:', qv( subnetId ) )
      self._updateSubnetOverlap( subnetId )
      subnetKeaId = self.status_.ipv6SubnetToId.get( subnetId, 0 )
      if not subnetKeaId:
         self.status_.ipv6SubnetLastUsedId = self.status_.ipv6SubnetLastUsedId + 1
         subnetKeaId = self.status_.ipv6SubnetLastUsedId
      self.status_.ipv6IdToSubnet[ subnetKeaId ] = subnetId
      self.status_.ipv6SubnetToId[ subnetId ] = subnetKeaId
      self.status_.subnetBroadcastStatus[ subnetId ] = D6DRM.NO_IP6_ADDRESS_MATCH

   def handleSubnetRemoved( self, subnetId ):
      qt0( 'removing subnet:', qv( subnetId ) )
      _id = self.status_.ipv6SubnetToId.get( subnetId )
      if _id:
         del self.status_.ipv6IdToSubnet[ _id ]
      del self.status_.ipv6SubnetToId[ subnetId ]
      del self.status_.subnetBroadcastStatus[ subnetId ]
      self._updateSubnetOverlap( subnetId, add=False )

   def configuredInterfaces( self ):
      return self.config_.interfacesIpv6

   def _leaseTime( self ):
      return int( self.config_.leaseTimeIpv6 )

   def handleDisable( self, disabled ):
      if disabled:
         self.status_.ipv6ServerDisabled = 'Server is disabled'
      elif self.serviceEnabled():
         self.status_.ipv6ServerDisabled = ''
      else:
         self.status_.ipv6ServerDisabled = self.status_.ipv6ServerDisabledDefault

   def handleClearLease( self, clearId, start=True ):
      t1( "V6: handle clear lease" )
      self.stopService()
      # remove lease file
      for lf in glob.glob( self.leasePath_ + '*' ):
         os.remove( lf )
      self.status_.lastClearIdIpv6 = clearId
      if start:
         self.startService()

   def shutdown( self ):
      # Stop the server and delete all leases
      self.handleClearLease( self.status_.lastClearIdIpv6 + 1, start=False )
      self.status_.ipv6ServerDisabled = self.status_.ipv6ServerDisabledDefault

   def dhcpInterfaceStatus( self ):
      return self.status_.interfaceIpv6Disabled

   def genKeaIntfCfgCmds( self, linuxIntfName, ipAddrs ):
      cfg = set()
      for addrKey in ipAddrs:
         # Add the interface name and all ipv6 non link local addresses to kea
         # This allows kea to reply to relayed (unicast) requests
         cfg.add( linuxIntfName + "/" + str( addrKey.addr ) )

      return cfg

   def serviceEnabled( self ):
      ''' Check if the DHCP server service is enabled  or if the configuration is
          adequate to start the service '''
      serverDisabled = self.status_.ipv6ServerDisabled == 'Server is disabled'
      return bool( self.intfCfg_ ) and not self.config_.disabled \
             and self.config_.dhcpServerMode and not serverDisabled

class DhcpServerVrfManager( object ):
   '''
   Starts the ipv4 or ipv6 dhcp server for a specific vrf
   '''
   def __init__( self, vrf, config, status, kniStatus, manager,
                 dhcpRelayHelperConfig ):
      self.vrf_ = vrf
      self.config_ = config
      self.status_ = status
      self.kniStatus_ = kniStatus
      self.manager_ = manager
      self.dhcpRelayHelperConfig_ = dhcpRelayHelperConfig
      self.interfaceToKni_ = {}
      self.intferfaceToAddrs_ = {}
      self.v4Service_ = DhcpServerIpv4Service( config, status, vrf )
      self.v6Service_ = DhcpServerIpv6Service( config, status, vrf )
      self.vrfManagerReactor_ = DhcpServerReactor.VrfManagerReactor( self.config_,
                                                                     self )
      self.kniReactor_ = DhcpServerReactor.KniReactor( self.kniStatus_, self )
      self.dhcpRelayHelperConfigReactor_ = (
         DhcpServerReactor.DhcpRelayHelperConfigReactor(
         self.dhcpRelayHelperConfig_, self ) )

      # resync config and status after startup
      self.handleDisable( self.config_.disabled )

   def services( self ):
      return [ self.v4Service_, self.v6Service_ ]

   def shutdown( self ):
      t2( "VrfM: shutdown" )
      for service in self.services():
         service.shutdown()

   def handleDisable( self, disabled ):
      t2( "VrfM: handleDisable" )
      for service in self.services():
         service.handleDisable( disabled )

   def handleIntfChange( self ):
      t2( "VrfM: handleIntfChange" )
      for service in self.services():
         self._handleActiveInterfaces( service, self.vrf_ )

   def _handleActiveInterfaces( self, service, vrf ):
      ipVersion = service.ipVersion_
      interfacesConfigured = service.configuredInterfaces()
      dhcpIntfStatus = service.dhcpInterfaceStatus()
      activeInterfaces = []
      intfCfg = set()
      if not interfacesConfigured:
         t3( "No interfaces configured for", ipVersion )
         service.handleActiveIntfs( intfCfg, activeInterfaces )
         return

      for intf in interfacesConfigured:
         intfLocal = self.manager_.intfStatusLocal_.intfStatusLocal.get( intf )
         if not intfLocal:
            t1( "Skipping %s (%s), no intf status local" % ( ipVersion, intf ) )
            dhcpIntfStatus[ intf ] = DISM.NO_INTF_STATUS_LOCAL_MSG
            continue
         if intfLocal.netNsName != vrf.nsName():
            t1( "Skipping %s (%s), not in default vrf" % ( ipVersion, intf ) )
            dhcpIntfStatus[ intf ] = DISM.NOT_IN_DEFAULT_VRF_MSG
            continue
         if self.dhcpRelayHelperConfig_.alwaysOn:
            t1( "Skipping %s, relay is always on" % intf )
            dhcpIntfStatus[ intf ] = DISM.DHCP_RELAY_ALWAYS_MSG
            continue
         if intf in self.dhcpRelayHelperConfig_.intfConfig:
            t1( "Skipping %s, relay is configured on this interface" % intf )
            dhcpIntfStatus[ intf ] = DISM.DHCP_RELAY_CFG_MSG
            continue
         linuxIntfName = eosIntfToKernelIntf( intf )
         kniIntf = self.interfaceToKni_.get( linuxIntfName )
         if not kniIntf:
            t1( "Skipping %s (%s), no kni" % ( ipVersion, intf ) )
            dhcpIntfStatus[ intf ] = DISM.NO_KNI_MSG
            continue
         if kniIntf.flags & IFF_RUNNING != IFF_RUNNING:
            t1( "Skipping %s (%s), not running" % ( ipVersion, intf ) )
            dhcpIntfStatus[ intf ] = DISM.NOT_UP_MSG
            continue
         ipAddrs = self._ipActiveOnInterface( kniIntf.key, ipVersion )
         if not ipAddrs:
            t1( "Skipping %s (%s), no ip address" % ( ipVersion, intf ) )
            dhcpIntfStatus[ intf ] = DISM.NO_IP_ADDRESS_MSG
            continue
         keaIntfCfgCmds = service.genKeaIntfCfgCmds( linuxIntfName, ipAddrs )

         intfCfg.update( keaIntfCfgCmds )
         activeInterfaces.append( ( intf, ipAddrs ) )
         del dhcpIntfStatus[ intf ]

      service.handleActiveIntfs( intfCfg, activeInterfaces )

   def _ipActiveOnInterface( self, intfKey, ipVersion ):
      addrKeys = self.intferfaceToAddrs_.get( intfKey )
      if not addrKeys:
         return []

      addrs = []

      addrFamily = (
         addrFamilyEnum.ipv4 if ipVersion == "4" else addrFamilyEnum.ipv6 )

      for addrKey in addrKeys:
         addrKey = addrKey
         ipState = self.kniStatus_.addr.get( addrKey )
         if not ipState:
            continue
         if ipState.key.addr.af != addrFamily:
            continue
         if addrKey.addr.isV6LinkLocal:
            # For now, V6 requires an ip address
            continue
         if ipState.assigned:
            addrs.append( addrKey )
            if ipVersion == "4":
               # V4 doesn't care about all the addrs, just that one is assigned
               return addrs
      return addrs

   def handleDebugLog( self, filePath ):
      t2( "VrfM: handleDebugLog <%s>" % filePath )
      for service in self.services():
         service.handleDebugLog()

class DhcpServerManager( SuperServer.SuperServerAgent ):
   '''
   Starts the DhcpCserverVrfManagers
   '''
   def __init__( self, entityManager ):
      SuperServer.SuperServerAgent.__init__( self, entityManager )
      mg = entityManager.mountGroup()
      self.vrfManagers_ = {}
      self.defaultVrf_ = Tac.ValueConst( 'L3::VrfName', DEFAULT_VRF )
      self.configs_ = {}
      self.status_ = {}
      self.kniStatus_ = {}
      self.intfStatusLocal_ = None
      self.intfStatusLocalReactor_ = None
      self.configReactors_ = {}

      # mounting configs to react to
      self.configs_[ self.defaultVrf_ ] = mg.mount( 'dhcpServer/config',
                                                    'DhcpServer::Config', 'r' )
      self.status_[ self.defaultVrf_ ] = mg.mount( 'dhcpServer/status',
                                                   'DhcpServer::Status', 'w' )
      self.dhcpRelayHelperConfig = mg.mount( 'ip/helper/dhcprelay/config',
                                       'Ip::Helper::DhcpRelay::Config', 'r' )
      shmemEm = SharedMem.entityManager( sysdbEm=self.entityManager )
      # Only for default vrf/ns for now
      self.kniStatus_[ self.defaultVrf_ ] = shmemEm.doMount(
         "kni/ns/%s/status" % self.defaultVrf_.nsName(), "KernelNetInfo::Status",
         Smash.mountInfo( 'keyshadow' ) )

      def _finished():
         t2( "Fishing mounting" )
         if self.active():
            self.onSwitchover( None )
      mg.close( _finished )

   def onSwitchover( self, protocol ):
      t2( "onSwitchover" )
      self.intfStatusLocal_ = self.intfStatusLocal()
      self.intfStatusLocalReactor_ = Tac.collectionChangeReactor(
          self.intfStatusLocal_.intfStatusLocal,
          DhcpServerReactor.IntfStatusLocalReactor,
          reactorArgs=( weakref.proxy( self ), ) )
      self.configReactors_[ self.defaultVrf_ ] = DhcpServerReactor.ManagerReactor(
         self.configs_[ self.defaultVrf_ ], self.defaultVrf_, self )

   def handleIntfChange( self ):
      t2( "M: handleIntfChange" )
      for vrfM in self.vrfManagers_.values():
         vrfM.handleIntfChange()

   def handleDhcpServerMode( self, mode, vrf ):
      t2( "handleDhcpServerMode" )
      config = self.configs_[ vrf ]
      status = self.status_[ vrf ]
      kniStatus = self.kniStatus_[ vrf ]
      if mode != config.dhcpServerModeDefault:
         t2( "M: Enabling" )
         if vrf not in self.vrfManagers_:
            t2( "M: Creating" )
            self.vrfManagers_[ vrf ] = DhcpServerVrfManager(
               vrf, config, status, kniStatus, self, self.dhcpRelayHelperConfig )
      else:
         t2( "M: Disabling" )
         vrfM = self.vrfManagers_.pop( vrf, None )
         if vrfM:
            t2( "M: Deleting" )
            vrfM.shutdown()

def Plugin( ctx ):
   ctx.registerService( DhcpServerManager( ctx.entityManager ) )
