# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os
import signal
import re
import weakref
import errno
from collections import OrderedDict
import Arnet.NsLib
import Cell
import Ethernet
import SuperServer
import QuickTrace
from IpUtils import Mask
from IpLibConsts import DEFAULT_VRF
from DhcpClientConsts import DHCP_DEFAULT_ROUTE_PREFERENCE
import Logging
import EntityManager
import Tac

qv = QuickTrace.Var
qt0 = QuickTrace.trace0
qt1 = QuickTrace.trace1

# The following table explains whether this SuperServerPlugin
# is active on a given supervisor, and which interfaces is it
# responsible for
#
# ------------------------------------------------------------------------------- #
#         |                SSO                |                RPR                #
# ------------------------------------------------------------------------------- #
# Standby |    Not active                     |    Local management interface     #
# ------------------------------------------------------------------------------- #
# Active  |    Front panel ports,             |    Front panel ports,             #
#         |    Port-channels, SVIs,           |    Port-channels, SVIs,           #
#         |    Local management interface,    |    Local management interface     #
#         |    Peer's management interface    |                                   #
# ------------------------------------------------------------------------------- #

def supeManagesIntf( intf, isActiveSupe, redundancyProtocol ):
   """
   Determines if intf is allowed to be managed by the current supervisor
   """
   match = re.match( r'Management(\d)/', intf )
   cellId = str( Cell.cellId() )

   # Non-management interfaces and Management0 should be configurable
   # by active supe only
   if not match:
      return isActiveSupe

   # intf is this supervisor's management interface
   if match.group( 1 ) == cellId:
      # sso-standby supe should not try to configure its own management interface
      if redundancyProtocol == 'sso' and not isActiveSupe:
         return False
      return True

   # Only sso-active supe should be allowed to configure its peer's
   # management interface
   if redundancyProtocol == 'sso' and isActiveSupe:
      return True
   return False

def canCreateServiceForStandbyMgmtIntf( intf, isActiveSupe, redundancyProtocol ):
   """
   Determines whether this supervisor should create a DhclientService
   for standby's management interface
   """
   if not isStandbyMgmtIntf( intf, isActiveSupe ):
      return False

   return ( redundancyProtocol == 'sso' and isActiveSupe ) or \
          ( redundancyProtocol == 'rpr' and not isActiveSupe )

def isStandbyMgmtIntf( intf, isActiveSupe ):
   match = re.match( r'Management(\d)/', intf )
   cellId = str( Cell.cellId() )

   # Not a management interface
   if not match:
      return False

   if match.group( 1 ) == cellId:
      return not isActiveSupe

   return isActiveSupe

def supportedIntf( allIntfStatusDir, intf ):
   return hasattr( allIntfStatusDir.intfStatus.get( intf ), 'routedAddr' )

class HostnameReactor( Tac.Notifiee ):
   """ Reacts to the hostname changes """
   notifierTypeName = "System::NetStatus"

   def __init__( self, netStatus, dhclientMgr ):
      self.dhclientMgr_ = weakref.proxy( dhclientMgr )
      Tac.Notifiee.__init__( self, netStatus )
      self.netStatus_ = netStatus

   @Tac.handler( 'hostname' )
   def handleHostname( self ):
      qt0( "handleHostname(): ", qv( self.notifier_.hostname ) )

      # Restart all the dhclients
      for _, dhclient in self.dhclientMgr_.services_.iteritems():
         dhclient.hostname_ = self.notifier_.hostname
         dhclient.sync()

class IpIntfConfigReactor( Tac.Notifiee ):
   notifierTypeName = "Ip::IpIntfConfig"

   def __init__( self, notifier, dhclientMgr ):
      self.dhclientMgr_ = weakref.proxy( dhclientMgr )
      Tac.Notifiee.__init__( self, notifier )
      self.handleAddrSource()

   @Tac.handler( 'addrSource' )
   def handleAddrSource( self ):
      qt0( "handleAddrSource()", qv( self.notifier_.intfId ),
                                 qv( self.notifier_.addrSource ) )
      self.dhclientMgr_.doCreateOrDeleteService( self.notifier_.intfId )

   @Tac.handler( 'defaultRouteSource' )
   def handledefaultRouteSource( self ):
      vrf = None
      qt0( "handleDefaultRouteSource()", qv( self.notifier_.intfId ),
                                 qv( self.notifier_.defaultRouteSource ) )

      # In case of default vrf, the notifier_.vrf is an empty string
      if self.notifier_.vrf:
         vrf = self.notifier_.vrf
      # XXX-panchamukhi: remove defaultRouteCli argument, temporarily
      # used to fix abuild SuperSeverDhcpTest failure
      self.dhclientMgr_.doCreateOrDeleteService( self.notifier_.intfId,
                                                 vrf,
                                                 defaultRouteCli=True )

   def close( self ):
      self.dhclientMgr_.doCreateOrDeleteService( self.notifier_.intfId )
      Tac.Notifiee.close( self )

class L3StatusReactor( Tac.Notifiee ):
   notifierTypeName = "L3::Intf::Status"

   def __init__( self, l3Status, ipIntfStatusReactor ):
      self.ipIntfStatusReactor_ = weakref.proxy( ipIntfStatusReactor )
      Tac.Notifiee.__init__( self, l3Status )

   @Tac.handler( 'vrf' )
   def handleVrf( self ):
      self.ipIntfStatusReactor_.handleVrf()

class IpIntfStatusReactor( Tac.Notifiee ):
   notifierTypeName = "Ip::IpIntfStatus"

   def __init__( self, notifier, dhclientMgr ):
      self.dhclientMgr_ = weakref.proxy( dhclientMgr )
      Tac.Notifiee.__init__( self, notifier )
      self.l3StatusReactor_ = L3StatusReactor( self.notifier_.l3Status, self )

      self.prevVrf = self.notifier_.vrf
      dhclientMgr.doCreateOrDeleteService( self.notifier_.intfId,
                                           vrf=self.notifier_.vrf )

   def handleVrf( self ):
      qt0( "handleVrf()", qv( self.notifier_.intfId ), qv( self.notifier_.vrf ) )
      # Note that the 'vrf' attribute in IpIntfStatus is an alias of the 'vrf'
      # attribute in L3::Intf::Status. This is why we don't need
      # IpIntfStatusReactor to react to anything.
      intf = self.notifier_.intfId
      vrf = self.notifier_.vrf

      managementStandby = canCreateServiceForStandbyMgmtIntf( intf,
                           self.dhclientMgr_.active(),
                           self.dhclientMgr_.redundancyProtocol() )
      prevService = self.dhclientMgr_.service( self.prevVrf, managementStandby )
      if prevService and prevService.managingIntf( intf ):
         prevService.maybeRelease( intf, release=False, cleanup=True )

      self.prevVrf = vrf
      self.dhclientMgr_.doCreateOrDeleteService( intf, vrf=vrf )

   def close( self ):
      self.dhclientMgr_.doCreateOrDeleteService( self.notifier_.intfId,
                                                 vrf=self.notifier_.vrf )
      Tac.Notifiee.close( self )

class EthPhyIntfStatusReactor( Tac.Notifiee ):
   notifierTypeName = "Interface::EthPhyIntfStatus"

   def __init__( self, notifier, dhclientMgr, serviceName ):
      self.dhclientMgr_ = dhclientMgr
      self.serviceName_ = serviceName
      Tac.Notifiee.__init__( self, notifier )

   @Tac.handler( 'operStatus' )
   def handleOperStatus( self ):
      qt0( "handleOperStatus()", qv( self.notifier_.intfId ),
                                 qv( self.notifier_.operStatus ) )
      dhclientService = self.dhclientMgr_.services_[ self.serviceName_ ]
      dhclientService.updateIntfsBlockingStatus( [ self.notifier_.intfId ] )

   @Tac.handler( 'forwardingModel' )
   def handleForwardingModel( self ):
      qt0( "handleForwardingModel()", qv( self.notifier_.intfId ),
                                      qv( self.notifier_.forwardingModel ) )
      dhclientService = self.dhclientMgr_.services_[ self.serviceName_ ]
      dhclientService.updateIntfsBlockingStatus( [ self.notifier_.intfId ] )

   def close( self ):
      self.dhclientMgr_.doCreateOrDeleteService( self.notifier_.intfId )
      Tac.Notifiee.close( self )

class VrfStatusLocalReactor( Tac.Notifiee ):
   notifierTypeName = "Ip::VrfStatusLocal"

   def __init__( self, notifier, dhclientMgr ):
      self.dhclientMgr_ = weakref.proxy( dhclientMgr )
      Tac.Notifiee.__init__( self, notifier )

   @Tac.handler( 'state' )
   def handleState( self ):
      qt0( "handleState()", qv( self.notifier_.vrfName ),
                            qv( self.notifier_.state ) )
      state = self.notifier_.state
      if state == 'deleting':
         self.dhclientMgr_.delServiceFromVrfAll( self.notifier_.vrfName )

   def close( self ):
      # XXX-bavishi: One can argue that this isn't needed because VrfSm first
      # changes the state to 'deleting', and then deletes the entity. Keeping
      # it anyway just to be safe from future VrfSm changes
      self.dhclientMgr_.delServiceFromVrfAll( self.notifier_.vrfName )
      Tac.Notifiee.close( self )

def unlinkFile( fileName ):
   try:
      os.unlink( fileName )
   except OSError, e:
      if e.errno != errno.ENOENT:
         raise

# Wrapper of Tac.ClockNotifiee so we don't have to manipulate
# Tac.ClockNotifiee.timeMin directly. W0223 (unimplemented abstract method)
# is disabled locally for DhclientActivity and ReleaseActivity definitions
# because we don't care about implementing Tac.ClockNotifiee._getHandler method.
class DhclientActivity( Tac.ClockNotifiee ): # pylint: disable-msg=W0223
   def __init__( self, handler, interval ):
      Tac.ClockNotifiee.__init__( self )
      self.handler = handler
      self.interval = interval
      # Activity should be inactive by default
      self.timeMin = Tac.endOfTime

   def start( self ):
      self.timeMin = Tac.now() + self.interval

   def stop( self ):
      self.timeMin = Tac.endOfTime

   def active( self ):
      return self.timeMin != Tac.endOfTime

class ReleaseActivity( DhclientActivity ): # pylint: disable-msg=W0223
   def __init__( self, handler, interval, maxIntfPerRelease=10 ):
      DhclientActivity.__init__( self, handler, interval )
      # Map interfaceName -> deviceName
      self.releasingIntfs = OrderedDict()
      self.maxIntfPerRelease = maxIntfPerRelease
      self.releasing = False
      self.cleanup = False
      # Number of call to dhclient -r, created for use in btest
      self.releases = 0

   def intfCount( self ):
      return len( self.releasingIntfs )

   def intfPending( self, intf ):
      return intf in self.releasingIntfs

   def addIntf( self, intf, devName ):
      if not self.intfPending( intf ):
         qt0( 'Scheduling release for', qv( intf ) )
         self.releasingIntfs[ intf ] = devName
         # Schedule activity
         self.start()

   def removeIntf( self, intf ):
      if self.intfPending( intf ):
         del self.releasingIntfs[ intf ]
         # No need to automatically stop the activity even if releasingIntfs
         # is empty since we may still need to clean up or restart service

   def popIntf( self ):
      # Remove and return the first item of OrderedDict
      # The returning value is a tuple: ( interfaceName, deviceName )
      return self.releasingIntfs.popitem( last=False )

   def releasingIs( self, releasing ):
      self.releasing = releasing
      if releasing:
         self.releases += 1

   def cleanupIs( self, cleanup ):
      self.cleanup = cleanup

class DhclientService( SuperServer.LinuxService ):
   notifierTypeName = "*"

   def __init__( self, dhclientMgr, sysname, ipConfig, ipStatus,
                 dhclientStatus, allIntfStatusDir, allVrfStatusLocal,
                 vrfName, hostname, isServiceForMgmtIntfOfStandby=False ):
      self.dhclientMgr_ = weakref.proxy( dhclientMgr )
      self.sysname_ = sysname
      self.ipConfig_ = ipConfig
      self.ipStatus_ = ipStatus
      self.dhclientStatus_ = dhclientStatus
      self.allIntfStatusDir_ = allIntfStatusDir
      self.allVrfStatusLocal_ = allVrfStatusLocal
      self.vrfName_ = vrfName
      self.hostname_ = hostname
      self.isServiceForMgmtIntfOfStandby_ = isServiceForMgmtIntfOfStandby

      filenameSuffix = vrfName
      if isServiceForMgmtIntfOfStandby:
         filenameSuffix = filenameSuffix + '-standby'
      self.configPath_ = '/etc/dhcp/dhclient-%s.conf' % filenameSuffix
      self.pidPath_ = '/var/run/dhclient-%s.pid' % filenameSuffix
      self.scriptPath_ = '/etc/dhcp/dhclient-script.py'
      self.leasePath_ = '/var/lib/dhclient/dhclient-%s.leases' % filenameSuffix
      unlinkFile( self.configPath_ )

      # List of all the interfaces that DHCP is currently enabled on and
      # are being managed by a dhclient process
      self.intfs_ = []
      # Functions that, once DHCP is enabled, would prevent an IP address
      # being obtained
      self.blockingFuncs_ = [ self._operStatusBlocking,
                              self._forwardingBlocking ]
      # List of interfaces that have DHCP enabled but are currently
      # blocked by the above functions
      self.intfsBlocked_ = []

      # The attribute below indicates whether dhclient was expected to be
      # running or not. In certain breadth tests, where we don't run dhclient,
      # this is used as a proxy to know whether it was expected to be running.
      self.dummyDhclientRunning_ = False

      # Batch release DHCP leases
      self.doReleaseActivity_ = ReleaseActivity( self._doRelease, interval=1 )

      SuperServer.LinuxService.__init__( self, 'dhclient', 'dhclient',
                                         self.ipConfig_, self.configPath_ )

      # If the config is enabled, this function will try to start the
      # dhclient by calling _maybeRestartService()
      self._syncFromSysdb()

      # NOTE: It is important to pass a weak reference here. If you pass a
      # strong reference, that reference will be held up in the
      # handleCollectionChange closure( in Tac.py ) and will not be GC'ed
      serviceName = vrfName + "-standby" if isServiceForMgmtIntfOfStandby \
                    else vrfName
      self.ethPhyIntfStatusCollReactor_ = Tac.collectionChangeReactor(
            allIntfStatusDir.intfStatus, EthPhyIntfStatusReactor,
            reactorArgs=( self.dhclientMgr_, serviceName ), )

   def serviceProcessWarm( self ):
      # Check whether the dhclient is running by looking
      # at the pid file
      if not self.serviceEnabled():
         return True

      return self.started()

   def serviceEnabled( self ):
      return len( self.intfs_ )

   def getPid( self ):
      try:
         return int( file( self.pidPath_ ).read() )
      except IOError as e:
         if e.errno != errno.ENOENT:
            raise e
      except ValueError:
         pass
      qt1( 'dhclient is not running: pid not found' )
      return -1

   def startService( self ):
      if self.serviceEnabled() and not self.started():
         devNames = map( self._devName, self.intfs_ )
         cmd = self._cmdPrefix()
         cmd += [ '-cf', self.configPath_ ]
         cmd += [ '-lf', self.leasePath_ ]

         for index in range( len( self.intfs_ ) ):
            # Pass devName=intfName mapping as environment variables.
            # dhclient will pass these on to the dhclient-script.
            # dhclient also passes an interface env variable to dhclient-script
            # which is actually the kernel net device name. In dhclient-script,
            # we instantiate an entity inside the IntfStatus collection
            # indexed by intfName. To get that intfName from the passed net-device
            # name, we pass this mapping.
            cmd += [ '-e', '%s=%s' % ( devNames[ index ], self.intfs_[ index ] ) ]

         qt0( "Starting dhclient" )
         self.runCmd( cmd, self.vrfName_ )

      if self.dhclientMgr_.runDummyDhclient_:
         self.dummyDhclientRunning_ = True

   def stopService( self ):
      if self.dhclientMgr_.runDummyDhclient_:
         self.dummyDhclientRunning_ = False
         return

      pid = self.getPid()
      if pid <= 0:
         return

      qt0( "Killing dhclient" )
      # Always run this in the default VRF because if the non-default VRF
      # is deleted before we could kill dhclient, this command would fail
      try:
         os.kill( pid, signal.SIGTERM )
         if self.started():
            # Kill forcefully
            os.kill( pid, signal.SIGKILL )
      except OSError:
         # Process might have already terminated
         pass

   def restartService( self ):
      qt0( 'Restart service' )
      self.stopService()
      self._dumpLeases()
      if self.started():
         # Dhclient process isn't going away yet, schedule another attempt
         # to make sure startService() works.
         self.sync()
      self.startService()

   def conf( self ):
      if not self.intfs_:
         qt0( "No interface has dhcp option configured. Return empty config file" )
         return ''

      if self.hostname_:
         hostNameLine = 'send host-name "%s";' % self.hostname_
      else:
         hostNameLine = ""

      # Time in lease file is stored as epoch seconds instead of # W yyyy/dd/mm h:m:s
      confFile = "db-time-format local;\n"
      for intf in self.intfs_:
         assert intf in self.allIntfStatusDir_.intfStatus
         intfStatus = self.allIntfStatusDir_.intfStatus[ intf ]
         assert self.ipConfig_.ipIntfConfig[ intf ].addrSource == 'dhcp'
         clientIdLine = 'send dhcp-client-identifier "Arista-%s-%s";' % (
                     Ethernet.convertMacAddrToDisplay( intfStatus.routedAddr ),
                     Tac.Value( "Arnet::IntfId", intf ).shortName )
         confFile += """
interface "%s" {
   %s
   %s
}
""" % ( self._devName( intf ),
        clientIdLine,
        hostNameLine )

      qt0( "Config file contents are" )
      qt0( qv( confFile ) )
      return confFile

   def _leases( self ):
      leases = ''
      for intf in self.intfs_:
         if intf not in self.dhclientStatus_.intfStatus:
            qt0( "dhclient-script has not yet responded to the request "
                "since", qv( intf ), "doesn't yet have an IntfStatus object" )
            continue
         dhclientStatus = self.dhclientStatus_.intfStatus[ intf ]
         leases += '''
lease {
   interface "%s";
   fixed-address %s;
   option subnet-mask %s;
   option routers %s;
   renew epoch %d;
   rebind epoch %d;
   expire epoch %d;
}
''' % ( self._devName( intf ), dhclientStatus.addrWithMask.address,
        str( Mask( dhclientStatus.addrWithMask.len ) ),
        dhclientStatus.dhcpRouters, dhclientStatus.renewalTime,
        dhclientStatus.rebindingTime, dhclientStatus.expiryTime )

      qt0( "Contents of the lease file are" )
      qt0( qv( leases ) )
      return leases

   def _dumpLeases( self ):
      """
      Write whatever we remember from the Sysdb to the lease file.
      This would ensure that in the event of switchover, the IP of
      all the interfaces on the new active is same as that on the
      old active. This also covers the SuperServer restart case.
      """
      open( self.leasePath_, 'w' ).write( self._leases() )

   def addIntf( self, intf ):
      if not self.managingIntf( intf ):
         self.doReleaseActivity_.removeIntf( intf )
         self.intfsBlocked_.append( intf )
         self.updateIntfsBlockingStatus( [ intf ] )

   def _delIntf( self, intf, keepStatus ):
      if intf in self.intfs_:
         self.intfs_.remove( intf )

      if intf in self.intfsBlocked_:
         self.intfsBlocked_.remove( intf )

      if intf in self.dhclientStatus_.intfStatus and not keepStatus:
         qt0( "Deleting dhclientIntfStatus for", qv( intf ) )
         del self.dhclientStatus_.intfStatus[ intf ]

   def _maybeCleanupOrSync( self, cleanup=False, sync=True ):
      if cleanup and not self.managingAnyIntf():
         self.dhclientMgr_.delServiceFromVrf( self.vrfName_,
             self.isServiceForMgmtIntfOfStandby_ )
      elif sync:
         self.sync()
      else:
         self._maybeRestartService()

   def _doRelease( self ):
      if self.doReleaseActivity_.releasing and self.started():
         qt0( 'Waiting for dhclient -r to die' )
         self.doReleaseActivity_.start()
         return

      intfCount = self.doReleaseActivity_.intfCount()
      if intfCount == 0:
         qt0( "Release done!" )
         self.doReleaseActivity_.releasingIs( False )
         self._maybeCleanupOrSync( cleanup=self.doReleaseActivity_.cleanup,
                                   sync=False )
         return

      count = min( self.doReleaseActivity_.maxIntfPerRelease, intfCount )
      cmd = self._cmdPrefix()
      cmd += [ '-r' ]
      # We don't want to pass the config file here since that will cause
      # the dhclient to release the lease on all the interfaces that are
      # present in the config file. We can't also leave the -cf option
      # as this will cause the dhclient to read the default config file
      # which is /etc/dhcp/dhclient.conf. Passing /dev/null as a workaround.
      cmd += [ '-cf', '/dev/null' ]
      cmd += [ '-lf', self.leasePath_ ]
      envArgs = []
      intfArgs = []
      for _ in range( count ):
         ( intf, devName ) = self.doReleaseActivity_.popIntf()
         qt0( "Releasing the lease for", qv( intf ) )
         envArgs += [ '-e', '%s=%s' % ( devName, intf ) ]
         # We are not passing the config file here. So, dhclient needs to know
         # the interface from the command line itself
         intfArgs += [ devName ]

      cmd += envArgs
      cmd += intfArgs
      # Don't try to release the lease from default VRF in case this command
      # fails in non-default VRF since the net-device for an interface in
      # a non-default VRF will not be available in the default VRF
      self.runCmd( cmd, self.vrfName_ )
      self.doReleaseActivity_.releasingIs( True )
      # Simulate the fact that 'dhclient -r' will kill the running dhclient
      if self.dhclientMgr_.runDummyDhclient_:
         self.dummyDhclientRunning_ = False
      self.doReleaseActivity_.start()

   def maybeRelease( self, intf, release=False,
                     cleanup=False, keepStatus=False ):
      '''
      This function does the following tasks:
      - Delete the interface and its dhclient status from the managing lists.
      - If its lease needs to be released, the interface is added to the
        releasing queue. Otherwise, depending on input parameters and current
        config, we will either clean up the service or call sync() to
        schedule a service restart.
      - If keepStatus=True, the dhclient status (if any) of intf will not be
        deleted by _delIntf( ... ). This status can be used later to generate
        the lease for intf when it comes back from blocked state.
      '''
      qt0( 'maybeRelease() vrf', qv( self.vrfName_ ), qv( intf ),
           'release', qv( release ), 'cleanup', qv( cleanup ) )
      # Call dhclient with a release option
      # DhclientService will kill dhclient for all the interfaces in
      # this VRF. We need to restart the service( which will generate the
      # config first )
      intfInNonBlockingList = intf in self.intfs_

      # Remove the interface from intfs list
      self._delIntf( intf, keepStatus )

      # Scheduling doReleaseActivity_
      if intfInNonBlockingList and release:
         # Disable self.sync(); _maybeRestartService will be called
         # after releasing is done
         self.activity_.timeMin = Tac.endOfTime
         self.doReleaseActivity_.addIntf( intf, self._devName( intf ) )

      if not self.doReleaseActivity_.active():
         # If there is no scheduled release, do cleanup or sync right away
         self._maybeCleanupOrSync( cleanup=cleanup )
      else:
         # Save the cleanup intention to be used later by _doRelease
         self.doReleaseActivity_.cleanupIs( cleanup )

   def _operStatusBlocking( self, intf ):
      qt0( "_operStatusBlocking()", qv( intf ),
           qv( self.allIntfStatusDir_.intfStatus[ intf ].operStatus ) )
      return self.allIntfStatusDir_.intfStatus[ intf ].operStatus != 'intfOperUp'

   def _forwardingBlocking( self, intf ):
      qt0( "_forwardingBlocking()", qv( intf ),
           qv( self.allIntfStatusDir_.intfStatus[ intf ].forwardingModel ) )
      return self.allIntfStatusDir_.intfStatus[ intf ].forwardingModel != \
               'intfForwardingModelRouted'

   def managingAnyIntf( self ):
      return self.intfs_ or self.intfsBlocked_

   def managingIntf( self, intf ):
      return intf in self.intfs_ or intf in self.intfsBlocked_

   def updateIntfsBlockingStatus( self, intfList ):
      for intf in intfList:
         if not self.managingIntf( intf ):
            continue

         # Check if the intf is blocked by any blocking function
         isBlocked = any( func( intf ) for func in self.blockingFuncs_ )
         if isBlocked:
            if intf in self.intfs_:
               self.maybeRelease( intf, release=False, keepStatus=True )
               self.intfsBlocked_.append( intf )
               if intf in self.dhclientStatus_.intfStatus:
                  self.dhclientStatus_.intfStatus[ intf ].pending = True
            else:
               assert intf not in self.dhclientStatus_.intfStatus or \
                   self.dhclientStatus_.intfStatus[ intf ].pending
         else:
            if intf in self.intfsBlocked_:
               self.intfsBlocked_.remove( intf )
               self.intfs_.append( intf )
               self.sync()

   def _syncFromSysdb( self ):
      """
      While (re)starting, look for the interfaces that have dhcp enabled
      and are members of the vrf in which this DhclientService is created.
      Finally call updateIntfsBlockingStatus() which will call _maybeRestartService()
      if necessary and cause dhclient to be started with the new config.
      """
      def _managing( intf ):
         isActiveSupe = self.dhclientMgr_.active()
         returnVal = supportedIntf( self.allIntfStatusDir_, intf ) and \
                self.ipConfig_.ipIntfConfig[ intf ].addrSource == 'dhcp' and \
                intf in self.ipStatus_.ipIntfStatus and \
                self.ipStatus_.ipIntfStatus[ intf ].vrf == self.vrfName_ and \
                not isStandbyMgmtIntf( intf, isActiveSupe )
         return returnVal

      intfs = []
      if self.isServiceForMgmtIntfOfStandby_:
         cellId = Cell.peerCellId() if self.dhclientMgr_.active() else Cell.cellId()
         standbyMgmtIntf = "Management%d/1" % cellId
         if standbyMgmtIntf in self.ipConfig_.ipIntfConfig:
            intfs = filter( _managing, [ standbyMgmtIntf ] )
      else:
         intfs = filter( _managing, self.ipConfig_.ipIntfConfig.keys() )
      self.intfsBlocked_ = list( intfs )
      self.updateIntfsBlockingStatus( intfs )

   def started( self ):
      if self.dhclientMgr_.runDummyDhclient_:
         # dhclient is not actually running. Return what we remember
         return self.dummyDhclientRunning_
      pid = self.getPid()
      if pid <= 0:
         return False
      try:
         os.kill( pid, 0 )
      except OSError as e:
         if e.errno == errno.ESRCH:
            return False
         elif e.errno == errno.EPERM:
            return True
         else:
            raise
      else:
         return True

   def _cmdPrefix( self ):
      prefix = []
      prefix += [ 'dhclient' ]
      # VRFNAME is passed to the dhclient to save per vrf dhcp 'option routers'
      # in the sysdb.
      prefix += [ '-e', 'SYSNAME=%s' % self.sysname_, '-e',
                  'SYSDBSOCKNAME=%s' % EntityManager.sysdbSockname(),
                  '-e', 'VRFNAME=%s' % self.vrfName_ ]
      prefix += [ '-sf', self.scriptPath_ ]
      prefix += [ '-pf', self.pidPath_ ]
      return prefix

   def runCmd( self, cmd, vrf=DEFAULT_VRF ):
      # QuickTrace only traces the first 24 characters. First 9 members
      # of cmd list are constant for a given DhclientService. The information
      # that what command ( start/release/stop ) is being executed is
      # contained after these 9 members that come from _cmdPrefix()
      qt0( "runCmd()", qv( cmd[ 9 : ] ), qv( vrf ) )
      if self.dhclientMgr_.runDummyDhclient_:
         # No need to run any commands.
         return

      if vrf == DEFAULT_VRF:
         ns = Arnet.NsLib.DEFAULT_NS
      else:
         ns = self.allVrfStatusLocal_.vrf[ self.vrfName_ ].networkNamespace
      try:
         Arnet.NsLib.runMaybeInNetNs( ns, cmd, asRoot=True, asDaemon=True )
      except ( Tac.SystemCommandError, Tac.Timeout ) as e:
         # We can hit one of the following errors here:
         #    * ENOENT: Network namespace was deleted when the command was issued
         #    * ENODEV: The kernel net-device was deleted and dhclient failed to
         #      find the hardware address corresponding to it
         #    * One more dhclient got started because of a race condition(BUG165979)
         # All of these are harmless and can be ignored
         qt1( qv( str( e ) ) )

   def _devName( self, intf ):
      # To retrieve the device name, the interface must be present in
      # allIntfStatusDir
      assert intf in self.allIntfStatusDir_.intfStatus
      deviceName = self.allIntfStatusDir_.intfStatus[ intf ].deviceName
      if self.dhclientMgr_.redundancyProtocol() == 'sso' and \
         self.isServiceForMgmtIntfOfStandby_:
         # Use management active intf to send dhcp request for management
         # standby intf
         deviceName = re.sub( r'(man?)(\d)', r"\g<1>%d" % Cell.cellId(),
                              deviceName )
      return deviceName

   def _checkServiceHealth( self ):
      if not self.serviceEnabled():
         return

      if not self.started() and not self.serviceRestartPending_ and \
         not self.doReleaseActivity_.active():
         Logging.log( SuperServer.SYS_RESTART_SERVICE, # pylint: disable-msg=E1101
                      self.linuxServiceName_ )
         self.startServicePunchWatchdog()

      self.healthMonitorActivity_.timeMin = Tac.now() + \
                                               self.healthMonitorInterval

   def sync( self ):
      if not self.doReleaseActivity_.active():
         SuperServer.LinuxService.sync( self )

   def cleanupDhclientStatus( self ):
      for intf in self.intfs_:
         self.maybeRelease( intf, release=False )
      for intf in self.intfsBlocked_:
         self.maybeRelease( intf, release=False )

   def cleanupService( self ):
      qt0( 'cleanupService()', qv( self.vrfName_ ) )
      self.ethPhyIntfStatusCollReactor_ = None
      self.blockingFuncs_ = []
      self.cleanupDhclientStatus()
      self.stopService()
      unlinkFile( self.configPath_ )
      self.doReleaseActivity_.stop()
      SuperServer.LinuxService.cleanupService( self )

class DhclientManager( SuperServer.SuperServerAgent ):
   def __init__( self, entityManager ):
      SuperServer.SuperServerAgent.__init__( self, entityManager )
      self.sysname_ = entityManager.sysname()
      mg = entityManager.mountGroup()

      self.ipConfig = mg.mount( 'ip/config', "Ip::Config", 'r' )
      # Mount l3/intf/config for ip/config l3Config Ptr.
      mg.mount( "l3/intf/config", "L3::Intf::ConfigDir", "r" )
      Tac.Type( "Ira::IraIpStatusMounter" ).doMountEntities( mg.cMg_, True, False )
      self.ipStatus = mg.mount( 'ip/status', "Ip::Status", 'r' )
      self.dhclientStatus = mg.mount( 'ip/dhclient/status',
                                      "Dhclient::Status", 'w' )
      self.allVrfStatusLocal = mg.mount( Cell.path( 'ip/vrf/status/local' ),
                                         "Ip::AllVrfStatusLocal", 'r' )
      self.dhcpVrfRouteInput = mg.mount( 'routing/vrf/input', "Tac::Dir", 'wi' )
      self.dhcpRouter = mg.mount( "routing/vrf/input/default/dhcpRouter",
                                  "Routing::RouteInput", "w" )
      self.netStatus = mg.mount( Cell.path( 'sys/net/status' ),
                                 "System::NetStatus", 'r' )

      self.allIntfStatusDir = self.intfStatusAll()
      self.vrfStatusLocalCollReactor_ = None
      self.ipIntfConfigCollReactor_ = None
      self.ipIntfStatusCollReactor_ = None
      self.hostnameReactor_ = None

      self.services_ = {}

      # The attribute below is set to True so that we don't run dhclient for
      # certain breadth tests.
      self.runDummyDhclient_ = 'DUMMY_DHCLIENT' in os.environ

      def _finished():
         self.updateDhcpRouterMount()
         # Don't do anything on sso-standby
         if self.isSsoStandby():
            qt0( "sso-standby: No-op" )
            return
         self.createReactors()

      mg.close( _finished )

   def isSsoStandby( self ):
      return ( self.redundancyProtocol() == 'sso' and
               not self.active() )

   def warm( self ):
      # Be always warm on the sso-standby supervisor
      if self.isSsoStandby():
         return True

      # Be warm if there are no DhclientServices
      if not self.services_:
         return True
      # Be warm only if all the DhclientServices are warm
      return all( service.warm() for service in self.services_.values() )

   def startService( self, vrfName=DEFAULT_VRF,
                     isServiceForMgmtIntfOfStandby=False ):
      # We start one DhclientService per VRF
      # isServiceForMgmtIntfOfStandby=True means this service will manage
      # standby supe's management interface. Note that this service will
      # run on active supe in sso mode and on standby in rpr mode
      return DhclientService( self, self.sysname_, self.ipConfig, self.ipStatus,
                              self.dhclientStatus, self.allIntfStatusDir,
                              self.allVrfStatusLocal, vrfName,
                              self.netStatus.hostname,
                              isServiceForMgmtIntfOfStandby )

   def service( self, vrfName, isServiceForMgmtIntfOfStandby=False ):
      name = vrfName
      if name and isServiceForMgmtIntfOfStandby:
         name = name + "-standby"
      return self.services_.get( name )

   def addDhcpVrfRouteInputDir( self, vrfName ):
      # XXX-panchamukhi: Move this code to vrf reactor, to avoid multi-writer issue.
      # Also kernelFib deletes subdir in routeInputDir, Add code to stop stop service
      # when subdirs in routeInputDir gets deleted.
      if vrfName not in self.dhcpVrfRouteInput.keys():
         vrfDir = self.dhcpVrfRouteInput.newEntity( "Tac::Dir", vrfName )
      else:
         vrfDir = self.dhcpVrfRouteInput[ vrfName ]

      if 'dhcpRouter' not in vrfDir.keys():
         vrfRouteInput = vrfDir.newEntity( "Routing::RouteInput", "dhcpRouter" )
         routingProtocol = Tac.Type( "Routing::RoutingProtocol" )
         vrfRouteInput.routingProto = routingProtocol.dhcp
         vrfRouteInput.defaultPreference = DHCP_DEFAULT_ROUTE_PREFERENCE
         vrfRouteInput.showIpRouteKey = "D"
         vrfRouteInput.redistributeName = "dhcp"

   def updateDhcpRouterMount( self ):
      if self.active():
         routingProtocol = Tac.Type( "Routing::RoutingProtocol" )
         self.dhcpRouter.routingProto = routingProtocol.dhcp
         self.dhcpRouter.defaultPreference = DHCP_DEFAULT_ROUTE_PREFERENCE
         self.dhcpRouter.showIpRouteKey = "D"

   def addServiceToVrf( self, vrfName, isServiceForMgmtIntfOfStandby=False ):
      name = vrfName
      if isServiceForMgmtIntfOfStandby:
         name = name + "-standby"

      if name not in self.services_:
         self.services_[ name ] = self.startService( vrfName=vrfName,
                  isServiceForMgmtIntfOfStandby=isServiceForMgmtIntfOfStandby )
      return self.services_[ name ]

   def delServiceFromVrf( self, vrfName, isServiceForMgmtIntfOfStandby=False ):
      name = vrfName
      if isServiceForMgmtIntfOfStandby:
         name = name + "-standby"

      if name in self.services_:
         self.services_[ name ].cleanupService()
         del self.services_[ name ]

   def delServiceFromVrfAll( self, vrfName ):
      self.delServiceFromVrf( vrfName )
      self.delServiceFromVrf( vrfName, isServiceForMgmtIntfOfStandby=True )

   def createReactors( self ):
      if self.vrfStatusLocalCollReactor_ is None:
         self.vrfStatusLocalCollReactor_ = Tac.collectionChangeReactor(
               self.allVrfStatusLocal.vrf, VrfStatusLocalReactor,
               reactorArgs=( self, ) )
      if self.ipIntfConfigCollReactor_ is None:
         self.ipIntfConfigCollReactor_ = Tac.collectionChangeReactor(
               self.ipConfig.ipIntfConfig, IpIntfConfigReactor,
               reactorArgs=( self, ) )
      if self.ipIntfStatusCollReactor_ is None:
         self.ipIntfStatusCollReactor_ = Tac.collectionChangeReactor(
               self.ipStatus.ipIntfStatus, IpIntfStatusReactor,
               reactorArgs=( self, ) )
      if self.hostnameReactor_ is None:
         self.hostnameReactor_ = HostnameReactor( self.netStatus, self )

   def delMgmtStandbyService( self ):
      mgmtStandbyServiceName = None
      for name in self.services_:
         if self.services_[ name ].isServiceForMgmtIntfOfStandby_:
            mgmtStandbyServiceName = name
            break
      self.delServiceFromVrf( mgmtStandbyServiceName,
                              isServiceForMgmtIntfOfStandby=False )

   def onProtocolChange( self, protocol ):
      # We don't need to do anything on the standby since it will
      # be rebooted when redundancy protocol changes
      if not self.active():
         return

      # This is active supe. If sso, we need to start a DhclientService for
      # standby's management interface. Otherwise, we need to delete the same
      if protocol == 'sso':
         mgmtStandbyIntf = ( "Management%d/1" % Cell.peerCellId() )
         self.doCreateOrDeleteService( mgmtStandbyIntf )
      else:
         self.delMgmtStandbyService()

   def onSwitchover( self, protocol ):
      """Called when switchover from standby to active happens"""
      self.updateDhcpRouterMount()
      if protocol == 'sso':
         self.createReactors()
      else:
         self.delMgmtStandbyService()
         for intf in self.ipConfig.ipIntfConfig:
            self.doCreateOrDeleteService( intf )

   def doCreateOrDeleteService( self, intf, vrf=None, defaultRouteCli=False ):
      if not supeManagesIntf( intf, self.active(), self.redundancyProtocol() ):
         return

      if vrf is None and intf in self.ipStatus.ipIntfStatus:
         vrfName = self.ipStatus.ipIntfStatus[ intf ].vrf
      else:
         vrfName = vrf
      qt0( "doCreateOrDeleteService", qv( intf ), qv( vrfName ) )
      standbyMgmtIntf = canCreateServiceForStandbyMgmtIntf( intf, self.active(),
                                                     self.redundancyProtocol() )
      service = self.service( vrfName, standbyMgmtIntf )

      if( intf not in self.ipStatus.ipIntfStatus or
          intf not in self.ipConfig.ipIntfConfig or
          intf not in self.allIntfStatusDir ):
         qt0( "IpIntfConfig or IpIntfStatus or EthPhyIntfStatus for",
              qv( intf ), "does not exist." )
         if service and service.managingIntf( intf ):
            service.maybeRelease( intf, release=False, cleanup=True )
         return
      if not supportedIntf( self.allIntfStatusDir, intf ):
         qt0( qv( intf ), "is not a supported interface type" )
         return

      addrSource = self.ipConfig.ipIntfConfig[ intf ].addrSource
      defaultRouteSource = self.ipConfig.ipIntfConfig[ intf ].defaultRouteSource
      dhcpEnabled = ( addrSource == 'dhcp' )
      defaultRouteEnabled = ( defaultRouteSource == 'dhcp' )
      if dhcpEnabled and service is None:
         service = self.addServiceToVrf( vrfName, standbyMgmtIntf )

      if dhcpEnabled:
         self.addDhcpVrfRouteInputDir( vrfName )

      handledByService = service.managingIntf( intf ) if service else False

      if dhcpEnabled and not handledByService:
         qt0( "Manual -> DHCP" )
         service.addIntf( intf )
      elif not dhcpEnabled and handledByService:
         qt0( "DHCP -> Manual" )
         service.maybeRelease( intf, release=True, cleanup=True )
      elif dhcpEnabled and handledByService and defaultRouteCli:
         if defaultRouteEnabled:
            qt0( "DHCP -> DHCP Default Route" )
         else:
            # XXX-panchamukhi: No need to delete the dhcpRoute subdir here, just
            # stop the service to avoid accessing dhcpRoute, since kernelFib
            # deletes the dhcpRoute subdir in routeInput
            qt0( "DHCP Default Route -> DHCP" )
         # XXX-panchamukhi: Should really call self.sync() here
         service.maybeRelease( intf, release=False )
         if not service.managingIntf( None ):
            self.delServiceFromVrf( vrfName, standbyMgmtIntf )
         self.addServiceToVrf( vrfName, standbyMgmtIntf )

def Plugin( ctx ):
   ctx.registerService( DhclientManager( ctx.entityManager ) )
