# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import SuperServer
import QuickTrace
import Tac
import os, sys
import Tracing
import PyWrappers.Docker as docker

traceHandle = Tracing.Handle( "ContainerMgr" )
t0 = traceHandle.trace0 # function calls
t1 = traceHandle.trace1 # error/exception
t2 = traceHandle.trace2 # login/logout from a registry
t3 = traceHandle.trace3 # container info traces

qv = QuickTrace.Var
qt0 = QuickTrace.trace0 # Important function calls
qt1 = QuickTrace.trace1 # error/exception
qt2 = QuickTrace.trace2 # Other important info

def isConnectionIssue( error, login=False ):
   t0( "isConnectionIssue called." )
   errorStr = [ 'docker: Error while pulling image',
                 'connect: network is unreachable',
                 'connection refused',
                 'server misbehaving' ]
   if login:
      errorStr += [ 'unauthorized: incorrect username or password' ]
   for e in errorStr:
      if e in error:
         return True
   return False

class RegistryReactor( Tac.Notifiee ):
   notifierTypeName = "ContainerMgr::RegistryConfig"

   def __init__( self, notifier, master ):
      Tac.Notifiee.__init__( self, notifier )
      self.master_ = master
      self.registry = notifier
      self.prevServerName = ""
      self.activity_ = Tac.ClockNotifiee()
      self.activity_.handler = self.login
      self.activity_.timeMin = Tac.endOfTime
      self.activityInterval = 30
      self.retryCounter = 0
      self.maxRetry = 30
      self.handleInsecure()
      self.handleAuth()

   def login( self ):
      t0( "login called. Registry is %s, server is %s and user is %s"
          % ( self.registry.name, self.registry.serverName,
             self.registry.userName ) )
      msg = ""
      if self.registry.insecure:
         msg = "insecure "
      if self.registry.userName == self.registry.invalidString:
         msg += "username "
      if self.registry.password == self.registry.invalidString:
         msg += "password "
      if self.registry.serverName == self.registry.invalidString:
         msg += "server "

      if msg:
         t1( "login cannot be done due to invalid %s" % msg )
         self.retryCounter = 0
         self.activity_.timeMin = Tac.endOfTime
         return
      cmd = [ 'docker', 'login', '-u', self.registry.userName,
              '-p', self.registry.password, self.registry.serverName ]
      err = Tac.run( cmd, stdout=sys.stdout, stderr=Tac.CAPTURE, asRoot=True,
                     ignoreReturnCode=True )
      if err:
         if isConnectionIssue( err, login=True ):
            # Retry after every 30 seconds for 15 minutes and then give up
            t1( "Login failed. Registry is %s, server is %s and user is %s"
                % ( self.registry.name, self.registry.serverName,
                    self.registry.userName ) )
            qt1( "Login failed. Registry is ", qv( self.registry.name ),
                 " server is ", qv( self.registry.serverName ), " and user is ",
                 qv( self.registry.userName ) )
            self.retryCounter += 1
            if self.retryCounter <= self.maxRetry:
               self.activity_.timeMin = Tac.now() + self.activityInterval
            else:
               self.activity_.timeMin = Tac.endOfTime
      else:
         self.activity_.timeMin = Tac.endOfTime
         self.retryCounter = 0

   def logout( self ):
      if not self.prevServerName:
         return
      t0( "logout called. Registry is %s and server is %s"
          % ( self.registry.name, self.prevServerName ) )
      cmd = [ 'docker', 'logout', self.prevServerName ]
      try:
         Tac.run( cmd, stdout=sys.stdout, stderr=sys.stderr, asRoot=True )
      except Tac.SystemCommandError:
         t1( "Logout failed.  Registry is %s and server is %s"
             % ( self.registry.name, self.prevServerName ) )
         qt1( "Logout failed.  Registry is ", qv( self.registry.name ),
              " and server is ", qv( self.prevServerName ) )

   def handleAuth( self ):
      t0( "handleAuth called. Registry is %s" % self.registry.name )
      if self.registry.insecure:
         return
      if self.registry.invalidString in [
            self.registry.userName,
            self.registry.password ]:
         self.logout()
         return
      if self.registry.serverName != self.prevServerName:
         if self.prevServerName:
            t2( "logging out from server %s" % self.prevServerName )
            self.logout()
         if self.registry.serverName:
            t2( "logging in into server %s" % self.registry.serverName )
            self.login()
         self.prevServerName = self.registry.serverName
      else:
         # registry became secure from insecure
         t2( "logging in into server %s" % self.registry.serverName )
         self.login()

   @Tac.handler( 'insecure' )
   def handleInsecure( self ):
      t0( "handleInsecure called. Registry is %s" % self.registry.name )
      if self.registry.serverName != self.registry.invalidString:
         self.master_.notifiee_.sync()
      # Ideally we should logout in else for below condition
      if not self.registry.insecure:
         self.handleAuth()

   @Tac.handler( 'serverName' )
   def handleServerName( self ):
      t0( "handleServer called. Server is %s" % self.registry.serverName )
      if self.registry.insecure:
         self.master_.notifiee_.sync()
      else:
         self.handleAuth()

   @Tac.handler( 'userName' )
   def handleUserName( self ):
      t0( "handleUserName called. Username is %s" % self.registry.userName )
      self.handleAuth()

   @Tac.handler( 'password' )
   def handlePassword( self ):
      t0( "handlePassword called" )
      self.handleAuth()

   def close( self ):
      t0( "close for RegistryReactor called" )
      # This check is just for safety. prevServerName should be empty.
      if self.prevServerName:
         self.logout()
      self.activity_.timeMin = Tac.endOfTime
      Tac.Notifiee.close( self )

class ContainerConfigReactor( Tac.Notifiee ):
   notifierTypeName = "ContainerMgr::Container"

   def __init__( self, notifier ):
      Tac.Notifiee.__init__( self, notifier )
      self.containerConfig = notifier
      self.activity_ = Tac.ClockNotifiee()
      self.activity_.handler = self.runContainer
      self.activity_.timeMin = Tac.endOfTime
      self.activityInterval = 30
      self.retryCounter = 0
      self.maxRetry = 30
      self.handleOnBoot()

   def runContainerCmd( self, cmd ):
      t0( "runContainerCmd called. cmd is %s" % cmd )
      err = Tac.run( cmd, stdout=sys.stdout, stderr=Tac.CAPTURE, asRoot=True,
                     ignoreReturnCode=True )
      if err:
         t1( "container is unable to run. cmd is %s due to %s" % ( cmd, err ) )
         qt1( "container is unable to run. cmd is ", qv( cmd ),
              "due to ", qv( err ) )
         if 'run' in cmd:
            if isConnectionIssue( err ):
               self.retryCounter += 1
               if self.retryCounter <= self.maxRetry:
                  self.activity_.timeMin = Tac.now() + self.activityInterval
               else:
                  self.activity_.timeMin = Tac.endOfTime
      else:
         if 'run' in cmd:
            self.retryCounter = 0
            self.activity_.timeMin = Tac.endOfTime

   def runContainer( self ):
      t0( "runContainer called. Container is %s" % self.containerConfig.name )
      imageName = self.containerConfig.imageName
      runCmd = [ 'docker', 'run' ]
      stopCmd = [ 'docker', 'stop' ]
      rmCmd = [ 'docker', 'rm' ]
      cmdOptions = []
      containerCmd = []

      cmdOptions += [ '--cpu-shares', str( self.containerConfig.cpuShares ) ]
      cmdOptions += [ '--cpuset-cpus', str( self.containerConfig.cpuCores ) ]
      cmdOptions += [ '--memory', str(  self.containerConfig.memory ) ]
      cmdOptions += [ '--name', self.containerConfig.name ]
      cmdOptions += [ '--restart', 'on-failure:10' ]
      cmdOptions += [ '--detach' ]
      if self.containerConfig.options:
         cmdOptions += self.containerConfig.options.split()
      if self.containerConfig.command:
         containerCmd = self.containerConfig.command.split()

      containerStopCmd = stopCmd + [ self.containerConfig.containerName ]
      t3( "Stopping container %s. Stop cmd is %s"
          % ( self.containerConfig.name, containerStopCmd ) )
      qt2( "Stopping container ", qv( self.containerConfig.name ),
           ". Stop cmd is ", qv( containerStopCmd ) )
      self.runContainerCmd( containerStopCmd )
      containerRemoveCmd = rmCmd +  [ self.containerConfig.name ]
      t3( "Removing container %s. Remove cmd is %s"
          % ( self.containerConfig.name, containerRemoveCmd ) )
      qt2( "Removing container ", qv( self.containerConfig.name ),
           ". Remove cmd is ", qv( containerRemoveCmd ) )
      self.runContainerCmd( containerRemoveCmd )

      if imageName:
         containerRunCmd = runCmd + cmdOptions + [ imageName ] + containerCmd
         t3( "Running container %s. Run cmd is %s"
             % ( self.containerConfig.name, containerRunCmd ) )
         qt2( "Running container ", qv( self.containerConfig.name ),
              ". Run cmd is ", qv( containerRunCmd ) )
         self.runContainerCmd( containerRunCmd )
      else:
         self.retryCounter = 0
         self.activity_.timeMin = Tac.endOfTime

   @Tac.handler( 'command' )
   def handleCommand( self ):
      t0( "handleCommand called. command is %s" % self.containerConfig.command )
      self.runContainer()

   @Tac.handler( 'options' )
   def handleOptions( self ):
      t0( "handleOptions called. options are %s" % self.containerConfig.options )
      self.runContainer()

   @Tac.handler( 'memory' )
   def handleMemory( self ):
      t0( "handleMemory called. memory is %s" % self.containerConfig.memory )
      self.runContainer()

   @Tac.handler( 'cpuCores' )
   def handleCpuCores( self ):
      t0( "handleCpuCores called. cpu cores is %s" % self.containerConfig.cpuCores )
      self.runContainer()

   @Tac.handler( 'cpuShares' )
   def handleCpuShares( self ):
      t0( "handleCpuShares called. cpu shares is %s"
          % self.containerConfig.cpuShares )
      self.runContainer()

   @Tac.handler( 'imageName' )
   def handleImageName( self ):
      t0( "handleImageName called. image name is %s"
          % self.containerConfig.imageName )
      self.runContainer()

   def handleOnBoot( self ):
      t0( "handleOnBoot called" )
      if self.containerConfig.onBoot:
         t3( "Starting on-boot container %s" % self.containerConfig.containerName )
         qt2( "Starting on-boot container ",
              qv( self.containerConfig.containerName ) )
         self.runContainer()

   def close( self ):
      t0( "close for ContainerConfigReactor called" )
      containerName = self.containerConfig.containerName
      containerRemoveCmd = [ 'docker', 'rm', '-f', containerName ]
      self.runContainerCmd( containerRemoveCmd )
      self.activity_.timeMin = Tac.endOfTime
      Tac.Notifiee.close( self )

class ContainerMgrConfigReactor( SuperServer.LinuxService ):
   notifierTypeName = "ContainerMgr::ContainerMgrConfig"

   def __init__( self, containerMgrConfig ):
      assert os.path.exists( "/usr/bin/docker" )
      SuperServer.LinuxService.__init__( self, "ContainerMgr", docker.name(),
                                         containerMgrConfig,
                                         "/etc/sysconfig/docker" )
      self.config_ = containerMgrConfig
      self.handleDaemonEnable()

   def serviceEnabled( self ):
      t0( "serviceEnabled called. daemonEnable is %s"
          % self.config_.daemonEnable )
      return self.config_.daemonEnable

   def serviceProcessWarm( self ):
      t0( "serviceProcessWarm called." )
      return self.serviceEnabled() and os.path.exists( "/var/run/docker.pid" )

   def insecureRegistry( self ):
      t0( "insecureRegistry called." )
      cf = ""
      for registry in self.config_.registryConfig.values():
         if registry.insecure and registry.serverName != registry.invalidString:
            cf += " --insecure-registry=%s" % registry.serverName

      return cf

   def handleAuth( self, login ):
      t0( "handleAuth from ContainerMgrConfigReactor called." )
      qt0( "handleAuth from ContainerMgrConfigReactor called." )
      for registry in self.config_.registryConfig.values():
         if not registry.insecure and registry.serverName and \
               registry.userName and registry.password:
            cmd = [ 'docker' ]
            if login:
               cmd += [ 'login', '-u', registry.userName,
                        '-p', registry.password ]
            else:
               cmd += [ 'logout' ]
            cmd += [ registry.serverName ]
            try:
               t2( "Trying login/logout from %s. cmd is %s"
                   % ( registry.serverName, cmd ) )
               Tac.run( cmd, stdout=sys.stdout, stderr=sys.stderr, asRoot=True )
            except Tac.SystemCommandError:
               t1( "login/logout cmd failed. cmd is %s" % cmd )
               qt1( "login/logout cmd failed. cmd is ", qv( cmd ) )

   def conf( self ):
      t0( "conf called." )
      qt0( "conf called." )
      cf = ""
      cf += "other_args='-s overlay %s %s'\n" % ( self.config_.containerMgrArgs,
                                                  self.insecureRegistry() )
      cf += "DOCKER_CERT_PATH=/etc/docker\n"
      cf += "DOCKER_NOWARN_KERNEL_VERSION=1\n"
      return cf

   def loadImages( self ):
      t0( "loadImages called." )
      qt0( "loadImages called." )
      files = []
      loadCmd = [ 'docker', 'load' ]
      persistPaths = [ self.config_.defaultPersistentPath ]
      if self.config_.persistentPath != self.config_.defaultPersistentPath:
         persistPaths += [ self.config_.persistentPath ]

      for path in persistPaths:
         persistPath = path + '.containermgr/'
         if os.path.exists( persistPath ):
            files += os.listdir( persistPath )
         print >> sys.stderr, "Loading backed up containers from %s" % (
            persistPath )
         for fileName in files:
            absolutePath = persistPath + fileName
            t3( "Loading from absolute path %s" % absolutePath )
            cmd = loadCmd + [ '-i', absolutePath ]
            try:
               Tac.run( cmd, stdout=sys.stdout, stderr=sys.stderr, asRoot=True )
               qt2( "Image loaded using ", qv( cmd ) )
               t3( "Image loaded using %s", cmd )
            except Tac.SystemCommandError as e:
               t1( 'ContainerMgr image backed up at %s cannot be loaded'\
               ' due to %s' % ( absolutePath, e.output ) )
               qt1( "ContainerMgr image backed up at ", qv( absolutePath ),
                    " cannot be loaded due to ", qv( e.output ) )

   @Tac.handler( 'daemonEnable' )
   def handleDaemonEnable( self ):
      t0( "handleDaemonEnable called." )
      qt0( "handleDaemonEnable called." )
      if self.config_.daemonEnable:
         self.startService()
         self.loadImages()
      else:
         self.stopService()
      # handleAuth not really required here, but let's keep it for safety.
      self.handleAuth( self.config_.daemonEnable )

   @Tac.handler( 'containerMgrArgs' )
   def handleContainerMgrArgs( self ):
      t0( "handleContainerMgrArgs called." )
      qt0( "handleContainerMgrArgs called." )
      self.sync()

   def close( self ):
      t0( "close for ContainerMgrConfigReactor called." )
      Tac.Notifiee.close( self )

class ContainerMgr( SuperServer.SuperServerAgent ):
   def __init__( self, entityManager ):
      SuperServer.SuperServerAgent.__init__( self, entityManager )
      mg = entityManager.mountGroup()
      self.containerMgrConfig = mg.mount( 'containerMgr/config',
                                          'ContainerMgr::ContainerMgrConfig', 'r' )
      self.containerConfig = mg.mount( 'containerMgr/container/config',
                                       'ContainerMgr::ContainerConfig', 'r' )
      self.notifiee_ = None
      self.registryNotifiee_ = None
      self.containerNotifee_ = None

      def _finished():
         if not self.active():
            return
         self.createReactors()
      mg.close( _finished )

   def createReactors( self ):
      self.notifiee_ = ContainerMgrConfigReactor( self.containerMgrConfig )
      self.registryNotifiee_ = Tac.collectionChangeReactor(
                  self.containerMgrConfig.registryConfig, RegistryReactor,
                  reactorArgs=( self, ) )
      self.containerNotifee_ = Tac.collectionChangeReactor(
            self.containerConfig.container,
            ContainerConfigReactor )

   def onSwitchover( self, protocol ):
      self.createReactors()

def Plugin( ctx ):
   ctx.registerService( ContainerMgr( ctx.entityManager ) )
