# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import SuperServer
import Tac 
import Tracing
import weakref

t0 = Tracing.t0
t8 = Tracing.t8

sessionLimitType = Tac.Type( "Mgmt::Account::SessionLimit" )
maxSessionLimit = sessionLimitType.max
unlimitedSessionLimit = sessionLimitType.unlimited
useGlobalSessionLimit = sessionLimitType.useGlobal

class AccountsConfigReactor( SuperServer.GenericService ):
   """ Reactor to "management accounts" config changes. """
   notifierTypeName = 'Mgmt::Account::Config'

   def __init__( self, serviceName, config ):
      self.config_ = config
      SuperServer.GenericService.__init__( self, serviceName, config,
                                           sync=False )
      self.userConfigReactor_ = Tac.collectionChangeReactor( 
                                    self.config_.account, 
                                    UserConfigReactor, 
                                    reactorArgs=( weakref.proxy( self ), ) )
      self.activity_ = None
      self.activityInterval_ = 60

   def warm( self ):
      return True

   def sync( self ):
      t8( "AccountsConfigReactor: starting sync" )
      if not self.writeConfigFile( "/etc/security/limits.conf", 
                                   self.limitsConf(), 
                                   writeHeader=True ):
         # Write failed, retry in a minute
         if self.activity_:
            t8( "Rescheduling timer for sync()" )
            self.activity_.timeMin = min( self.activity_.timeMin,
                                          Tac.now() + self.activityInterval_ )
         else:
            t8( "Starting timer for sync()" )
            self.activity_ = Tac.ClockNotifiee()
            self.activity_.handler = self.sync
            self.activity_.timeMin = Tac.now() + self.activityInterval_
      else:
         t8( "AccountsConfigReactor: sync finished" )

   def limitsConf( self ):
      t0( "Generating session limits config" )
      conf = ""
      accountsConf = self.config_ 
      userConf = accountsConf.account

      # Add limit entry iff limit is not limits.conf's default. 
      # When no limit is set in Limits.conf, the default "unlimited" takes effect.
      if accountsConf.sessionLimitGlobal != unlimitedSessionLimit:
         conf += "* - maxlogins %d\n" % accountsConf.sessionLimitGlobal
      for user in sorted( userConf ): 
         # add new entry only if limit is different from the global limit
         if userConf[ user ].sessionLimit != useGlobalSessionLimit:
            conf += "%s - maxlogins %d\n" % ( user, userConf[ user ].sessionLimit )
      t0( "limits conf: %s", conf )
      return conf

   @Tac.handler( "sessionLimitGlobal" )
   def handleSessionLimitGlobal( self ):
      self.sync()

class UserConfigReactor( Tac.Notifiee ):
   """ Reactor to account-specific config changes """
   notifierTypeName = 'Mgmt::Account::UserConfig'

   def __init__ ( self, userConfig, parentReactor ):
      t8( "UserConfigReactor.__init__ start for:", userConfig.name )
      self.parentReactor_ = parentReactor
      Tac.Notifiee.__init__( self, userConfig )
   
   @Tac.handler( 'sessionLimit' )
   def handleSessionLimit( self ):
      t8( "UserConfigReactor.handleSessionLimit start" )
      self.parentReactor_.sync()
      t8( "UserConfigReactor.handleSessionLimit end" )
      
class AccountManager( SuperServer.SuperServerAgent ):
   def __init__( self, entityManager ):
      SuperServer.SuperServerAgent.__init__( self, entityManager )
      mg = entityManager.mountGroup()
      self.config_ = mg.mount( 'mgmt/acct/config', 
                               'Mgmt::Account::Config', 'r' )
      self.accountsConfigReactor_ = None 
      def _finished():
         # pylint: disable-msg=W0201
         self.accountsConfigReactor_ = AccountsConfigReactor( "AccountsConfig", 
                                                              self.config_ )
      mg.close( _finished )

def Plugin( ctx ):
   ctx.registerService( AccountManager( ctx.entityManager ) )
