#!/usr/bin/env python
# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
import Tac
import Cell

class ProgressDirSm( Tac.Notifiee ):
   notifierTypeName = 'Stage::ProgressDir'

   def __init__( self, progressDir, handler ):
      self.progressDir_ = progressDir
      self.handler_ = handler
      self.progressSm_ = {}
      Tac.Notifiee.__init__( self, progressDir )

      for key in self.progressDir_:
         self.handleProgress( key )

   class ProgressSm( Tac.Notifiee ):
      notifierTypeName = 'Stage::Progress'

      def __init__( self, instance, progress, handler ):
         self.instance_ = instance
         self.progress_ = progress
         self.handler_ = handler
         Tac.Notifiee.__init__( self, progress )

         for key in self.progress_.stage:
            self.handleStage( key )

      @Tac.handler( 'stage' )
      def handleStage( self, key ):
         if key in self.progress_.stage:
            if self.progress_.stage[ key ]:
               self.handler_( self.instance_, key )

   @Tac.handler( 'progress' )
   def handleProgress( self, key ):
      if key in self.progressDir_:
         self.progressSm_[ key ] = self.ProgressSm(
            key, self.progressDir_.progress[ key ], self.handler_ )
      else:
         del self.progressSm_[ key ]

class StageHelper( object ):
   def __init__( self, entMan, stageClass ):
      self.entMan_ = entMan
      self.stageClass_ = stageClass

   def applicationIs( self, application ):
      config = self._stageConfig()
      config.application = application

   def stageTimeoutIsFatalIs( self, isFatal ):
      config = self._stageConfig()
      config.stageTimeoutIsFatal = isFatal

   def continueOnTimeoutIs( self, continueOn ):
      config = self._stageConfig()
      config.continueOnTimeout = continueOn

   def resetModeIs( self, mode ):
      config = self._stageConfig()
      config.resetMode = mode

   def reloadCauseDescIs( self, desc ):
      config = self._stageConfig()
      config.reloadCauseDesc = desc

   def registerStageDependency( self, stage, dependencies, instanceName='default' ):
      dummyAgent = "__dummyInternal1__"
      self.registerStage( dummyAgent, stage, dependencies=dependencies,
                          instanceName=instanceName, complete=True )

   def registerStage( self, agent, stage, dependencies=None,
                      timeout=None, enableAutoComplete=False,
                      autoCompleteTimeout=None, complete=None,
                      completeNotRunnable=None,
                      instanceName='default', critical=None ):
      eventConfig = self._eventConfig( agent, stage )
      self._eventDependenciesIs( eventConfig, dependencies )
      self._eventTimeoutIs( eventConfig, timeout )
      self._eventAutoCompleteIs(
         eventConfig, enableAutoComplete, autoCompleteTimeout )
      self._eventCompleteIs( eventConfig, complete, instanceName )
      self._eventCompleteNotRunnableIs( eventConfig, completeNotRunnable )
      self._eventCriticalIs( eventConfig, critical )

   def registerStageClass( self ):
      config = self._stageConfig()
      config.stageClass = self.stageClass_
      config.agentDir = self._agentConfig()
      self.applicationIs( self.stageClass_ )
      self._progressInstanceIs()

   def initializedIs( self, initialized ):
      completionStatusDir = self._completionStatusDir()
      completionStatusDir.initialized = initialized

   def _completionStatusDir( self ):
      completionStatusDirPath = Cell.path(
            'stage/%s/completionstatus' % self.stageClass_ )
      return self.entMan_.lookup( completionStatusDirPath )

   def _progressInstanceIs( self, instance='default' ):
      progressDir = self._progressDir()
      progressDir.progress.newMember( instance )

   def _progressDir( self ):
      progressDirPath = Cell.path( 'stage/%s/progress' % self.stageClass_ )
      return self.entMan_.lookup( progressDirPath )

   def _eventCriticalIs( self, eventConfig, critical ):
      if critical is not None:
         eventConfig.critical = critical

   def _eventCompleteIs( self, eventConfig, complete, instance='default' ):
      if complete is not None:
         eventConfig.complete[ instance ] = complete

   def _eventAutoCompleteIs( self, eventConfig, enable, timeout ):
      if enable:
         eventConfig.enableAutoComplete = True
         if timeout is None:
            timeout = eventConfig.autoCompleteTimeout
         eventConfig.autoCompleteTimeout = timeout
         self._eventTimeoutIs( eventConfig, timeout + 5 )

   def _eventTimeoutIs( self, eventConfig, timeout ):
      if timeout is not None:
         eventConfig.timeout = timeout

   def _eventDependenciesIs( self, eventConfig, dependencies ):
      allStages = self._allStages()

      if dependencies is None:
         dependencies = []
      for stage in dependencies:
         assert stage in allStages
         eventConfig.dependency[ stage ] = True

   def _eventCompleteNotRunnableIs( self, eventConfig, completeNotRunnable ):
      if completeNotRunnable is not None:
         eventConfig.completeNotRunnable = completeNotRunnable

   def _eventConfig( self, agent, stage ):
      agentConfig = self._agentConfig( agent )
      eventConfig = agentConfig.event.get( stage )
      if not eventConfig:
         agentConfig.newEvent( stage, agent )
         eventConfig = agentConfig.event[ stage ]
      return eventConfig

   def _agentConfig( self, agent=None ):
      agentConfigPath = Cell.path( 'stageInput/%s' % self.stageClass_ )
      if agent:
         agentConfigPath += "/%s" % agent
      return self.entMan_.lookup( agentConfigPath )

   def _stageConfig( self ):
      configPath = Cell.path( 'stage/%s/config/config' % self.stageClass_ )
      return self.entMan_.lookup( configPath )

   def _allStages( self ):
      stages = set()
      agentConfigDir = self._agentConfig()
      for agentConfig in agentConfigDir.values():
         stages |= set( agentConfig.event.keys() )
      return stages

