# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os
import Cell
import Logging
import Tac
import Tracing
import Agent
import StageMgrAgentPluginLoader
import StageHelper
import QuickTrace
from StageGraphs import registerStageGraphs

# add dep to launcher
# pkgdeps: import LauncherPlugin.Launcher

traceHandle = Tracing.Handle( 'StageMgrAgent' )
t0 = traceHandle.trace0

Logging.logD(
   id="LAUNCHER_BOOT_STATUS",
   severity=Logging.logInfo,
   format=( "'reload hitless' reconciliation complete." ),
   explanation=( "Prioritized agents participating in 'reload hitless' have "
                 "completed reconciliation." ),
   recommendedAction=Logging.NO_ACTION_REQUIRED )

class BootStatusSm( Tac.Notifiee ):
   notifierTypeName = 'Stage::Status'

   def __init__( self, bootConfig, bootStatus ):
      self.bootConfig_ = bootConfig
      self.bootStatus_ = bootStatus
      Tac.Notifiee.__init__( self, self.bootStatus_ )
      self.instStatusSm_ = None
      self.handleInstStatus( key='default' )

   class InstanceStatusSm( Tac.Notifiee ):
      notifierTypeName = 'Stage::InstanceStatus'

      def __init__( self, bootConfig, instStatus ):
         self.bootConfig_ = bootConfig
         self.instStatus_ = instStatus
         Tac.Notifiee.__init__( self, self.instStatus_ )
         self.handleComplete()

      @Tac.handler( "complete" )
      def handleComplete( self ):
         if( self.instStatus_.complete and
             self.bootConfig_.application == "Asu Hitless boot" ):
            Logging.log( LAUNCHER_BOOT_STATUS ) #pylint: disable-msg=E0602

   @Tac.handler( "instStatus" )
   def handleInstStatus( self, key=None ):
      if key != 'default':
         return
      if key in self.bootStatus_.instStatus:
         assert self.instStatusSm_ is None
         self.instStatusSm_ = self.InstanceStatusSm( self.bootConfig_,
                                 self.bootStatus_.instStatus[ key ] )
      else:
         if self.instStatusSm_:
            del self.instStatusSm_
         self.instStatusSm_ = None

class FatalErrorSm( object ):
   '''
   This class holds all the configuration needed to set up fatal error request
   handling.
   '''
   def __init__( self, entityManager, fatalErrorDir, fatalErrorGlobalConfig ):
      t0( 'FatalErrorSm.__init__ called' )
      self.entMan_ = entityManager
      self.fatalErrorDir_ = fatalErrorDir
      self.fatalErrorGlobalConfig_ = fatalErrorGlobalConfig
      self.fatalErrorDirSm_ = None

      # in future, this depends on whether the system has SSDs
      feConstants = Tac.Type( "Stage::FatalErrorConstants" )
      persistDir = feConstants.secondaryPersistentFs

      # this class is responsible for creating the fatalErrorDirSm_, and
      # the StageStatusReactor is responsible for destroying it.
      # instantiate the fatal error handler mechanism 
      if not os.getenv( 'DISABLE_FATALERRORSM' ):
         self.fatalErrorDirSm_ = Tac.newInstance(
            'Stage::FatalErrorDirSm', feConstants.syslogDir,
            feConstants.coreDir, feConstants.agentLogDir,
            persistDir, self.fatalErrorDir_, self.fatalErrorGlobalConfig_ )

class StageMgrAgent( Agent.Agent ):
   def __init__( self, entityManager ):
      self.entMan_ = entityManager
      self.stageDirReactor_ = None
      self.bootStatusSm_ = None
      self.fatalErrorSm_ = None
      self.launcherConfig_ = None
      self.bootStatus_ = None
      self.asuHwStatus_ = None
      self.persistentRestartLogSm_ = None

      Agent.Agent.__init__( self, entityManager, agentName='StageMgr' )
      if "QUICKTRACE_DISABLE" not in os.environ:
         qtfile = "%s%s.qt" % ( 'StageMgr', "-%d" if "QUICKTRACEDIR"
                  not in os.environ else "" )
         QuickTrace.initialize( qtfile )


   def doStartupSanity( self ):
      '''
      Basic starup sanity to validate if StageMgr starts and the dut has been up
      for >15min and boot stages are not complete then something could be wrong
      with the fatalErrorSm or stageMgrSm handling to invoke a cold boot. As a
      fail safe, reboot the dut directly.
      '''
      asuMode = Tac.Type( 'Asu::AsuMode' )
      if self.asuHwStatus_.asuReboot and \
         self.asuHwStatus_.asuRebootMode == asuMode.hitless:
         uptimeFile = os.environ.get( "UPTIME_FILE", "/proc/uptime" )
         uptimeInSeconds, _ = map( float, file( uptimeFile ).read().split() )
         if uptimeInSeconds > self.launcherConfig_.maxUptimeInSeconds:
            if not StageHelper.stageProgressionComplete( self.bootStatus_ ):
               # boot stages are still not complete when Launcher was started
               # after the dut has been up for maxUptime. Invoke fail safe reboot. 
               ReloadConstants = Tac.Value( "ReloadCause::ReloadCauseConstants" )
               ReloadConstants.writeLocalReloadCause(
                     ReloadConstants.failSafeRebootDesc, "", True )
               simReloadFile = os.environ.get( "SIMULATION_RELOAD_FILE" )
               if simReloadFile:
                  cmd = "touch " + simReloadFile 
                  Tac.run( cmd.split( " " ) )
               else:
                  shutdownCmd = "shutdown -r now"
                  Tac.run( shutdownCmd.split( " " ) )

   def doInit( self, entityManager ):
      t0( 'doInit' )
      mg = entityManager.mountGroup()

      stageDir = mg.mount( Cell.path( 'stage' ), 'Tac::Dir', 'fwi' )
      # Temporary: still need to mount as 'fwi' until multi-writer cleanup is
      # complete. For example, if a stage times out and auto-completes,
      # StageMgr writes the complete flag.
      stageInputDir = mg.mount( Cell.path( 'stageInput' ), 'Tac::Dir', 'fwi' )
      stageGlobalConfig = mg.mount( 'stage/stageGlobalConfig',
                                    'Stage::StageGlobalConfig', 'fri' )
      # Agents write stage complete status here and StageMgr mounts it read-only
      stageAgentStatusDir = mg.mount( Cell.path( 'stageAgentStatus' ),
            'Tac::Dir', 'ri' )
      stageEnableConfigDir = mg.mount( Cell.path( 'stageEnable' ), 'Tac::Dir', 'ri' )

      fatalErrorRequestPath = Cell.path( 'fatalError/config/request/stageMgr' )
      fatalErrorRequest = mg.mount( fatalErrorRequestPath,
                                    'Stage::FatalError', 'fcw' )

      # for fatal error reboot requests
      fatalErrorDir = mg.mount( Cell.path( 'fatalError/config/request' ), 
                                           'Tac::Dir', 'ri' )
      fatalErrorGlobalConfig = mg.mount( 'fatalError/config/globalConfig',
                                         'Stage::FatalErrorGlobalConfig', 'fri' )

      self.launcherConfig_ = mg.mount( "launcher/config",
                                       "Launcher::LauncherConfig", "r" )
      self.asuHwStatus_ = mg.mount( 'asu/hardware/status', 'Asu::AsuStatus', 'r' )
      self.persistentRestartLogSm_ = Tac.newInstance(
         'Stage::PersistentRestartLogSm' )

      def _finish():
         t0( 'Mounts complete' )

         bootConfig = stageDir.entity[ 'boot/config/config' ]
         self.bootStatus_ = stageDir.entity[ 'boot/status' ]

         # register stage graphs
         t0( 'StageMgrAgent invoking registerStageGraphs' )
         registerStageGraphs( entityManager )

         self.doStartupSanity()

         StageMgrAgentPluginLoader.loadPlugins( self.entMan_ )

         self.stageDirReactor_ = Tac.newInstance( "Stage::StageMgrStageClassDirSm",
                                                stageDir,
                                                stageInputDir,
                                                stageAgentStatusDir,
                                                stageEnableConfigDir,
                                                stageGlobalConfig,
                                                fatalErrorRequest )

         self.bootStatusSm_ = BootStatusSm( bootConfig, self.bootStatus_ )

         self.fatalErrorSm_ = FatalErrorSm( self.entMan_, fatalErrorDir,
                                            fatalErrorGlobalConfig )

      mg.close( _finish )

def main():
   tfm = Tac.singleton( 'Tac::TraceFacilityMan' )
   traceSetting = 'StageMgrAgent/0,StageMgr/0'
   if 'TRACE' in os.environ.keys():
      traceSetting = os.environ[ 'TRACE' ] + ',' + traceSetting
   os.environ[ 'TRACE' ] = traceSetting
   tfm.doReadEnvironment()

   container = Agent.AgentContainer( [ StageMgrAgent ],
                                     agentTitle="StageMgr",
                                     passiveMount=True )
   container.runAgents()

def name():
   ''' Call this to establish an explicit dependency on the StageMgr agent
   executable, to be discovered by static analysis. '''
   return 'StageMgr'
