# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os
import re
import Tac
import Tracing
import Cell
import ProductAttributes

traceHandle = Tracing.Handle( 'StageGraphs' )
t0 = traceHandle.trace0

class BootModeConfig( object ):
   def __init__( self, entityManager ):
      self.entityManager_ = entityManager
      self.bootMode_ = self._determineBootMode()

   def _determineBootMode( self ):
      bootMode = os.environ.get( "STAGE_BOOTMODE_SIM" )
      if bootMode is not None:
         return bootMode

      # Boot attributes for this system
      bootAttributes = ProductAttributes.productAttributes().bootAttributes

      # If this is a supervisor is coming up as 'standby', just use
      # 'defaultStandbyBootMode'
      if self.entityManager_.redundancyStatus().mode == 'standby':
         return bootAttributes.defaultStandbyBootMode

      asuHwStatus = self.entityManager_.lookup( 'asu/hardware/status' )
      asuMode = Tac.Type( "Asu::AsuMode" )
      if asuHwStatus.asuReboot and asuHwStatus.asuRebootMode == asuMode.fastboot:
         bootMode = bootAttributes.asuFastBootMode
      elif asuHwStatus.asuReboot and asuHwStatus.asuRebootMode == asuMode.hitless:
         bootMode = bootAttributes.asuHitlessBootMode
      else: 
         # For Strata defaultBootMode will be 'stagedBoot', for others 'normalBoot'
         bootMode = bootAttributes.defaultBootMode
      return bootMode

   def bootModeStagesIs( self, bootMode, registerFnc ):
      if bootMode == self.bootMode_:
         registerFnc( self.entityManager_ )

#
# We need to figure out the platform to pick the correct Stages.
# After an ASU/fastboot this information can be found in the prefdl.txt file.
#

def isSandSystem():
   # Get platform from prefdl
   # BUG244087: should have a better scheme than looking for product names
   platform = 'None'
   try:
      with open( '/etc/prefdl', 'r' ) as f:
         data = f.read()
   except IOError:
      return False
   m = re.search( r'.*SID: (\w+)', data )
   if m:
      platform = m.group( 1 )
      platformPrefixes = (
         'Larkfield',
         'Forestville',
         'Guerneville',
         'Hopland',
         'Graton',
      )
      if platform.startswith( platformPrefixes ):
         return True
   # Not Sand fixed System.
   return False

def getPlatformType():
   platform = None

   if os.environ.get( 'SIMULATION_VMID' ):
      # Product attributes are not populated in btests
      if os.environ.get( 'SIMULATION_STRATA' ) or os.environ.get( 'STRATA_SYSTEM' ) :
         platform = "strata"
      elif os.environ.get( 'SIMULATION_SAND' ) or os.environ.get( 'SAND_SYSTEM' ):
         platform = "sand"
      elif os.environ.get( 'SIMULATION_BFN' ) or os.environ.get( 'BFN_SYSTEM' ):
         platform = "barefoot"
   elif isSandSystem():
      # Sand does not use product attributes
      platform = "sand"
   else:
      platform = ProductAttributes.productAttributes().productAttributes.platform

   return platform


# register stage against linecard slice agents
# args:
# sh - StageSysdbHelper.StageHelper instance
# stage - stage name (string)
def registerLinecardSliceStage( sh, stage, timeout=None, complete=None ):
   bootAttributes = ProductAttributes.productAttributes().bootAttributes
   linecardSliceNames = bootAttributes.linecardSliceName
   
   for sliceName in linecardSliceNames:
      sh.registerStage( sliceName, stage, timeout=timeout, complete=complete )

# register stage against all slice (linecard + fabric) agents
# args:
# sh - StageSysdbHelper.StageHelper instance
# stage - stage name (string)
def registerAllSliceStage( sh, stage, timeout=None, complete=None ):
   bootAttributes = ProductAttributes.productAttributes().bootAttributes
   linecardSliceNames = bootAttributes.linecardSliceName
   fabricSliceNames = bootAttributes.fabricSliceName
   allSliceNames = linecardSliceNames.keys() + fabricSliceNames.keys()
   
   for sliceName in allSliceNames:
      sh.registerStage( sliceName, stage, timeout=timeout, complete=complete )

# register stage against all PhyIsland slice agents
# args:
# sh - StageSysdbHelper.StageHelper instance
# stage - stage name (string)
def registerPhyIslandStage( sh, stage, timeout=None, complete=None ):
   # BUG471964: change this to dynamic slice names
   if Cell.product() in [ "brooksIsland", "roeIsland", "birdIsland", "sealIsland" ]:
      sh.registerStage( "PhyIsland-FixedSystem", stage, timeout=timeout,
                        complete=complete )
