# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import Tracing
import pyinotify
import glob
import os
import SslCertKey
import ManagedSubprocess
import re
import shutil
import hashlib
import errno
import Logging

Logging.logD( id="SECURITY_SSL_DIFFIE_HELLMAN_PARAMETERS_INIT_FAILED",
              severity=Logging.logError,
              format=( "Initializing SSL Diffie-Hellman parameters failed. Try "
                       "'reset ssl diffie-hellman parameters' to recover" ),
              explanation=( "Attempt to initialize SSL Diffie-Hellman "
                            "parameters has failed." ),
              recommendedAction=Logging.CALL_SUPPORT_IF_PERSISTS )

Logging.logD( id="SECURITY_SSL_DIFFIE_HELLMAN_PARAMETERS_RESET_FAILED",
              severity=Logging.logError,
              format=( "Resetting SSL Diffie-Hellman parameters failed. "
                       "Existing parameters will be used" ),
              explanation=( "Attempt to reset SSL Diffie-Hellman parameters "
                            "has failed. Existing Diffie-Hellman parameters "
                            "will be used." ),
              recommendedAction=Logging.CALL_SUPPORT_IF_PERSISTS )

Logging.logD( id="SECURITY_SSL_PROFILE_INVALID",
              severity=Logging.logError,
              format=( "SSL profile '%s' is invalid. "
                       "Check 'show management security ssl profile %s' "
                       "for details" ),
              explanation=( "The SSL profile is invalid. Possible causes "
                            "are missing certificate, missing key, mismatch "
                            "of certificate with the key, expired or not yet "
                            "valid certificate and non root certificate in the "
                            "trusted list." ),
              recommendedAction=( "Check 'show management security ssl profile' "
                                  "command and fix the errors reported under "
                                  "'Error' column." ) )

Logging.logD( id="SECURITY_SSL_PROFILE_VALID",
              severity=Logging.logInfo,
              format="SSL profile '%s' is valid",
              explanation="The SSL profile is valid.",
              recommendedAction=Logging.NO_ACTION_REQUIRED )
             
traceHandle = Tracing.Handle( 'MgmtSecuritySslReactor' )
error = traceHandle.trace0
warn  = traceHandle.trace1
info  = traceHandle.trace2
trace = traceHandle.trace3
debug = traceHandle.trace4

CertKeyPair = Tac.Type( "Mgmt::Security::Ssl::CertKeyPair" )
CertificateInfo = Tac.Type( "Mgmt::Security::Ssl::CertificateInfo" )
ProfileState = Tac.Type( "Mgmt::Security::Ssl::ProfileState" )
ErrorType = Tac.Type( "Mgmt::Security::Ssl::ErrorType" )
ErrorAttr = Tac.Type( "Mgmt::Security::Ssl::ErrorAttr" )
ProfileError = Tac.Type( "Mgmt::Security::Ssl::ProfileError" )

defaultDhParams = (
      "-----BEGIN DH PARAMETERS-----\n"
      "MIIBCAKCAQEAtTPLn9zRROp7KqTMpI5ATDMZKypv668/h/WjDYtA7OlxpQXPxIVY\n"
      "zfa+j2uBnhG+jnRsSV7kmZARrAo/bUK/fsI1pw/PCc4W4jOi9lOObfX7dQuNpPD9\n"
      "teEPAmzHMPNOjZvF3WPDZaj/M2FbkE/8lgcFDjVovNBfzFFSwXN6JkD3l1PJMgx/\n"
      "HYPAzdnL9Hm9+V6mSBJeiN4K76wVq3DmEv/unzxYT9uX683FrMh9Ti1Zmuke7x2e\n"
      "nYPvmxFMVM3tx8EXjZqGtcvV0BE13AkG5KyW8zG8L/tDyv/z99iJ7YoreGNyfRaz\n"
      "kvArMBA2Nvakdhe4xlWLR2JhG5NDJAhYQwIBAg==\n"
      "-----END DH PARAMETERS-----" )

dhParamSize = str( 2048 )

class FileEventHandler( pyinotify.ProcessEvent ):
   def __init__ ( self, fileReactor ):
      trace( "FileEventHandler.__init__ start" )
      self.fileReactor_ = fileReactor 
      pyinotify.ProcessEvent.__init__( self )
      trace( "FileEventHandler.__init__ end" )
       
   def process_IN_MOVED_TO( self, event ):
      self.fileReactor_.handleFile( event.wd, event.name )
       
   # MOVED_FROM happnes in two contexts
   # 1. When copying file using tempfile, rename machanism
   # 2. When using rename command
   # In both cases we send delete event of the file. If its
   # case 1, then nothing happens as none of the ssl profiles
   # will be using tempfile.
   def process_IN_MOVED_FROM( self, event ):
      self.fileReactor_.handleFileDelete( event.wd, event.name )

   def process_IN_DELETE( self, event ):
      self.fileReactor_.handleFileDelete( event.wd, event.name )

        
class InotifyReactor( Tac.Notifiee, pyinotify.Notifier ):
   notifierTypeName = 'Tac::FileDescriptor'

   def __init__( self, watch_manager, default_proc_fun=None, read_freq=0,
                threshold=0, timeout=None ):
      
      pyinotify.Notifier.__init__( self, watch_manager, default_proc_fun, 
                                   read_freq, threshold, timeout )
      fileDesc_ = Tac.newInstance( 'Tac::FileDescriptor', "ssl" )
      fileDesc_.descriptor = watch_manager.get_fd()
      fileDesc_.nonBlocking = True
      Tac.Notifiee.__init__( self, fileDesc_ )
     
   @Tac.handler( 'readableCount' )
   def handleReadableCount( self ):
      self.read_events()
      self.process_events()

class FileReactor( object ):
   def __init__( self, constants, sslReactor ):
      trace( "FileReactor.__init__ start" )
      self.sslReactor_ = sslReactor
      self.constants_ = constants
      wm = pyinotify.WatchManager()
      # pylint: disable-msg=E1101
      mask =  pyinotify.IN_MOVED_FROM | pyinotify.IN_MOVED_TO | pyinotify.IN_DELETE
      self.inotifyReactor = InotifyReactor( wm, FileEventHandler( self ) ) 
      wd = wm.add_watch( self.constants_.certsDirPath(), mask )
      self.certsWd_ = wd[ self.constants_.certsDirPath() ] 
      wd = wm.add_watch( self.constants_.keysDirPath(), mask )
      self.keysWd_ = wd[ self.constants_.keysDirPath() ]
      wd = wm.add_watch( self.constants_.rotationBaseDirPath(),
                         pyinotify.IN_DELETE | pyinotify.IN_MOVED_FROM )
      self.rotDirsWd_ = wd[ self.constants_.rotationBaseDirPath() ]
      debug( "Certs WatchDescriptor: ", self.certsWd_ )
      debug( "Keys WatchDescriptor: ", self.keysWd_ )
      debug( "Rotation dirs WatchDescriptor: ", self.rotDirsWd_ )
      trace( "FileReactor.__init__ end" )

   def handleFile( self, wd, fileName ):
      trace( "FileReactor.handleFile start for:", fileName, "WD:", wd )
      if wd == self.certsWd_:
         debug( "Certificate added/modified" )
         self.sslReactor_.handleCertOrCrl( fileName )
      else:
         debug( "Key added/modified" )
         self.sslReactor_.handleKey( fileName )
      trace( "FileReactor.handleFile end for: ", fileName, "WD:", wd )

   def handleFileDelete( self, wd, fileName ):
      trace( "FileReactor.handleFileDelete start for:", fileName, "WD:", wd )
      if wd == self.certsWd_:
         debug( "Certificate deleted" )
         self.sslReactor_.handleCertOrCrl( fileName )
      elif wd == self.keysWd_:
         debug( "Key deleted" )
         self.sslReactor_.handleKey( fileName )
      else:
         debug( "Rotation dir deleted" )
         self.sslReactor_.handleRotationDir( fileName )
      trace( "FileReactor.handleFileDelete end for:", fileName, "WD:", wd )

class ExecRequestReactor( Tac.Notifiee ):
   notifierTypeName = 'Mgmt::Security::Ssl::ExecRequest'
   def __init__( self, execRequest, sslReactor ):
      trace( "ExecRequestReactor.__init__ start" )
      self.execRequest_ = execRequest
      self.sslReactor_ = sslReactor
      Tac.Notifiee.__init__( self, self.execRequest_ ) 
      trace( "ExecRequestReactor.__init__ end" )
   
   @Tac.handler( 'dhparamsResetRequest' )
   def handleDhparamsResetRequest( self ):
      trace( "ExecRequestReactor.handleDhparamsResetRequest start" )
      self.sslReactor_.handleDhparamsResetRequest()
      trace( "ExecRequestReactor.handleDhparamsResetRequest end" )

class NetStatusReactor( Tac.Notifiee ):
   # pkgdeps: rpm NetConfig
   notifierTypeName = 'System::NetStatus'

   def __init__( self, netStatus, profileConfig, sslReactor ):
      trace( "NetStatusReactor.__init__ start" )
      self.profileConfig_ = profileConfig
      self.sslReactor_ = sslReactor
      self.netStatus_ = netStatus
      Tac.Notifiee.__init__( self, self.netStatus_ )

   @Tac.handler( 'hostname' )
   def handleHostname( self ):
      trace( "NetStatusReactor.handleHostname start" )
      for profileName in self.profileConfig_:
         self.sslReactor_.handleProfileChange( profileName )
      trace( "NetStatusReactor.handleHostname end" )

class ProfileConfigReactor( Tac.Notifiee ):
   notifierTypeName = 'Mgmt::Security::Ssl::ProfileConfig'
   def __init__ ( self, profileConfig, sslReactor ):
      trace( "ProfileConfigReactor.__init__ start for:", profileConfig.name )
      self.profileConfig_ = profileConfig
      self.sslReactor_ = sslReactor
      self.dateCheck = Tac.ClockNotifiee( self.handleDateCheck,
                                          timeMin=Tac.endOfTime )
      Tac.Notifiee.__init__( self, self.profileConfig_ )
      if ( self.profileConfig_.certKeyPair.certFile or 
           self.profileConfig_.trustedCert or
           self.profileConfig_.chainedCert or
           self.profileConfig_.crl ):
         trace( "ProfileConfigReactor.__init__ non empty profileConfig" )
         self.sslReactor_.handleProfileChange( self.profileConfig_.name, 
                                               profileConfigReactor=self )
      trace( "ProfileConfigReactor.__init__ end for:", profileConfig.name )
   
   @Tac.handler( 'tlsVersion' )
   def handleTlsVersion( self ):
      trace( "ProfileConfigReactor.handleTlsVersion start" )
      self.sslReactor_.handleProfileChange( self.profileConfig_.name )
      trace( "ProfileConfigReactor.handleTlsVersion end" )

   @Tac.handler( 'fipsMode' )
   def handleFipsMode( self ):
      trace( "ProfileConfigReactor.handleFipsMode start" )
      self.sslReactor_.handleProfileChange( self.profileConfig_.name )
      trace( "ProfileConfigReactor.handleFipsMode end" )

   @Tac.handler( 'cipherSuite' )
   def handleCipherSuite( self ):
      trace( "ProfileConfigReactor.handleCipherSuite start" )
      self.sslReactor_.handleProfileChange( self.profileConfig_.name )
      trace( "ProfileConfigReactor.handleCipherSuite end" )

   @Tac.handler( 'certKeyPair' )
   def handleCertKeyPair( self ):
      trace( "ProfileConfigReactor.handleCertKeyPair start" )
      self.sslReactor_.handleProfileChange( self.profileConfig_.name )
      trace( "ProfileConfigReactor.handleCertKeyPair end" )
   
   @Tac.handler( 'trustedCert' )
   def handleTrustedCert( self, certName ):
      trace( "ProfileConfigReactor.handleTrustedCert start" )
      self.sslReactor_.handleProfileChange( self.profileConfig_.name )
      trace( "ProfileConfigReactor.handleTrustedCert end" )

   @Tac.handler( 'chainedCert' )
   def handleChainedCert( self, certName ):
      trace( "ProfileConfigReactor.handleChainedCert start" )
      self.sslReactor_.handleProfileChange( self.profileConfig_.name )
      trace( "ProfileConfigReactor.handleChainedCert end" )

   @Tac.handler( 'crl' )
   def handleCrl( self, crlName ):
      trace( "ProfileConfigReactor.handleCrl start" )
      self.sslReactor_.handleProfileChange( self.profileConfig_.name )
      trace( "ProfileConfigReactor.handleCrl end" )
   
   @Tac.handler( 'verifyExtendedParameters' )
   def handleVerifyExtendedParameters( self ):
      trace( "ProfileConfigReactor.handleVerifyExtendedParameters start" )
      self.sslReactor_.handleProfileChange( self.profileConfig_.name )
      trace( "ProfileConfigReactor.handleVerifyExtendedParameters end" )

   @Tac.handler( 'verifyChainHasRootCA' )
   def handleVerifyChainHasRootCA( self ):
      trace( "ProfileConfigReactor.handleVerifyChainHasRootCA start" )
      self.sslReactor_.handleProfileChange( self.profileConfig_.name )
      trace( "ProfileConfigReactor.handleVerifyChainHasRootCA end" )

   @Tac.handler( 'verifyBasicConstraintTrust' )
   def handleVerifyBasicConstraintTrust( self ):
      trace( "ProfileConfigReactor.handleVerifyBasicConstraintTrust start" )
      self.sslReactor_.handleProfileChange( self.profileConfig_.name )
      trace( "ProfileConfigReactor.handleVerifyBasicConstraintTrust end" )

   @Tac.handler( 'verifyBasicConstraintChain' )
   def handleVerifyBasicConstraintChain( self ):
      trace( "ProfileConfigReactor.handleVerifyBasicConstraintChain start" )
      self.sslReactor_.handleProfileChange( self.profileConfig_.name )
      trace( "ProfileConfigReactor.handleVerifyBasicConstraintChain end" )

   @Tac.handler( 'verifyHostnameMatch' )
   def handleVerifyHostnameMatch( self ):
      trace( "ProfileConfigReactor.handleVerifyHostnameMatch start" )
      self.sslReactor_.handleProfileChange( self.profileConfig_.name )
      trace( "ProfileConfigReactor.handleVerifyHostnameMatch end" )

   @Tac.handler( 'verifyExpiryDateEndCert' )
   def handleVerifyExpiryDateEndCert( self ):
      trace( "ProfileConfigReactor.handleVerifyExpiryDateEndCert start" )
      self.sslReactor_.handleProfileChange( self.profileConfig_.name )
      trace( "ProfileConfigReactor.handleVerifyExpiryDateEndCert end" )

   def handleDateCheck( self ):
      trace( "ProfileConfigReactor.handleDateCheck start" )
      self.sslReactor_.handleProfileChange( self.profileConfig_.name )
      trace( "ProfileConfigReactor.handleDateCheck end" )
      
class ConfigReactor ( Tac.Notifiee ):
   notifierTypeName = 'Mgmt::Security::Ssl::Config'
   def __init__( self, config, status, sslReactor ):
      trace( "ConfigReactor.__init__ start" )
      self.config_ = config
      self.sslReactor_ = sslReactor
      self.status_ = status
      Tac.Notifiee.__init__( self, self.config_ )
      for profileName in self.config_.profileConfig:
         self.handleProfileConfig( profileName )
      for profileName in status.profileStatus:
         if profileName not in self.config_.profileConfig:
            self.handleProfileConfig( profileName )
      trace( "ConfigReactor.__init__ end" )
   
   @Tac.handler( 'profileConfig' )
   def handleProfileConfig( self, profileName ):
      trace( "ConfigReactor.handleProfileConfig start for: ", profileName )
      if profileName in self.config_.profileConfig:
         info( "profile added:", profileName )
         self.sslReactor_.handleProfileAdd( profileName )
      else:
         info( "profile deleted:", profileName )
         self.sslReactor_.handleProfileDelete( profileName )
      trace( "ConfigReactor.handleProfileConfig end for: ", profileName )

class Dhparams ( object ):
   def __init__( self, constants, sslReactor ):
      trace( "Dhparams.__init__ start" )
      self.constants_ = constants
      self.sslReactor_ = sslReactor
      self.tmpDhParam_ = self.constants_.baseDir + "dhparam.tmp"
      self.dhParamSubProcess_ = None
      self.dhparamsPoller_ = None
      trace( "Dhparams.__init__ end" )
   
   def initialized( self ):
      return os.path.isfile( self.constants_.dhParamPath() )

   def returnSignalingSubprocess( self, success ):
      """
      Return a signalling subprocess for the dhParam generation
      that will exit or succeed instantly for edge cases.
      """
      if success:
         exitCode = "0"
      else:
         exitCode = "1"
      subProc = ManagedSubprocess.Popen( [ "exit", exitCode ],
            stdout=open( '/dev/null', 'w' ),
            stderr=ManagedSubprocess.STDOUT )
      return subProc
   
   def reset( self, init=False, retries=3 ):
      trace( "Dhparams.reset start" )

      try:
         os.remove( self.tmpDhParam_ )
      except OSError:
         pass

      if self.sslReactor_.execRequest_.dhparamTestMode:
         # Decrease dhparam size to a still reasonable, but smaller
         # size if in test mode
         global dhParamSize
         dhParamSize = str( 1024 )
   
      debug( "Generating dhparams in tmp file:", self.tmpDhParam_ )
      debug( "Generating dhparam of size %s bits" % dhParamSize )
      if os.path.isdir( self.tmpDhParam_ ):
         # Fail quickly if there is a folder here. Used to speed up breadth
         # tests
         self.dhParamSubProcess_ = self.returnSignalingSubprocess( success=False )
      elif init:
         trace( "Writing default dhparams file instead of generating new one" )
         try:
            with open( self.tmpDhParam_, "w" ) as fileHandle:
               fileHandle.write( defaultDhParams )
         # Use the subprocess as a simple way to signal success or failure
         except IOError, err:
            error( "Error writing default params: " + str( err ) )
            self.dhParamSubProcess_ = self.returnSignalingSubprocess( success=False )
         else:
            self.dhParamSubProcess_ = self.returnSignalingSubprocess( success=True )
      else:
         self.dhParamSubProcess_ = ManagedSubprocess.Popen( [ "openssl", "dhparam",
            "-outform", "PEM", "-out", self.tmpDhParam_ , dhParamSize ],
            stdout=open( '/dev/null', 'w' ),
            stderr=ManagedSubprocess.STDOUT )

      self.dhparamsPoller_ = Tac.Poller(
            lambda: self.dhParamSubProcess_.wait( block=False ) != None,
            handler=lambda ignored: self._resetDone( retries ),
            timeoutHandler=lambda: self._resetDone( retries, timedOut=True ),
            timeout=900, description="scheduled dhparams reset" )
   
   def _resetDone( self, retries, timedOut=False ):
      err = None
      if timedOut :
         trace( "Generation of dhparams taking more than 15 mins" )
         self.dhParamSubProcess_.kill()
         err = "Timed out while generating dhparams"
      elif not self._validateDhparams():
         error( "Error generating dhparams" )
         err = "Error generating dhparams"
   
      if not err:
         os.rename( self.tmpDhParam_, self.constants_.dhParamPath() )
      elif retries:
         retries = retries - 1
         trace( 'Retrying generating dhparams' )
         self.reset( retries=retries )
         return
      else:
         trace( 'Retries exhausted for generating dhparams' )
                

      self.dhParamSubProcess_ = None
      self.dhparamsPoller_ = None
      self.sslReactor_.handleDhparamsResetResult( err )

   def _validateDhparams( self ):
      trace( "Dhparams._validateDhparams start" )
      if not os.path.isfile( self.tmpDhParam_ ):
         warn( "File", self.tmpDhParam_, "does not exist" )
         return False
      output = Tac.run( [ 'openssl', 'dhparam', '-in', self.tmpDhParam_, '-check', 
                         '-noout' ], asRoot=True, stdout=Tac.CAPTURE, 
                       stderr=Tac.CAPTURE, ignoreReturnCode=True )
      debug( '_validateDhparams ', output )
      return re.match( '.* ok.', output )
   
class SslReactor( object ):
   def __init__( self, config, status, execRequest, netStatus ):
      trace( "SslReactor.__init__ start" )
      self.config_ = config
      self.status_ = status
      self.caCerts = {}   
      self.execRequest_ = execRequest
      self.netStatus_ = netStatus
      self.constants_ = Tac.Type( "Mgmt::Security::Ssl::Constants" )
      
      SslCertKey.createSslDirs()
      
      self.fileReactor = FileReactor( self.constants_, self )
      self.configReactor = ConfigReactor( self.config_, self.status_, self )
      self.profileConfigReactor_ = Tac.collectionChangeReactor(
                                       self.config_.profileConfig,
                                       ProfileConfigReactor,
                                       reactorArgs=( self, ) )
      self.netStatusReactor_ = NetStatusReactor( self.netStatus_,
                                                 self.config_.profileConfig,
                                                 self )
      self.execRequestReactor_ = ExecRequestReactor( self.execRequest_, self )
      self.dhparams_ = Dhparams( self.constants_, self )
      if not self.dhparams_.initialized():
         info( "Generating initial dhparams" )
         self.handleDhparamsResetRequest( init=True )
      trace( "Reactor.__init__ end" )
      
   def _isProfileUsingCertOrCrl( self, profileName, certName ):
      profileConfig = self.config_.profileConfig[ profileName ]
      if ( profileConfig.certKeyPair.certFile == certName or
           certName in profileConfig.trustedCert or
           certName in profileConfig.chainedCert or
           certName in profileConfig.crl ):
         return True
      else:
         return False

   def _checkCertOrCrlInProfile( self, certName ):
      for profileName in self.config_.profileConfig:
         if self._isProfileUsingCertOrCrl( profileName, certName ):
            debug( "Profile", profileName, "is using cert", certName )
            self.handleProfileChange( profileName )

   def _checkKeyInProfile( self, keyName ):
      for profileName in self.config_.profileConfig:
         profileConfig = self.config_.profileConfig[ profileName ]
         if profileConfig.certKeyPair.keyFile == keyName:
            debug( "Profile", profileName, "is using key", keyName )
            self.handleProfileChange( profileName )

   def _readFile( self, filePath ):
      buf = ""
      try:
         with open( filePath , 'r' ) as fp:
            buf = fp.read()
      except IOError as e:
         if e.errno != errno.ENOENT:
            error( "Cannot open file", filePath,
                   "errno", e.errno  )
            raise
         else:
            buf = ""
      return buf
   
   def handleCertOrCrl ( self, certName ):
      trace( "SslReactor.handleCertOrCrl start for:", certName )
      self._checkCertOrCrlInProfile( certName )
      trace( "SslReactor.handleCertOrCrl end for:", certName )
   
   def handleKey ( self, keyName ):
      trace( "SslReactor.handleKey start for:", keyName )
      self._checkKeyInProfile( keyName )
      trace( "SslReactor.handleKey end for:", keyName )

   def handleRotationDir( self, rotationDir ):
      trace( "SslReactor.handleRotationDir start for:", rotationDir )
      match = re.match( r'[^.]+\.(.*)\.commit', rotationDir )
      if match and match.group( 1 ) in self.config_.profileConfig:
         self.handleProfileChange( match.group( 1 ) )
      trace( "SslReactor.handleRotationDir end for:", rotationDir )

   def _same( self, buf1, buf2 ):
      # pylint: disable-msg=E1101
      d1 = hashlib.md5( buf1 ).hexdigest()
      # pylint: disable-msg=E1101      
      d2 = hashlib.md5( buf2 ).hexdigest()
      trace( "buf1 hash:", d1 )
      trace( "buf2 hash:", d2 )
      return d1 == d2

   def _getCertKeyStr( self, keyData, profileConfig, certDict ):
      certList = ( [ profileConfig.certKeyPair.certFile ] +
                   list( profileConfig.chainedCert ) )
      certListData = map( certDict.get, certList )
      return '\n'.join( [ keyData ] + certListData )
   
   def _getTrustedCertsStr( self, profileConfig, certDict ):
      certsDataList = map( certDict.get, list( profileConfig.trustedCert ) )
      return '\n'.join( certsDataList )

   def _getCrlsStr( self, profileConfig, crlDict ):
      crlsDataList = map( crlDict.get, list( profileConfig.crl ) )
      return '\n'.join( crlsDataList )
   
   def _isServerCert( self, profileConfig, certDict ):
      cert = profileConfig.certKeyPair.certFile
      if cert:
         return SslCertKey.isServerCert( certDict[ cert ] )
      else:
         return False
   
   def _isClientCert( self, profileConfig, certDict ):
      cert = profileConfig.certKeyPair.certFile
      if cert:
         return SslCertKey.isClientCert( certDict[ cert ] )
      else:
         return False

   def _getCertInfo( self, profileConfig, certDict ):
      cert = profileConfig.certKeyPair.certFile
      if not cert:
         return CertificateInfo( '', 0, 0 )
      return SslCertKey.getCertInfo( certDict[ cert ] )

   def isCommitInProgress( self, profileName ):
      return glob.glob( self.constants_.rotationBaseDirPath() +
                        '*.{}.commit'.format( profileName ) )

   def _checkCertificates( self, profileConfig, certDict, keyData ):
      cert = profileConfig.certKeyPair.certFile
      key = profileConfig.certKeyPair.keyFile
      errList = []
      warningList = []
      update = False

      def _checkCertKeyPair():
         if cert not in certDict:
            error( "Certificate does not exist:", cert )
            errList.append( ProfileError( ErrorAttr.certificate,
                                          cert,
                                          ErrorType.notExist ) )
         else:
            errType = SslCertKey.validateCertificateData( certDict[ cert ],
                  validateExtended=profileConfig.verifyExtendedParameters,
                  validateCa=False,
                  validateExpiryDate=profileConfig.verifyExpiryDateEndCert,
                  isTrust=False,
                  validateHostname=True
                  )
            if errType != ErrorType.noError:
               if errType == ErrorType.hostnameMismatch and \
                     not profileConfig.verifyHostnameMatch:
                  warningList.append( ProfileError( ErrorAttr.certificate,
                                                    cert,
                                                    errType ) )
               else:
                  error( "Certificate", cert, "is", errType )
                  errList.append( ProfileError( ErrorAttr.certificate,
                                                cert,
                                                errType ) )
         if not keyData:
            error( "Key does not exist:", key )
            errList.append( ProfileError( ErrorAttr.key,
                                          key,
                                          ErrorType.notExist ) )
         # Either certificate or key does not exist.
         # no need to validate. Return from here
         if cert not in certDict or not keyData:
            return
         if not SslCertKey.isCertificateMatchesKey( certDict[ cert ], keyData ):
            error( "Certificate does not match with key" )
            errList.append( ProfileError( ErrorAttr.certificate,
                                          cert,
                                          ErrorType.notMatchingCertKey ) )

      def _checkChainedCert():
         if ( not cert or cert not in certDict or
              not SslCertKey.verifyCertChain( cert,
                                              profileConfig.chainedCert,
                                              certDict,
                                              profileConfig.verifyChainHasRootCA ) ):
            errList.append( ProfileError( ErrorAttr.profile,
                                          profileConfig.name,
                                          ErrorType.certChainNotValid ))
         for c in profileConfig.chainedCert:
            if c != profileConfig.certKeyPair.certFile:
               if c not in certDict:
                  error( "Chained certificate", c, "does not exist" )
                  errList.append( ProfileError( ErrorAttr.chainedCertificate,
                                                c,
                                                ErrorType.notExist ) )
               else:
                  errType = SslCertKey.validateCertificateData( certDict[ c ],
                        validateExtended=False,
                        validateCa=profileConfig.verifyBasicConstraintChain,
                        validateExpiryDate=True,
                        isTrust=False,
                        )
                  if errType != ErrorType.noError:
                     error( "Chained certificate", c, "is", errType )
                     errList.append( ProfileError( ErrorAttr.chainedCertificate,
                                                   c,
                                                   errType ) )

      if not cert and not profileConfig.chainedCert:
         return ( [], False, [] )

      if cert:
         _checkCertKeyPair()

      if profileConfig.chainedCert or profileConfig.verifyChainHasRootCA:
         _checkChainedCert()

      if not len( errList ):
         oldData = self._readFile( self.constants_.certKeyPath(
                                      profileConfig.name ) )
         newData = self._getCertKeyStr( keyData, profileConfig, certDict )
         trace( "Comparing old certkey file with new one" )
         update = not self._same( oldData, newData )
      return ( errList, update, warningList )

   def _checkTrustedCert( self, profileConfig, certDict, certKeyErrList, crlDict ):
      errList = []
      update = False
      for cert in profileConfig.trustedCert:
         if cert != profileConfig.certKeyPair.certFile:
            if cert not in certDict:
               error( "Trusted certificate", cert, "does not exist" )
               errList.append( ProfileError( 
                                          ErrorAttr.trustedCertificate,
                                          cert,
                                          ErrorType.notExist ) )
            else:
               errType = SslCertKey.validateCertificateData( certDict[ cert ],
                     validateExtended=False,
                     validateCa=profileConfig.verifyBasicConstraintTrust,
                     validateExpiryDate=True,
                     isTrust=True,
                     )
               if errType != ErrorType.noError:
                  error( "Trusted certificate", cert, "is", errType )          
                  errList.append( ProfileError( ErrorAttr.trustedCertificate,
                                                cert,
                                                errType ) )
         
      if not self._isEmpty( profileConfig ):
         errType =  SslCertKey.validateTrustedChain( profileConfig, 
                                                     certDict, crlDict )
         if errType != ErrorType.noError:
            error( "Profile", profileConfig.name, "is", errType )
            errList.append( ProfileError( ErrorAttr.profile,
                                          profileConfig.name,
                                          errType ) )
               
      if not certKeyErrList and not errList:
         oldData = self._readFile( self.constants_.trustedCertsPath( 
                                                   profileConfig.name ) )
         newData = self._getTrustedCertsStr( profileConfig, certDict )
         trace( "Comparing old trusted certs file with new one" )            
         update = not self._same( oldData, newData )
         for cert in profileConfig.trustedCert:
            self.caCerts.setdefault(\
                  profileConfig.name, [] ).append( certDict[ cert ] )
 
      return ( errList, update )

   def _checkCrl( self, profileConfig, crlDict, prevErrList ):
      errList = []
      update = False
      for crl in profileConfig.crl:
         if crl not in crlDict:
            error( "CRL", crl, "does not exist" )
            errList.append( ProfileError( ErrorAttr.crl,
                                          crl,
                                          ErrorType.notExist ) )
         else:
            errType = SslCertKey.validateCertificateData( crlDict[ crl ],
                  validateExtended=profileConfig.verifyExtendedParameters,
                  validateCa=False,
                  validateExpiryDate=True )
            if errType == ErrorType.noError:
               errType = SslCertKey.validateCrlCa( crlDict[ crl ], 
                                          self.caCerts,
                                          profileConfig.name,
                                          profileConfig.verifyExtendedParameters )
            if errType != ErrorType.noError:
               error( "CRL", crl, "is", errType )
               errList.append( ProfileError( ErrorAttr.crl,
                                             crl,
                                             errType ) )
      if not len( prevErrList ) and not len( errList ):
         oldData = self._readFile( self.constants_.crlsPath( 
                                                   profileConfig.name ) )
         newData = self._getCrlsStr( profileConfig, crlDict )
         trace( "Comparing old CRL file with new one" )
         update = not self._same( oldData, newData )

      return ( errList, update )

   def _catTrustedCert( self, profileConfig, certDict ):
      trustedCertsFile = self.constants_.trustedCertsPath( profileConfig.name )
      tmpFile = trustedCertsFile + ".tmp"
      with open( tmpFile, 'w' ) as tmpFp:
         trustedCertsStr = self._getTrustedCertsStr( profileConfig, certDict )
         tmpFp.write( trustedCertsStr )
      os.rename( tmpFile, trustedCertsFile )

   def _catCrls( self, profileConfig, crlDict ):
      crlFile = self.constants_.crlsPath( profileConfig.name )
      tmpFile = crlFile + ".tmp"
      with open( tmpFile, 'w' ) as tmpFp:
         crlsStr = self._getCrlsStr( profileConfig, crlDict )
         tmpFp.write( crlsStr )
      os.rename( tmpFile, crlFile )
   
   def _catCertKey( self, profileConfig, certDict, key ):
      certKeyFile = self.constants_.certKeyPath( profileConfig.name )
      tmpFile = certKeyFile + ".tmp"
      oldmask = os.umask( 0006 )
      try:
         with open( tmpFile, 'w' ) as tmpFp:
            certKeyStr = self._getCertKeyStr( key, profileConfig, certDict )
            tmpFp.write( certKeyStr )
         os.rename( tmpFile, certKeyFile )
      except:
         raise
      finally:
         os.umask( oldmask )

   def handleProfileAdd( self, profileName ):
      trace( "SslReactor.handleProfileAdd start for: ", profileName )
      if not os.path.isdir( self.constants_.profileDirPath( profileName ) ):
         SslCertKey.dirCreate( self.constants_.profileDirPath( profileName ) )
      if profileName in self.status_.profileStatus:
         if not os.path.isdir( self.constants_.profileDirPath( profileName ) ):
            SslCertKey.dirCreate( self.constants_.profileDirPath( profileName ) )
         debug( "ProfileStatus already exists for:", profileName )
      else:
         debug( "Adding new ProfileStatus for:", profileName )
         profileStatus = self.status_.newProfileStatus( profileName )
         profileStatus.error.enq( ProfileError( ErrorAttr.profile,
                                                profileName,
                                                ErrorType.noProfileData ) )
      trace( "SslReactor.handleProfileAdd end for: ", profileName )
      
   def _isEmpty( self, profileConfig ):
      return ( not profileConfig.certKeyPair.certFile and
               len( profileConfig.trustedCert ) == 0 and
               len( profileConfig.chainedCert ) == 0 and
               len( profileConfig.crl ) == 0 )
   
   def _getCertDict( self, profileConfig ):
      certDict = { }
      if profileConfig.certKeyPair.certFile:
         buf = self._readFile( self.constants_.certPath( 
                                    profileConfig.certKeyPair.certFile ) )
         if buf:
            certDict[ profileConfig.certKeyPair.certFile ] = buf 
      
      for cert in ( list( profileConfig.trustedCert ) +
                    list( profileConfig.chainedCert ) ):
         buf = self._readFile( self.constants_.certPath( cert ) )
         if buf and SslCertKey.hasCertificate( buf ):
            certDict[ cert ] = buf
      return certDict

   def _getCrlDict( self, profileConfig ):
      crlDict = { }
      for crl in list( profileConfig.crl ):
         buf = self._readFile( self.constants_.certPath( crl ) )
         if buf and SslCertKey.hasCrl( buf ):
            crlDict[ crl ] = buf
      return crlDict
   
   def _getKey( self, profileConfig ):
      buf = ""
      if profileConfig.certKeyPair.certFile:
         buf = self._readFile( self.constants_.keyPath( 
                                    profileConfig.certKeyPair.keyFile ) )
      else:
         buf = ""
      return buf

   def _getEarliestTimeCheck( self, certDict, crlDict ):
      minimum = None
      now = int( Tac.utcNow() )
      trace( "UTC now is", now )
      for cert, certData in certDict.items() + crlDict.items():
         ( nb, na ) = SslCertKey.getCertificateOrCrlDates( certData )
         trace( "Certificate", cert, "nb, na:", nb, na )
         if ( nb >= now ):
            if ( minimum == None or nb < minimum ):
               minimum = nb
         elif ( na >= now ):
            if ( minimum == None or na < minimum ):
               minimum = na
      if minimum != None:
         return ( minimum - now )
      else:
         return None 

   def _isEmptyProfileStatus( self, profileStatus ):
      if profileStatus.state != ProfileState.invalid:
         return False
      if len( profileStatus.error.values() ) > 1:
         return False
      err = profileStatus.error.values()[ 0 ]
      if err.errorType != ErrorType.noProfileData:
         return False
      return True

   def handleProfileChange( self, profileName, profileConfigReactor=None ):
      if self.isCommitInProgress( profileName ):
         trace( "Commit in progress for:", profileName )
         return

      trace( "SslReactor.handleProfileChange start for:", profileName )
      profileConfig = self.config_.profileConfig[ profileName ]
      profileStatus = self.status_.profileStatus[ profileName ]
      errList = []
      warningList = []
      certDict = self._getCertDict( profileConfig )
      crlDict = self._getCrlDict( profileConfig )
      key = self._getKey( profileConfig )
      oldState = profileStatus.state
      oldEmptyProfile = self._isEmptyProfileStatus( profileStatus )

      trace( "certDict keys are", certDict.keys() )
      trace( "crlDict keys are", crlDict.keys() )

      if not profileConfigReactor:
         profileConfigReactor = self.profileConfigReactor_.reactors_[ profileName ]
         trace( "config reactors:", self.profileConfigReactor_.reactors_ )
      else:
         trace( "Profile config reatcor passed as arg" )

      seconds = self._getEarliestTimeCheck( certDict, crlDict )
      if seconds != None:
         trace( "Earliest time check is after", seconds, "seconds" )
         profileConfigReactor.dateCheck.timeMin = Tac.now() + seconds
      else:
         trace( "Resetting time check timer to end of time" )
         profileConfigReactor.dateCheck.timeMin = Tac.endOfTime

      errCk, updateCk, warningCk = self._checkCertificates( profileConfig,
                                                            certDict, key )
      errList.extend( errCk )
      warningList.extend( warningCk )
      errTc, updateTc = self._checkTrustedCert( profileConfig, certDict,
                                                errList, crlDict )
      errList.extend( errTc )
      ( errCrl, updateCrl ) = self._checkCrl( profileConfig, crlDict, errList )
      errList.extend( errCrl )

      if not errList:
         if self._isEmpty( profileConfig ):
            errList.append( ProfileError( ErrorAttr.profile,
                                          profileName,
                                          ErrorType.noProfileData ))
            newState = ProfileState.invalid
         else:
            newState = ProfileState.valid
      else:
         newState = ProfileState.invalid

      def updateRequired():
         if ( profileStatus.state != newState ):
            return True
         if ( newState == ProfileState.invalid ):
            for e in errList:
               if e not in profileStatus.error.values():
                  return True
            if len( errList ) != len( profileStatus.error.values() ):
               return True
         if ( newState == ProfileState.valid ):
            if profileConfig.certKeyPair.certFile:
               newCertKeyPath = self.constants_.certKeyPath( profileConfig.name )
            else:
               newCertKeyPath = ""
            if len( profileConfig.trustedCert ) > 0:
               newTrustedCertsPath = self.constants_.trustedCertsPath( 
                                                            profileConfig.name )
            else:
               newTrustedCertsPath = ""
            if len( profileConfig.crl ) > 0:
               newCrlsPath = self.constants_.crlsPath( profileConfig.name )
            else:
               newCrlsPath = ""
            if ( profileStatus.certKeyPath !=  newCertKeyPath or
                 profileStatus.trustedCertsPath != newTrustedCertsPath or
                 profileStatus.crlsPath != newCrlsPath or
                 profileStatus.tlsVersion != profileConfig.tlsVersion or
                 profileStatus.fipsMode != profileConfig.fipsMode or
                 profileStatus.cipherSuite != profileConfig.cipherSuite or
                 updateCk or updateTc or updateCrl or
                 profileStatus.verifyExtendedParameters != \
                       profileConfig.verifyExtendedParameters ):
               return True
            if len( errList ) != len( profileStatus.error.values() ):
               return True
            if len( warningList ) != len( profileStatus.warning.values() ):
               return True
         return False

      if ( updateRequired() ):
         profileStatus.state = ProfileState.updating
         if ( newState == ProfileState.invalid ):
            trace( "Updating profile status to invalid" )
            profileStatus.error.clear()
            profileStatus.warning.clear()
            profileStatus.tlsVersion = self.constants_.allTlsVersion
            profileStatus.fipsMode = False
            profileStatus.cipherSuite = self.constants_.defaultCipherSuite()
            profileStatus.certKeyPath = ""
            profileStatus.chainedCerts = False
            profileStatus.trustedCertsPath = ""
            profileStatus.crlsPath = ""
            profileStatus.certInfo = CertificateInfo( "", 0, 0 )
            profileStatus.verifyExtendedParameters = False
            profileStatus.isServerCert = True
            profileStatus.isClientCert = True
            for e in errList:
               profileStatus.error.enq( e )
         else:
            trace( "Updating profile status to valid" )
            profileStatus.error.clear()
            profileStatus.warning.clear()
            profileStatus.chainedCerts = len( profileConfig.chainedCert ) > 0
            profileStatus.tlsVersion = profileConfig.tlsVersion
            profileStatus.fipsMode = profileConfig.fipsMode
            profileStatus.cipherSuite = profileConfig.cipherSuite
            if profileConfig.certKeyPair.certFile:
               self._catCertKey( profileConfig, certDict, key )
               profileStatus.certKeyPath = self.constants_.certKeyPath(
                                                            profileConfig.name )
            else:
               profileStatus.certKeyPath = ""
            if len( profileConfig.trustedCert ) > 0:
               self._catTrustedCert( profileConfig, certDict )
               profileStatus.trustedCertsPath = self.constants_.trustedCertsPath( 
                                                               profileConfig.name )
            else:
               profileStatus.trustedCertsPath = ""
            if len( profileConfig.crl ) > 0:
               self._catCrls( profileConfig, crlDict )
               profileStatus.crlsPath = self.constants_.crlsPath(
                                            profileConfig.name )
            else:
               profileStatus.crlsPath = ""
            profileStatus.certInfo = self._getCertInfo( profileConfig,
                                                        certDict )
            profileStatus.isServerCert = self._isServerCert( profileConfig,
                                                             certDict )
            profileStatus.isClientCert = self._isClientCert( profileConfig,
                                                             certDict )
            profileStatus.verifyExtendedParameters = \
                  profileConfig.verifyExtendedParameters

         for warning in warningList:
            trace( "Add warnings to profile status" )
            profileStatus.warning.enq( warning )

         profileStatus.state = newState

      # Log the state change messages
      if oldState != profileStatus.state:
         if profileStatus.state == ProfileState.invalid:
            if not self._isEmptyProfileStatus( profileStatus ):
               # pylint: disable-msg=E0602
               Logging.log( SECURITY_SSL_PROFILE_INVALID,
                            profileName, profileName )
         else:
            trace( "Logging invalid->valid state change" )
            # pylint: disable-msg=E0602
            Logging.log( SECURITY_SSL_PROFILE_VALID, profileName )
      else:
         if oldEmptyProfile:
            # pylint: disable-msg=E0602
            Logging.log( SECURITY_SSL_PROFILE_INVALID,
                         profileName, profileName )

      for err in profileStatus.error.values():
         debug( "Error: ", err.errorAttr, err.errorAttrValue, 
                 err.errorType )         

      trace( "SslReactor.handleProfileChange end for: ", profileName )

   def handleProfileDelete( self, profileName ):
      trace( "SslReactor.handleSslProfileDelete start for: ", profileName )
      if hasattr( self, 'profileConfigReactor_' ):
         debug( "On delete config reactors:", 
                self.profileConfigReactor_.reactors_ )
         profileConfigReactor = self.profileConfigReactor_.reactors_[ profileName ]
         profileConfigReactor.dateCheck.timeMin = Tac.endOfTime
      if profileName in self.status_.profileStatus:
         debug( "Deleting ProfileStatus for:", profileName )
         shutil.rmtree( self.constants_.profileDirPath( profileName ), 
                        ignore_errors=True )
         del self.status_.profileStatus[ profileName ]
         self.caCerts.pop( profileName, None )
      trace( "SslReactor.handleSslProfileDelete end for: ", profileName )
      
   def handleDhparamsResetRequest( self, init=False ):
      trace( "SslReactor.handleDhparamsResetRequest start" )
      if self.status_.dhparamsResetInProgress:
         trace( "Dhparams reset is already in progress" )
         return
      if ( init or self.execRequest_.dhparamsResetRequest > 
           self.status_.dhparamsResetProcessed ):
         self.status_.dhparamsResetInProgress = True
         self.dhparams_.reset( init=init )
      else:
         trace( "Ignoring reset request" )
      trace( "SslReactor.handleDhparamsResetRequest end" )
      
   def handleDhparamsResetResult( self, err=None ):
      trace( "SslReactor.handleDhparamsResetResult start:", err )
      if not err:
         self.status_.dhparamsResetProcessed = Tac.now()   
         self.status_.dhparamsLastResetFailed = False   
      else:
         error( "Dhparams reset error:", err )
         self.status_.dhparamsLastResetFailed = True         
         if not self.dhparams_.initialized():
            # pylint: disable-msg=E0602
            Logging.log( SECURITY_SSL_DIFFIE_HELLMAN_PARAMETERS_INIT_FAILED )
         else:
            # pylint: disable-msg=E0602
            Logging.log( SECURITY_SSL_DIFFIE_HELLMAN_PARAMETERS_RESET_FAILED )
      self.status_.dhparamsResetInProgress = False
      trace( "SslReactor.handleDhparamsResetResult end:", err )
