# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import datetime
from dateutil import parser
import errno
import grp
import os
import pwd
import re
import tempfile
import time
import hashlib
import Tac
import Tracing
import Logging
import socket

traceHandle = Tracing.Handle( 'MgmtSecuritySslCertKey' )
error = traceHandle.trace0
warn  = traceHandle.trace1
info  = traceHandle.trace2
trace = traceHandle.trace3
debug = traceHandle.trace4

ErrorType = Tac.Type( "Mgmt::Security::Ssl::ErrorType" )

Constants = Tac.Type( "Mgmt::Security::Ssl::Constants" )
CertificateInfo = Tac.Type( "Mgmt::Security::Ssl::CertificateInfo" )
PublicKeyAlgorithm = Tac.Type( "Mgmt::Security::Ssl::PublicKeyAlgorithm" )

# See RFC3279 section 2.3
PUBKEY_ALGO_TO_ENUM = { 'rsaEncryption': PublicKeyAlgorithm.RSA,
                        'id-ecPublicKey': PublicKeyAlgorithm.ECDSA,
                      }

Logging.logD( id="SECURITY_SSL_KEY_CERT_CREATED",
              severity=Logging.logInfo,
              format="SSL %s %s has been created with the %s hash of %s%s.",
              explanation="The SSL private key, certificate or certificate signing "
                          "request has been created. This can happen when a SSL "
                          "private key or certificate is generated in the system.",
              recommendedAction=Logging.NO_ACTION_REQUIRED )
Logging.logD( id="SECURITY_SSL_KEY_CERT_UPDATED",
              severity=Logging.logInfo,
              format="SSL %s %s has been updated with the %s hash of %s%s from %s.",
              explanation="The SSL private key, certificate or certificate signing "
                          "request has been updated. This can happen when an "
                          "existing SSL private key or certificate is updated "
                          "in the system.",
              recommendedAction=Logging.NO_ACTION_REQUIRED )
Logging.logD( id="SECURITY_SSL_KEY_CERT_DELETED",
              severity=Logging.logInfo,
              format="SSL %s %s has been deleted with the previous %s hash of %s%s.",
              explanation="The SSL private key, certificate or certificate signing "
                          "has been deleted. This can happen when a SSL private key "
                          "or certificate is deleted from the system.",
              recommendedAction=Logging.NO_ACTION_REQUIRED )
Logging.logD( id="SECURITY_SSL_KEY_CERT_IMPORTED",
              severity=Logging.logInfo,
              format="SSL %s %s has been imported with the %s hash of %s%s.",
              explanation="The SSL private key or certificate has been imported. "
                          "This can happen when a SSL private key or cerficiate "
                          "is installed in the system.",
              recommendedAction=Logging.NO_ACTION_REQUIRED )

def dirCreate( path ):
   trace( "dirCreate start for: ", path )
   try:
      os.makedirs( path )
   except OSError as e:
      if e.errno == errno.EEXIST:
         trace( "Directory already exists: ", path )
      else:
         error( "Cannot create directory: ", path, "errno: ", e.strerror )
         raise

   uid = pwd.getpwnam( Constants.sslDirOwner ).pw_uid
   gid = grp.getgrnam( Constants.sslDirGroup ).gr_gid
   os.chown( path, uid, gid )
   os.chmod( path, Constants.sslDirPerm )

def createSslDirs():
   dirCreate( Constants.baseDir )
   dirCreate( Constants.certsDirPath() )
   dirCreate( Constants.keysDirPath() )
   dirCreate( Constants.profileBaseDirPath() )
   dirCreate( Constants.rotationBaseDirPath() )

def isSslDirsCreated():
   # rotationBaseDirPath is the last one to be created. Only
   # check if this dir is created.
   lastDir = Constants.rotationBaseDirPath()
   try:
      if not os.path.isdir( lastDir ):
         trace( "Dir not present", lastDir )
         return False

      uid = pwd.getpwnam( Constants.sslDirOwner ).pw_uid
      gid = grp.getgrnam( Constants.sslDirGroup ).gr_gid
      statInfo = os.stat( lastDir )
      
      trace( "statInfo uid:gid:mode", statInfo.st_uid, 
             statInfo.st_gid, oct( statInfo.st_mode ) )
      
      if statInfo.st_uid != uid or statInfo.st_gid != gid:
         return False
      
      if statInfo.st_mode & 07777 != Constants.sslDirPerm:
         return False
      return True
   except EnvironmentError as e:
      trace( "isSslDirsCreated error:", str( e ) )
      return False

def getAllPem( pem, isFile=True ):
   if isFile:
      with open( pem, 'r' ) as fp:
         filetext = fp.read()
   else:
      filetext = pem
   matches = re.findall( "-----BEGIN.*?-----.*?-----END.*?-----", filetext, 
                         flags=re.S )
   debug( "Matches are:", matches )
   return matches

def _getPemCount( pem, isFile=True ):
   trace( "_getPemCount start:", pem, "isFile:", isFile )
   matches = getAllPem( pem, isFile=isFile )
   trace( "_getPemCount end:", pem )
   return len( matches )

def _parseDates( *dates ):
   # pytz imports zipfile which is a memory hog modul blacklisted
   # by Cli. Since CliPlugin imports SslCertKey, conditionally import
   # pytz
   import pytz
   epochDt = datetime.datetime( 1970, 1, 1, tzinfo=pytz.utc )
   def parseDate( date ):
      dt = parser.parse( date, tzinfos={ 'GMT' : pytz.utc } )
      return int( ( dt - epochDt ).total_seconds() )
   return map( parseDate, dates )

def getCertificateDates( certData ):
   output = Tac.run( [ 'openssl', 'x509', '-dates', '-noout' ],
                     stdout=Tac.CAPTURE, stderr=Tac.CAPTURE,
                     ignoreReturnCode=True, input=certData )
   matchObj = re.match( r'notBefore=(.*?)\s*notAfter=(.*)', output )
   if matchObj is None:
      raise SslCertKeyError( "Can't get certificate dates" )
   return _parseDates( matchObj.group( 1 ), matchObj.group( 2 ) )

def getCrlDates( crlData ):
   output = Tac.run( [ 'openssl', 'crl', '-lastupdate', '-nextupdate',
                       '-noout' ], 
                   stdout=Tac.CAPTURE, stderr=Tac.CAPTURE,
                   ignoreReturnCode=True, input=crlData )
   matchObj = re.match( r'lastUpdate=(.*?)\s*nextUpdate=(.*)', output )
   if matchObj is None:
      raise SslCertKeyError( "Can't get certificate dates" )
   return _parseDates( matchObj.group( 1 ), matchObj.group( 2 ) )

def getCertificateOrCrlDates( certData ):
   if hasCrl( certData ):
      return getCrlDates( certData )
   return getCertificateDates( certData )

def validateCertificateData( certData,
                             validateExtended=False,
                             validateCa=False,
                             validateStartDate=True,
                             validateExpiryDate=True,
                             isTrust=False,
                             validateHostname=False ):
   now = int( Tac.utcNow() )
   ( nb, na ) = getCertificateOrCrlDates( certData )
   if ( nb > now ) and validateStartDate:
      return ErrorType.certNotYetValid
   elif ( na <= now ) and validateExpiryDate:
      return ErrorType.certExpired
   if hasCertificate( certData ):
      if validateExtended:
         trace( "validating extended key usage" )
         errorType = validateExtendedKeyUsage( certData )
         if errorType != ErrorType.noError:
            return errorType
      if validateCa:
         errorType = validateBasicConstraint( certData, isTrust )
         if errorType != ErrorType.noError:
            return errorType
      if validateHostname:
         # this check needs to be the last one
         # because it could just be a warning if hostname verification is not on
         # in this case, we don't want to have it return before other real errors
         errorType = validateHostnameMatch( certData )
         if errorType is not ErrorType.noError:
            return errorType
   return ErrorType.noError

def validateCrlCa( crlData, caCerts, profileName, validateExtended=False ):
   from M2Crypto import X509, m2
   if validateExtended:
      crl = X509.load_crl_string( crlData ) 
      if profileName in caCerts:
         for caCert in caCerts[ profileName ]:
            cert = X509.load_cert_string( removeTrusted( caCert, isFile=False ) )
            crlIssuer = crl.get_issuer().as_hash()
            caSubject = cert.get_subject().as_hash()
            if crlIssuer == caSubject:
               if cert.check_purpose( m2.X509_PURPOSE_CRL_SIGN, 0 ):
                  return ErrorType.noError
               else:
                  return ErrorType.noCrlSign
      return ErrorType.crlNotSignedByCa
   return ErrorType.noError

def isClientCert( certData ):
   from M2Crypto import X509, m2
   cert = X509.load_cert_string( removeTrusted( certData, isFile=False ) )
   return ( cert.check_purpose( m2.X509_PURPOSE_SSL_CLIENT, 0 ) == 1 )

def isServerCert( certData ):
   from M2Crypto import X509, m2
   cert = X509.load_cert_string( removeTrusted( certData, isFile=False ) )
   return ( cert.check_purpose( m2.X509_PURPOSE_SSL_SERVER, 0 ) == 1 )

def getCertInfo( certData ):
   ( nb, na ) = getCertificateOrCrlDates( certData )
   return CertificateInfo( certData, nb, na )

def validateExtendedKeyUsage( certData ):
   output = Tac.run( [ 'openssl', 'x509', '-text' ],
                     stdout=Tac.CAPTURE, stderr=Tac.CAPTURE,
                     ignoreReturnCode=True, input=certData )
   if "X509v3 Extended Key Usage:" in output:
      return ErrorType.noError
   else:
      return ErrorType.noExtendedKeyUsage

def validateBasicConstraint( certData, isTrust ):
   from M2Crypto import X509
   cert = X509.load_cert_string( removeTrusted( certData, isFile=False ) )
   if cert.check_ca() != 1:
      if isTrust:
         return ErrorType.noCABasicConstraintTrust
      else:
         return ErrorType.noCABasicConstraintChain
   return ErrorType.noError

def validateHostnameMatch( certData ):
   from M2Crypto import X509, m2
   hostname = socket.gethostname()
   trace( "Hostname is", hostname )
   cert = X509.load_cert_string( removeTrusted( certData, isFile=False ) )
   for commonName in cert.get_subject().get_entries_by_nid( m2.NID_commonName ):
      trace( "Common name is", commonName )
      if hostname == commonName.get_data().as_text():
         return ErrorType.noError
   try:
      hostStr = ":" + hostname
      sanStr = cert.get_ext( "subjectAltName" ).get_value()
      trace( "Subject alt names are", sanStr )
      if ( hostStr + "," ) in sanStr or sanStr.endswith( hostStr ):
         return ErrorType.noError
   except LookupError:
      trace( "Certificate does not use subjectAltName" )
   return ErrorType.hostnameMismatch

def removeTrusted( cert, isFile=True ):
   if isFile:
      with open( cert, 'r' ) as fp:
         certData = fp.read()
   else:
      certData = cert
   return certData.replace( "TRUSTED CERTIFICATE", "CERTIFICATE" )

def hasCrl( buf ):
   return 'X509 CRL' in buf

def hasCertificate( buf ):
   return 'CERTIFICATE' in buf

def validateCertificateOrCrl( filename ):
   with open( filename ) as f:
      buf = f.read()
      if hasCrl( buf ):
         validateCrl( filename )
      elif hasCertificate( buf ):
         validateCertificate( filename )
      else:
         raise SslCertKeyError( "Invalid certificate or CRL" )

def validateCrl( crlFile ):
   """ Validates the following about the CRL:
   1) Whether its a valid crl file.
   3) If there is exactly one PEM entity in the file.
   """ 
   from M2Crypto import X509
   trace ( "validateCrl start:", crlFile )
   try:
      X509.load_crl( crlFile )
   except X509.X509Error as e:
      error( "validateCrl exception:", e )
      raise SslCertKeyError( "Invalid CRL" )
   pemCount = _getPemCount( crlFile )
   debug( "PEM tags count is", pemCount )
   if pemCount > 1:
      error( "validateCrl: crl ", crlFile, "has", pemCount, "PEM tags" )
      raise SslCertKeyError( "Multiple PEM entities in single file not supported" )

def validateCertificate( cert, isFile=True, validateDates=False, maxPemCount=1 ):
   """ Validates the following about certificate:
   1) Whether its a valid certificate file.
   2) Whether the certificate has RSA or ECDSA key (DSA is not supported).
   3) If there is exactly <maxPemCount> PEM entity in the file. 
      None means by the check
   """ 
   from M2Crypto import X509
   trace ( "validateCertificate start:", cert )
   try:
      # M2Crypto cannot handle TRUSTED 
      # in pem header. So remove TRUSTED before
      # giving it to M2Crypto
      X509.load_cert_string( removeTrusted( cert, isFile=isFile ) )
   except X509.X509Error as e:
      error( "validateCertificate exception:", e )
      raise SslCertKeyError( "Invalid certificate" )

   if maxPemCount is not None:
      pemCount = _getPemCount( cert, isFile=isFile )
      debug( "PEM tags count is", pemCount )
      if pemCount > maxPemCount:
         error( "validateCertificate: cert", cert, "has", pemCount, "PEM tags" )
         raise SslCertKeyError( "Multiple PEM entities in single file not "
                                "supported" )

   try:
      if isFile:
         with open( cert, 'r' ) as fp:
            certData = fp.read()
      else:
         certData = cert

      pubKeyAlgo = getCertPublicKeyAlgo( certData )
      if pubKeyAlgo == PublicKeyAlgorithm.UNSUPPORTED:
         raise SslCertKeyError( "Certificate does not have supported key"
                                " (RSA, ECDSA)" )

      if validateDates:
         result = validateCertificateData( certData ) 
         if result == ErrorType.certNotYetValid:
            raise SslCertKeyError( "Certificate is not yet valid" )
         elif result == ErrorType.certExpired:
            raise SslCertKeyError( "Certificate has expired" )
   except IOError:
      raise SslCertKeyError( "Unable to read Certificate" )

def validateRsaPrivateKey( key, isFile=True ):
   """ Validates if the key has a valid RSA key 
   which is not password protected and it has exactly 
   one PEM entity.
   """ 
   from M2Crypto import RSA, util
   trace( "validateRsaPrivateKey start:", key )
   try:
      if isFile:
         RSA.load_key( key, callback=util.no_passphrase_callback )
      else:
         RSA.load_key_string( key, callback=util.no_passphrase_callback )
   except RSA.RSAError as e:
      error( "validateRsaPrivateKey exception:", e )
      if str( e ) == "bad password read":                                  
         raise SslCertKeyError( "Password protected keys are not supported" )
      else:
         raise SslCertKeyError( "Invalid RSA key" )   
         
   pemCount = _getPemCount( key, isFile=isFile )
   debug( "PEM tags count is", pemCount )
   if pemCount > 1:
      error( "validateRsaPrivateKey: key", key, "has", pemCount, "PEM tags" )
      raise SslCertKeyError( "Multiple PEM entities in single file not supported" )

def validateCSR( csrFile ):
   """ Validates if the csrFile has a valid CSR and 
   it has exactly one PEM entity.
   """ 
   from M2Crypto import X509
   trace( "validateCSR start:", csrFile )
   try:
      X509.load_request( csrFile )
   except ( ValueError, X509.X509Error ) as e:
      error( "validateCSR exception:", str( e ) )
      raise SslCertKeyError( "Invalid CSR" )
   
   pemCount = _getPemCount( csrFile )
   debug( "PEM tags count is", pemCount )
   if pemCount > 1:
      error( "validateCSR: ", csrFile, "has", pemCount, "PEM tags" )
      raise SslCertKeyError( "Multiple PEM entities in single file not supported" )

def isCertificateMatchesKey( certData, keyData ):
   from M2Crypto import X509, RSA, m2
   cert = X509.load_cert_string( removeTrusted( certData, isFile=False ) )
   rsa = RSA.load_key_string( keyData )
   try:
      certModulus = cert.get_pubkey().get_modulus()
   except ValueError:
      # We accept ECDSA certs, but not keys. ECDSA certs don't support get_modulus
      return False
   ( _, m ) = rsa.pub()
   keyModulus = m2.bn_to_hex( m2.mpi_to_bn( m ) )
   return str( certModulus ) == str( keyModulus )

def verifyCertCorrect( cert, issuerCert ):
   """
   Checks that the cert was signed by the issuerCert and
   that the cert is within the current valid time period.
   """
   if not cert.verify( issuerCert.get_pubkey() ):
      error( "Cert failed signature check" )
      return False
   try:
      certNotBefore, certNotAfter = getCertificateDates( cert.as_pem() )
   except SslCertKeyError:
      # If parsing fails for whatever reason, assume the cert is bad.
      error( "Failed to parse cert date" )
      return False
   timeNow = time.time()
   if not( certNotBefore < timeNow < certNotAfter ):
      error( "Cert was not in valid time" )
      return False
   return True

def verifyCertChain( certFile, chainedCert, certDict, verifyEndsInRootCA=False ):
   from M2Crypto import X509
   trace ( "verifyCertChain start:", certFile, chainedCert, 
           " verifyEndsInRootCA:", verifyEndsInRootCA )
   if certFile not in certDict:
      return False
   cert = X509.load_cert_string( removeTrusted( certDict[ certFile ], False ) )
   chainDict = {}
   for c in chainedCert:
      if c not in certDict:
         return False
      x509Cert = X509.load_cert_string( removeTrusted( certDict[ c ], False ) )
      chainDict[ x509Cert.get_subject().as_hash() ] = x509Cert
   for __ in chainedCert:
      try:
         issuer = chainDict[ cert.get_issuer().as_hash() ]
      except KeyError:
         # The chain is not complete yet.
         return False
      if not verifyCertCorrect( cert, issuer ):
         error( "verifyCertCorrect: cert", cert.get_subject().as_text(),
                "failed check by", cert.get_issuer().as_text() )
         return False
      cert = issuer
   if verifyEndsInRootCA and ( not verifyCertCorrect( cert, cert ) ):
      error( "Failed to verify final cert as self-signed" )
      return False
   return True

def validateTrustedChain( profileConfig, certDict, crlDict ):
   from M2Crypto import X509
   trustedCert = profileConfig.trustedCert
   configuredTrustedCert = []
   for crl in profileConfig.crl:
      if crl not in crlDict:
         # Return with noError and let _checkCrl in SslReactor handle this
         return ErrorType.noError 
   for certName in trustedCert:
      if certName not in certDict:
         # This has been handle in _checkTrustedCert in SslReactor
         # Return with noError
         return ErrorType.noError
      configuredTrustedCert.append( certName )
   validatedTrustedCert = {}
   
   def hasIssuedCrl( cert ):
      # Check if a given cert has issued any configured CRL
      for crlName in profileConfig.crl:
         crl = X509.load_crl_string( crlDict[ crlName ] )
         if ( crl.get_issuer().as_hash() == cert.get_subject().as_hash() ):
            return True
      return False
       
   # Find all self-signed certs and mark them if they've issued any CRL
   for certName in trustedCert:
      cert = X509.load_cert_string( removeTrusted( certDict[ certName ], 
                                                   isFile=False ) )
      if ( cert.get_subject().as_hash() == cert.get_issuer().as_hash() ):
         configuredTrustedCert.remove( certName )
         validatedTrustedCert[ certName ] = hasIssuedCrl( cert )
   if trustedCert and len( validatedTrustedCert ) == 0:
      return ErrorType.certTrustChainNotValid
   noChangesMade = False
   # Find all certs that are signed by validated trusted certs
   while not noChangesMade:
      noChangesMade = True
      curConfiguredTrustedCert = configuredTrustedCert[:]
      for certName in curConfiguredTrustedCert:
         cert = X509.load_cert_string( removeTrusted( certDict[ certName ], 
                                                      isFile=False ) )
         for validatedCertName, issuerHasCrl in validatedTrustedCert.items():
            validatedCert = X509.load_cert_string( \
                  removeTrusted( certDict[ validatedCertName ], 
                                 isFile=False ) )
            if ( validatedCert.get_subject().as_hash() ==\
                  cert.get_issuer().as_hash() ):
               certHasCrl = hasIssuedCrl( cert )
               if issuerHasCrl == certHasCrl:
                  configuredTrustedCert.remove( certName )
                  validatedTrustedCert[ certName ] = certHasCrl
                  noChangesMade = False
                  break
               else:
                  error( "Cert ", certName, " has not signed any CRL" )
                  return ErrorType.missingCrlForTrustChain
   if len( configuredTrustedCert ) == 0:
      return ErrorType.noError
   else:
      return ErrorType.certTrustChainNotValid

def isSelfSignedRootCertificate( certData ):
   from M2Crypto import X509
   cert = X509.load_cert_string( removeTrusted( certData, isFile=False ) )
   return ( str( cert.get_subject() ) == str( cert.get_issuer() ) )

def generateRsaPrivateKey( keyFilepath, keyBits, useTmpFile=True ):
   """
   Generates new RSA private key in keyFilepath
   """
   trace( "generateRsaPrivateKey start:", keyFilepath, ":", str( keyBits ) )
   tempKeyFilepath = None
   keyFilepath = os.path.abspath( keyFilepath )   
   
   if useTmpFile:
      keyDir = os.path.dirname( keyFilepath )
      try:
         tempKeyHandle, tempKeyFilepath = tempfile.mkstemp( dir=keyDir )
      except ( OSError, IOError ) as e:
         raise SslCertKeyError( "%s" % e.strerror )
      
      # Close file handle since we will no longer write to it directly in this code
      os.close( tempKeyHandle )
      os.chmod( tempKeyFilepath, Constants.sslKeyPerm )
   else:
      tempKeyFilepath = keyFilepath
   
   # Always use FIPS mode for this
   cmd = [ "openssl", "--fips", "genrsa", "-out", tempKeyFilepath, str( keyBits ) ]
   output = Tac.run( cmd, stdout=Tac.CAPTURE, 
                     stderr=Tac.CAPTURE, asRoot=True,
                     ignoreReturnCode=True )
   debug( "genrsa output:", output )

   validateRsaPrivateKey( tempKeyFilepath )

   if useTmpFile:
      # Rename to final name since key seems correct
      os.rename( tempKeyFilepath, keyFilepath )

def _genOpensslConf( signRequest=False, xKeyUsage=None, 
                     sanIp=None, sanDns=None, sanEmailAddress=None ):

   sanIpList = [ "IP:%s" % x for x in sanIp ] if sanIp else []
   sanDnsList = [ "DNS:%s" % x for x in sanDns ] if sanDns else []
   sanEmailList = [ "email:%s" % x for x in sanEmailAddress ]\
                  if sanEmailAddress else []
   sanValue = ",".join( sanIpList + sanDnsList + sanEmailList )

   cnf = ""
   cnf += "[req]\n"
   cnf += "distinguished_name=req_distinguished_name\n"
   cnf += "%s_extensions=v3_ext\n" % ( "req" if signRequest else "x509" ) 
   cnf += "[req_distinguished_name]\n"
   cnf += "[v3_ext]\n"
   cnf += "subjectAltName=%s\n" % sanValue if sanValue else ""
   cnf += "extendedKeyUsage=%s\n" % xKeyUsage if xKeyUsage else ""
   return cnf
   
def generateCertificate( commonName="self.signed",
                         keyFilepath=None, 
                         certFilepath=None,
                         signRequest=False,
                         genNewKey=True,
                         newKeyBits=2048,
                         digest="sha256",
                         country=None,
                         state=None,
                         locality=None,
                         orgName=None,
                         orgUnitName=None,
                         emailAddress=None,
                         sanIp=None,
                         sanDns=None,
                         sanEmailAddress=None,
                         xKeyUsage=None,
                         validity=30000 ):
   """Generates either a certificate request or a self-signed x509 certificate.
   Returns the certificate as a (certificate, key) string pair. If an error occurs,
   throws SslCertKeyError with the msg attribute set to error text.
   
   If genNewKey is set, a new key is generated.
   If genNewKey is not set, keyFilepath must be supplied.
   If genNewKey and keyFilepath are set, the new key will be saved to keyFilepath.
   If certFilepath is set, the cert or CSR will be saved to certFilepath.
   """
   debug( "locals:", locals() )
   
   assert commonName
   if genNewKey:
      assert newKeyBits > 0
   else:
      assert keyFilepath and os.path.isfile( keyFilepath )

   keyFilepath = os.path.abspath( keyFilepath ) if keyFilepath else None
   certFilepath = os.path.abspath( certFilepath ) if certFilepath else None
   certDir = os.path.dirname( certFilepath ) if certFilepath else None
   keyDir = os.path.dirname( keyFilepath ) if keyFilepath else None
   
   tmpCertFile = tempfile.NamedTemporaryFile( dir=certDir, delete=False )
   os.chmod( tmpCertFile.name, Constants.sslCertPerm )
   tmpKeyFile = tempfile.NamedTemporaryFile( dir=keyDir, delete=False )
   os.chmod( tmpKeyFile.name, Constants.sslKeyPerm )
   confFile = tempfile.NamedTemporaryFile()
   conf = _genOpensslConf( signRequest=signRequest, 
                           xKeyUsage=xKeyUsage,
                           sanIp=sanIp, sanDns=sanDns, 
                           sanEmailAddress=sanEmailAddress )
   debug( "openssl conf:" )
   debug( "\n" + conf )
   confFile.write( conf )
   confFile.flush()
   errorOccured = False

   try:
      if genNewKey:
         trace( "Generating new RSA key" )
         generateRsaPrivateKey( tmpKeyFile.name, newKeyBits, useTmpFile=False )
      else:
         trace( "Using RSA key from", keyFilepath )
         with open( keyFilepath,'r' ) as f:
            tmpKeyFile.write( f.read() )
            tmpKeyFile.flush()
      
      def paramStr( param ):
         return "" if not param else param

      subjStr = "/C={country}/ST={state}/L={locality}/O={org}"
      subjStr += "/OU={orgUnit}/CN={commonName}/emailAddress={email}/"
      subjStr = subjStr.format( country=paramStr( country ), 
                                state=paramStr( state ), 
                                locality=paramStr( locality ),
                                org=paramStr( orgName ),
                                orgUnit=paramStr( orgUnitName ), 
                                commonName=paramStr( commonName ),
                                email=paramStr( emailAddress ) )

      cmd = [ "openssl", "--fips", "req", "-new", "-%s" % digest, 
              "-subj", subjStr, "-key", "%s" % tmpKeyFile.name, 
              "-out", "%s" % tmpCertFile.name, "-config", 
              "%s" % confFile.name ]
      if not signRequest:
         cmd += [ "-x509", "-days", "%d" % validity ]
      
      trace( "Openssl cmd is", " ".join( cmd ) )

      output = Tac.run( cmd, stdout=Tac.CAPTURE, stderr=Tac.CAPTURE,
                        ignoreReturnCode=True )
      debug( "Openssl output is:" )
      debug( "\n" + output )
      
      if signRequest:
         validateCSR( tmpCertFile.name ) 
      else:
         validateCertificate( tmpCertFile.name )

      return ( tmpCertFile.read(), tmpKeyFile.read() )
   except SslCertKeyError as e:
      errorOccured = True
      error( "SslCertKeyError:", str( e ) )
      raise SslCertKeyError( str( e ) )
   except StandardError as e:
      errorOccured = True
      error( "StandardError: ", str( e ) )
      raise
   finally:
      tmpCertFile.close()
      tmpKeyFile.close()
      confFile.close()
      
      if errorOccured:
         os.remove( tmpCertFile.name )
         os.remove( tmpKeyFile.name )
      else:
         if certFilepath:
            os.rename( tmpCertFile.name, certFilepath )
         else:
            os.remove( tmpCertFile.name )
            
         if keyFilepath and genNewKey:
            os.rename( tmpKeyFile.name, keyFilepath )
         else:
            os.remove( tmpKeyFile.name )

def getCertPem( certFilepath ):
   cmd = [ "openssl", "x509", "-in", certFilepath ]
   return Tac.run( cmd, stdout=Tac.CAPTURE, stderr=Tac.CAPTURE,
                   ignoreReturnCode=True )

class SslCertKeyError( Exception ):
   pass

def getCertPublicKeyAlgo( certData ):
   cmd = [ "openssl", "x509", "-text", "-noout" ]
   output = Tac.run( cmd, stdout=Tac.CAPTURE, stderr=Tac.CAPTURE,
                     ignoreReturnCode=True, input=certData )
   matchObj = re.search( r'Public Key Algorithm: (.*)', output )
   if matchObj is None:
      raise SslCertKeyError( "Can't get public key algorithm of certificate" )
   pubKeyAlgo = matchObj.group( 1 )
   debug( "Public key algorithm is ", pubKeyAlgo )
   return PUBKEY_ALGO_TO_ENUM.get( pubKeyAlgo, PublicKeyAlgorithm.UNSUPPORTED )

def getCertPublicKeySize( certData ):
   cmd = [ "openssl", "x509", "-text", "-noout" ]
   output = Tac.run( cmd, stdout=Tac.CAPTURE, stderr=Tac.CAPTURE,
                     ignoreReturnCode=True, input=certData )
   matchObj = re.search( r'Public-Key: \((\d+) bit\)', output )
   if matchObj is None:
      raise SslCertKeyError( "Can't get size of certificate's public key" )
   size = matchObj.group( 1 )
   debug( "Public key size is ", size )
   return int( size )

def getRsaPublicKey( privateKeyPath, publicKeyPath ):
   # Extract the RSA public key part from SSL key pair
   from M2Crypto import RSA
   try:
      RSA.load_key( privateKeyPath ).save_pub_key( publicKeyPath )
   except RSA.RSAError as e:
      error( "getRsaPublicKey exception:", e )
      raise SslCertKeyError( "Unable to extract public key from SSL private key %s"
                             % privateKeyPath.split( "/" )[ -1 ] )

def generateKeyCertHash( filePath, logType, hashAlgo='SHA-256', blockSize=65536 ):
   if not os.path.exists( filePath ):
      raise SslCertKeyError( "SSL %s does not exist" % filePath )

   # For now, we always use SHA-256.
   assert hashAlgo == 'SHA-256'

   # Get the file path to hash
   tmpPubKey = None
   if logType == "sslkey:":
      tmpPubKey = tempfile.NamedTemporaryFile()
      getRsaPublicKey( filePath, tmpPubKey.name )

   # generate the SHA-256 hash of file content
   fileToHash = tmpPubKey.name if logType == "sslkey:" else filePath
   sha256sum = hashlib.sha256()
   with open( fileToHash, 'rb' ) as f:
      for block in iter( lambda: f.read( blockSize ), b'' ):
         sha256sum.update( block )
   return sha256sum.hexdigest()

def getLogActionAndFileHash( filePath, logType, defaultAction, hashAlgo='SHA-256' ):
   if filePath and os.path.exists( filePath ):
      logAction = "updated"
      fileHash = generateKeyCertHash( filePath, logType, hashAlgo )
   else:
      logAction = defaultAction
      fileHash = ""
   return logAction, fileHash

def generateSslKeyCertSysLog( filePath, logType, logAction,
                              oldFileHash='', hashAlgo='SHA-256' ):
   trace( "generateSslKeyCertSysLog for", logAction, filePath )
   fileName = filePath.split( "/" )[ -1 ]
   logTypeStr = "private key" if logType == "sslkey:" else "certificate"
   fileTypeStr = "corresponding public " if logType == "sslkey:" else ""
   if logAction == "created":
      # pylint: disable-msg=undefined-variable
      Logging.log( SECURITY_SSL_KEY_CERT_CREATED, logTypeStr, fileName,
                   hashAlgo, fileTypeStr,
                   generateKeyCertHash( filePath, logType, hashAlgo ) )
   elif logAction == "updated":
      # pylint: disable-msg=undefined-variable
      Logging.log( SECURITY_SSL_KEY_CERT_UPDATED, logTypeStr, fileName,
                   hashAlgo, fileTypeStr, oldFileHash,
                   generateKeyCertHash( filePath, logType, hashAlgo ) )
   elif logAction == "deleted":
      # pylint: disable-msg=undefined-variable
      Logging.log( SECURITY_SSL_KEY_CERT_DELETED, logTypeStr, fileName,
                   hashAlgo, fileTypeStr, oldFileHash )
   elif logAction == "imported":
      # pylint: disable-msg=undefined-variable
      Logging.log( SECURITY_SSL_KEY_CERT_IMPORTED, logTypeStr, fileName,
                   hashAlgo, fileTypeStr,
                   generateKeyCertHash( filePath, logType, hashAlgo ) )
