# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import
import errno
import numbers
import os
import re
import shutil

from SpaceMgmtLib import Utils
import Tac

#####################################################################################
# Space management
#####################################################################################

def computeReclaimSpaceSize( totalSize, reservedPct, freePct, usedSize ):
   """
   Compute how much space needs to be reclaimed in kibibytes to comply with the
   space usage requirement.

   Notes
   -----
   To meet the space usage requirement, the following condition needs to be True:
   usedPct <= reservedPct - freePct

   Parameters
   ----------
   totalSize: str
      Total size available in KiB.
   reservedPct: int
      constraint: integer in the range [0-100]
      Percentage of the total size we can use.
   freePct: int
      constraints:
         integer in the range [0-100]
         freePct <= reservedPct
      Percentage of the total size that should be free.
   usedSize: int
      consraint: >= 0 and <= totalSize
      Size of the already used space in KiB.

   Returns
   -------
   int
      Size in KiB of the space that needs to be freed to meet the requirements.
   """

   assert Utils.validPct( reservedPct ), 'reservedPct is not a valid percentage'
   assert Utils.validPct( freePct ), 'freePct is not a valid percentage'
   assert freePct <= reservedPct, 'Can not compute freePct greather than reservedPct'
   assert usedSize <= totalSize, 'Can not use more size than the total size.'

   # How many percent of 'totalSize' does 'usedSize' represents
   usedPct = Utils.ratioToPct( usedSize, totalSize )
   # Maximum percentage of total size usable to meet the requirement
   maxPct = reservedPct - freePct

   # The actual space usage alredy meet the requirements, not space needs to be free.
   if usedPct <= maxPct:
      return 0

   # Percentage of 'totalSize' that needs to be free to meet the requirements
   reclaimPct = usedPct - maxPct

   return Utils.pctToRatio( reclaimPct, totalSize )

def _triageLogFiles( filenames, keyFromFilename ):
   """
   Helper function to triage a list of log filenames by creators (.i.e service or
   agent the log file belongs to.)

   Parameters
   ----------
   fileList: list of str
      filenames to triage
   keyFromFilename: function
      Takes a filename as argument and return a string which will be the key under
      which this filename should be stored.

   Returns
   -------
   dict of str: (list of str)
      key: logfile creator's name deducted from the filename using keyFromFilename
      value: list of filenames associated to the following key.
   """

   triagedFiles = {}

   for filename in filenames:
      key = keyFromFilename( filename )
      triagedFiles.setdefault( key, [] ).append( filename )

   return triagedFiles

def triageLogFiles( dirPath ):
   """
   Triage services' log files and sort them by service.

   Notes
   -----
   The given directory should contain general logs like /var/log directory, .i.e
   filenames should have the following format: <service name>.<suffix>
   Note that usually the suffix is just '.log' but it can be anything.
   If it is empty, the '.' is not needed. The all filename will be used as the
   the service name.
   For each service, the associated list of filenames will be sorted by modification
   time with the oldest one first.

   Parameters
   ----------
   dirPath: str
      Absolute path of the directory containing log files.

   Returns
   -------
   dict of str: (list of str)
      key: service name.
      value: list of names of this service's log files.
   """

   serviceNameFromFilename = lambda filename: filename.split( '.' )[ 0 ]
   return _triageLogFiles( Utils.sortDirByMtime( dirPath ), serviceNameFromFilename )

def triageAgentsLogFiles( dirPath ):
   """
   Triage agents' log files and sort them by agent.

   Notes
   -----
   The given directory should contain agent logs /var/log/agents directory, i.e.
   filenames should have the following format: <agent name>-<pid>
   For each agent, the associated list of filenames will be sorted by modification
   time with the oldest one first.

   Parameters
   ----------
   dirPath: str
      Absolute path of the directory containing log files.

   Returns
   -------
   dict of str: (list of str)
      key: agent name.
      value: list of names of this agent's log files.
   """

   agentNameFromFilename = lambda filename: filename.split( '-' )[ 0 ]
   return _triageLogFiles( Utils.sortDirByMtime( dirPath ), agentNameFromFilename )

def triageCoreFiles( dirPath ):
   """
   Triage core files and sort them first by process name and then by pids.

   Notes
   -----
   The given directory should contain core files like /var/log/core directory, i.e.
   filenames should have the following format:
      core.<pid>.<epoch time>.<name>-<suffix>.gz

   Parameters
   ----------
   dirPath: str
      Absolute path of the directory containing core files.

   Returns
   -------
   dict of str: (dict of int: str)
      key: process name
      value: dict of int: str
         key: pid
         value: filename
   """

   triagedFiles = {}

   for filename in Utils.sortDirByMtime( dirPath ):
      try:
         _, pid, _, name, _ = filename.split( '.' )
         name = name.split( '-' )[ 0 ]
      except ( ValueError, IndexError ):
         # The filename doesn't respect the known format. Ignore that file.
         pass
      else:
         triagedFiles.setdefault( name, {} ).update( { int( pid ) : filename } )

   return triagedFiles

def _removeOlderLogIterations( triagedFiles, dirPath, maxFileCount ):
   """
   Reduce the number of log files by deleting older iterations.

   Parameters
   ----------
   triagedFiles: dict of str: (list of str)
      Log files triaged by service/agent names. See _triageLogFiles for details.
   maxFileCount: int
      constraint: >= 0
      Max number of iterations to keep for a of log file.

   Returns
   -------
   tuple( size, filenames ):
      size: int
         Total freed size in KiB.
      filenames: list of str
         List of deleted filenames.
   """

   assert maxFileCount >= 0, 'File count must be equal to or greater than 0'

   rmSize = 0
   deletedFilenames = []
   rmFile = ( lambda name:
              Utils.removeFile( os.path.join( dirPath, name ), retSize=True ) )

   for _, filenames in triagedFiles.iteritems():
      extraFileCount = len( filenames ) - maxFileCount
      if extraFileCount > 0:
         extraFilenames = filenames[ : extraFileCount ]
         rmSize += sum( map( rmFile, extraFilenames ) )
         deletedFilenames += extraFilenames

   return rmSize, deletedFilenames

def _removeCoreFiles( dirPath, agentLogFilenames ):
   """
   Delete core files associated to agent log files.

   Parameters
   ----------
   dirPath: str
      Absolute path of the directory containing core files.
   logFilenames: list of str
      List of log filenames. The deleted core files are the one associated to these
      log files.

   Returns
   -------
   tuple( size, filenames ):
      size: int
         Total freed size in KiB.
      filenames: list of str
         List of deleted filenames.
   """

   deletedFilenames = []
   coreFilesByNameByPid = triageCoreFiles( dirPath )

   for filename in agentLogFilenames:
      agentName, pid = filename.split( '-' )
      pid = pid.split( '.' )[ 0 ]

      # For the same pid we can have two core files, one named after the agent
      # and/or the netns core file.
      for name in agentName, 'netns':
         try:
            deletedFilenames.append( coreFilesByNameByPid[ name ][ int( pid ) ] )
         except KeyError:
            continue

   rmFile = ( lambda name:
              Utils.removeFile( os.path.join( dirPath, name ), retSize=True ) )
   rmSize = sum( map( rmFile, deletedFilenames ) )

   return rmSize, deletedFilenames

def reclaimSpaceFromLogsAt( logDirPath,
                            agentLogDirPath,
                            coreDirPath,
                            maxLogFileCount,
                            maxAgentLogFileCount ):
   """
   Reclaim space from common log directories.

   Notes
   -----
   The space is reclaimed by deleting older iterations of the same log file.
   A log file can have multiple iterations, created by logrotate. Agents also
   create a new file each time they are restarted and run with a new pid.

   The above logic apply to the log directory `logDirPath` and the agent log
   directory `agentLogDirPath`.
   Core files are only removed if the associated agent log files are removed.

   If one of the directories paths is None, that directory will not be checked.
   However note that if `coreDirPath` is given alone, no space will be reclaimed.

   Parameters
   ----------
   logDirPath: str
      Absolute path of directory containing general log files, .e.g /var/log
   agentLogDirPath: str
      Absolute path of directory containing agent log files, .e.g /var/log/agents
   coreDirPath: str
      Absolute path of directory containing core files, .e.g /var/core
   maxLogFileCount: int
      constraint: >= 0
      Max number of iterations to keep for a general log file.
   maxAgentLogFileCount: int
      constraint: >= 0
      Max number of iterations to keep for an agent log file.

   Returns
   -------
   tuple( size, fileCount, stats )
      size: int
         Total reclaimed size in KiB.
      fileCount: int
         Total number of files removed.
      stats: dict of str: tuple
         Detailed report of how space was reclaimed.
         key: category name corresponding to one of the specifided directories:
            'logs': files removed from `logDirPath`.
            'agents': files removed from `agentLogDirPath`.
            'cores': files removed from `coreDirPath`.
         value: tuple( size, filenames )
            size: int
               Total reclaimed size for that directory in KiB.
            filenames: list of str
               List of filenames removed from that directory.
   """

   # log directory
   rmLogFilesSz, rmLogFiles = 0, []
   if logDirPath is not None:
      logFiles = triageLogFiles( logDirPath )
      rmLogFilesSz, rmLogFiles = _removeOlderLogIterations( logFiles,
                                                            logDirPath,
                                                            maxLogFileCount )
   # agent log directory
   rmAgentLogFilesSz, rmAgentLogFiles = 0, []
   if agentLogDirPath is not None:
      agentLogFiles = triageAgentsLogFiles( agentLogDirPath )
      ( rmAgentLogFilesSz,
        rmAgentLogFiles ) = _removeOlderLogIterations( agentLogFiles,
                                                       agentLogDirPath,
                                                       maxAgentLogFileCount )
   # core files directory
   rmCoreFilesSz, rmCoreFiles = 0, []
   if coreDirPath is not None:
      rmCoreFilesSz, rmCoreFiles = _removeCoreFiles( coreDirPath,
                                                     rmAgentLogFiles )

   # reclaim stats
   totalFileCount = len( rmLogFiles ) + len( rmAgentLogFiles ) + len( rmCoreFiles )
   totalSize = rmLogFilesSz + rmAgentLogFilesSz + rmCoreFilesSz
   statsByDir = {
      'logs' : ( rmLogFilesSz, rmLogFiles ),
      'agents' : ( rmAgentLogFilesSz, rmAgentLogFiles ),
      'cores' : ( rmCoreFilesSz, rmCoreFiles )
   }

   return totalSize, totalFileCount, statsByDir

def reclaimSpaceFromEosLogDirs( maxLogFileCount, maxAgentLogFileCount ):
   """
   Reclaim space from EOS log directories:
      * /var/log
      * /var/log/agents
      * /var/core

   Notes
   -----
   This function is just a wrapper over reclaimSpaceFromLogsAt which drives the
   reclaiming space logic.

   Parameters
   ----------
   maxLogFileCount: int
      constraint: >= 0
      Max number of iterations to keep for a general log file.
   maxAgentLogFileCount: int
      constraint: >= 0
      Max number of iterations to keep for an agent log file.

   Returns
   -------
   tuple( size, fileCount, stats )
      See reclaimeSpaceFromLogAt for details.
   """

   return reclaimSpaceFromLogsAt( Utils.EOS_LOG_DIR_PATH,
                                  Utils.EOS_AGENT_LOG_DIR_PATH,
                                  Utils.EOS_CORE_DIR_PATH,
                                  maxLogFileCount,
                                  maxAgentLogFileCount )
