#!/usr/bin/env python
# Copyright (c) 2008, 2009, 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import sys, time, re
# pylint: disable-msg=F0401
import Tac, SmbusUtil, PyClient

def strToInt( s ):
   try:
      i = int( s )
   except ValueError:
      if s.startswith( "0b" ) or s.startswith( "0B" ):
         i = int( s[2:], 2 )
      else:
         i = int( s, 16 )
   return i



defaultTimeout = 10
class RenesasChip( object ):
   def close( self ):
      pass

   def read8( self, addr ):
      try:
         # pylint: disable-msg=E1101
         result = self.helper.read8( addr )
      # pylint: disable-msg=W0612
      except SmbusUtil.SmbusFailure, e:
         result = None
      return result

   def write8( self, addr, data ):
      try:
         # pylint: disable-msg=E1101
         self.helper.write8( addr, data )
      except SmbusUtil.SmbusFailure, e:
         print "warning: smbus error: %s" % e

   def reset( self ):
      assert False, "ERROR: reset not implemented"

class SysdbChip( RenesasChip ):
   # pylint: disable-msg=W0231
   def __init__( self, solConfig, accelId, busId, deviceId, busTimeout,
                 verbose=False ):
      self.solConfig = solConfig
      addrSize = 1
      factory = SmbusUtil.Factory()
      self.helper = factory.device( accelId, busId, deviceId, addrSize,
                                    writeDelayMs='delay50ms', busTimeout=busTimeout )

   def reset( self ):
      if self.solConfig.reset is not None:
         self.solConfig.reset.asserted = True
         Tac.flushEntityLog()
         Tac.runActivities( 2.0 )
         self.solConfig.reset.asserted = False
         Tac.flushEntityLog()
         Tac.runActivities( 2.0 )
      elif self.solConfig.resetPinConfig is not None:
         self.solConfig.resetPinConfig.value = 1
         Tac.flushEntityLog()
         Tac.runActivities( 2.0 )
         self.solConfig.resetPinConfig.value = 0
         Tac.flushEntityLog()
         Tac.runActivities( 2.0 )
      else:
         print "error: undefined reset mechanism"
         sys.exit( 1 )

class SmbusChip( RenesasChip ):
   # pylint: disable-msg=W0231
   def __init__( self, accelId, busId, deviceId, busTimeout, reset, verbose=False,
         pciAddress=None, smbusAddrSpacing=0x100, mode = None ):
      addrSize = 1
      factory = SmbusUtil.Factory( mode )
      self.helper = factory.device( accelId, busId, deviceId, addrSize,
                                    writeDelayMs='delay50ms',
                                    busTimeout=busTimeout, pciAddress=pciAddress,
                                    smbusAddrSpacing=smbusAddrSpacing )
      self.resetType = reset
      self.pciAddress = pciAddress

   def reset( self ):
      if self.resetType == "Fixed":
         print "zapping scd"
         Tac.run( [ 'scd', 'write', '0x4000', '0x00800000' ] )  # reset chip
         time.sleep( 0.500 )
         Tac.run( [ 'scd', 'write', '0x4010', '0x00800000' ] )  # un-reset
         time.sleep( 0.250 )
      elif self.resetType.startswith( "Supervisor" ):
         # Very subtle difference here. "Supervisor" is Ahwahnee and
         # "supervisor" is EaglePeak/Oak/Beyond
         print "zapping sup"
         Tac.run( [ 'scd', 'write', '0x140', '0x01' ] )  # reset chip
         time.sleep( 0.500 )
         Tac.run( [ 'scd', 'write', '0x150', '0x01' ] )  # un-reset
         time.sleep( 0.250 )
      elif self.resetType.startswith( "supervisor" ):
         print "zapping sup"
         Tac.run( [ 'scd', 'write', '0x4000', '0x01' ] )  # reset chip
         time.sleep( 0.500 )
         Tac.run( [ 'scd', 'write', '0x4010', '0x01' ] )  # un-reset
         time.sleep( 0.250 )
      elif ( self.resetType.startswith( "Linecard" ) and
              self.resetType.endswith( "MtDan" ) ):
         Tac.run( [ 'pciwrite32', self.pciAddress, '0',
               '0x4020', '0x02000000' ] )  # reset chip
         time.sleep( 0.500 )
         Tac.run( [ 'pciwrite32', self.pciAddress, '0',
               '0x4030', '0x02000000' ] )  # un-reset chip
         time.sleep( 0.250 )
      elif  ( self.resetType.startswith( "Linecard" ) and
              self.resetType.endswith( "ElCapitan" ) ):
         Tac.run( [ 'pciwrite32', self.pciAddress, '0',
               '0x4040', '0xc0000fff' ] )  # enable GPIO reset
         time.sleep( 0.500 )
         Tac.run( [ 'pciwrite32', self.pciAddress, '0',
               '0x4020', '0x00001000' ] )  # reset chip
         time.sleep( 0.500 )
         Tac.run( [ 'pciwrite32', self.pciAddress, '0',
               '0x4030', '0x00001000' ] )  # un-reset chip
         time.sleep( 0.250 )
      elif ( self.resetType.startswith( "Linecard" ) or
            self.resetType.startswith( "Fabric" ) ):
         print "zapping slice"
            # set GPIO to output
         Tac.run( [ 'm8', 'gpio', 'set', self.resetType,
            'SecurityReset', 'output' ] )
         time.sleep( 0.100 )
            # reset security chip
         Tac.run( [ 'm8', 'gpio', 'set', self.resetType, 'SecurityReset', 'on' ] )
         time.sleep( 0.500 )
            # un-reset
         Tac.run( [ 'm8', 'gpio', 'set', self.resetType, 'SecurityReset', 'off' ] )
         time.sleep( 0.250 )
      else:
         print "error: undefined reset mechanism"
         sys.exit( 1 )

def chipFromCmdLine( path, reset, pciAddress=None,
      smbusAddrSpacing=0x100, mode=None ):

   # this is just copied from smbus...
   if not path or not path.startswith('/'):
      print "ERROR: Device path must begin with a /"
      sys.exit( 1 )
   path = path[1:]
   path = path.split( '/' )
   if not path[0] == 'scd':
      print "ERROR: Unknown device: %s" % path[0]
      sys.exit( 1 )
   if len( path ) != 4:
      print "ERROR: Device path should be /scd/<accelId>/<busId>/<deviceId>"
      sys.exit( 1 )
   accelId = strToInt(path[1])
   busId = strToInt(path[2])
   deviceId = strToInt(path[3])
   busTimeout = 'busTimeout1000ms'

   handle = SmbusChip( accelId, busId, deviceId, busTimeout, reset,
         pciAddress=pciAddress, smbusAddrSpacing=smbusAddrSpacing, mode=mode )
   return handle

def chipsFromSysdb( sysname ):
   Tac.sysnameIs( sysname )
   sysnameRoot = PyClient.PyClient( sysname, 'Sol' ).root()
   solRoot = sysnameRoot.entity[ '%s/Sysdb/hardware/sol/' % sysname ]
   configDir = solRoot[ 'config' ][ 'cell' ]

   solChips = {}

   for config in configDir.values():
      if not config:
         continue

      for name, value in config.solConfig.items():
         wrapper = Tac.newInstance("Smbus::AhamAddressWrapper")
         try:
            wrapper.addr = value.pyClient().eval( '%r.aham.baseAddr' %
                                                     value.ahamDesc )
         except SystemError, exception:
            # This assertion is thrown when iterating over the
            # backup supervisor and should be ignored since the
            # backup supe does not have a aham defined.
            assert "Could not find device master" in exception.message
            continue
         solChips[ name ] = SysdbChip( value, wrapper.accelId, wrapper.busId,
                                       wrapper.deviceId, wrapper.busTimeout )
   return solChips

def chipFromMagic( chipName ):

   cnLower = chipName.lower()
   magicChipMap = { "supervisor1" : ( 0, 1, 0x73, chipName ),
                    "supervisor2" : ( 0, 1, 0x73, chipName ),
                    "linecard3" :   ( 0, 0, 0x73, chipName ),
                    "linecard4" :   ( 1, 0, 0x73, chipName ),
                    "linecard5" :   ( 0, 1, 0x73, chipName ),
                    "linecard6" :   ( 1, 1, 0x73, chipName ),
                    "linecard7" :   ( 0, 2, 0x73, chipName ),
                    "linecard8" :   ( 1, 2, 0x73, chipName ),
                    "linecard9" :   ( 0, 3, 0x73, chipName ),
                    "linecard10" :  ( 1, 3, 0x73, chipName ),
                    "fabric1" :     ( 2, 0, 0x73, chipName ),
                    "fabric2" :     ( 2, 1, 0x73, chipName ),
                    "fabric3" :     ( 2, 2, 0x73, chipName ),
                    "fabric4" :     ( 2, 3, 0x73, chipName ),
                    "fabric5" :     ( 2, 4, 0x73, chipName ),
                    "fabric6" :     ( 2, 5, 0x73, chipName ),
                    }

   MtDanaPciMap = {'Ahwahnee': {
            "linecard3" : "0000:94:00.0",
            "linecard4" : "0000:80:00.0",
            "linecard5" : "0000:6c:00.0",
            "linecard6" : "0000:1c:00.0",
            "linecard7" : "0000:58:00.0",
            "linecard8" : "0000:44:00.0",
            "linecard9" : "0000:30:00.0",
            "linecard10" : "0000:08:00.0",
            },
            'EaglePeak':{
            "linecard3" : "0000:48:00.0",
            "linecard4" : "0000:58:00.0",
            "linecard5" : "0000:68:00.0",
            "linecard6" : "0000:78:00.0",
            "linecard7" : "0000:37:00.0",
            "linecard8" : "0000:27:00.0",
            "linecard9" : "0000:17:00.0",
            "linecard10" : "0000:07:00.0",
               }
            }

   ElCapPciMap = {'Ahwahnee': {
            "linecard3" : "0000:93:00.0",
            "linecard4" : "0000:7f:00.0",
            "linecard5" : "0000:6b:00.0",
            "linecard6" : "0000:1b:00.0",
            "linecard7" : "0000:57:00.0",
            "linecard8" : "0000:43:00.0",
            "linecard9" : "0000:2f:00.0",
            "linecard10" : "0000:07:00.0",
            },
            'EaglePeak': {
            "linecard3" : "0000:47:00.0",
            "linecard4" : "0000:57:00.0",
            "linecard5" : "0000:67:00.0",
            "linecard6" : "0000:77:00.0",
            "linecard7" : "0000:36:00.0",
            "linecard8" : "0000:26:00.0",
            "linecard9" : "0000:16:00.0",
            "linecard10" : "0000:06:00.0",
               }
            }
   if cnLower.startswith( "oaksupervisor"
         ) or cnLower.startswith( "supervisor" ):
      (accelId, busId, chipId, reset) = magicChipMap[ cnLower ]
      smbus = "/scd/%d/%d/%d" % (accelId, busId, chipId)
      chip = chipFromCmdLine( smbus, reset )
   else:
      output = Tac.run( ['cat', '/proc/cmdline'], stdout=Tac.CAPTURE )
      if output.find( 'eaglepeak' ) != -1:
         supType = 'EaglePeak'
      elif output.find( 'platform' ) == -1:
         supType = 'Ahwahnee'
      else:
         assert False, "Unsupported Sup Type"

      decodedPrefdl = ""
      mChip = re.search( '(supervisor|linecard|fabric)(\d+)', cnLower,
            re.IGNORECASE )

      if mChip:
         decodedPrefdl = Tac.run ( [ 'modulargenprefdl', '--%s'
            % mChip.group( 1 ), mChip.group( 2 ) ], stdout=Tac.CAPTURE )

      SID = re.search(  "SID : (\S+)", decodedPrefdl ).group(1)
      # Gen2 linecard needs to use raw mode which supplies
      # a PCI address to access Zuma-SCD Smbus
      if SID.startswith( 'MtDan' ):
         smbus = "/scd/%d/%d/%d" % ( 3, 1, 0x73 )
         chip = chipFromCmdLine( smbus, chipName+"_MtDan",
            MtDanaPciMap[ supType ][ cnLower ], 0x80, 'raw' )
      elif SID.startswith( 'ElCapitan') or SID.startswith('ThreeBrother'):
         smbus = "/scd/%d/%d/%d" % ( 1, 4, 0x73 )
         chip = chipFromCmdLine( smbus, chipName+"_ElCapitan",
            ElCapPciMap[ supType ][ cnLower ], 0x80, 'raw' )
      else: # Failover e.g., G1 cards
         (accelId, busId, chipId, reset) = magicChipMap[ cnLower ]
         smbus = "/scd/%d/%d/%d" % (accelId, busId, chipId)
         chip = chipFromCmdLine( smbus, reset )
   return chip


# Default setup
# pylint: disable-msg=F0401
import SmbusClient
SmbusClient.setup( Tac.sysname(), "SolUtil" )
