#!/usr/bin/env python
# Copyright (c) 2008, 2009, 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import array, sys, time
import Tac,SolUtil

class SolError( Exception ):
   def __init__(self, routine, message):
      self.routine = routine
      self.message = message

   def __str__(self):
      return "[%s][%s]" % (self.routine, self.message)
                                  

# Register Layout
SolMemControl      = 0x0
SolMemControlXid   = 0x1
SolMemStatus       = 0x2
SolMemStatusXid    = 0x3
SolMemDataIn       = 0x4
SolMemDataOut      = 0x5
SolMemPadding      = 0x6
SolMemBufferLength = 0x7
SolMemBuffer       = 0x8
SolMemBufferEnd    = 0x47

# Commands
SolCmdM2Info       = 0x01
SolCmdSWVersion    = 0x02
SolCmdInfo         = 0x0D

SolCmdSetData      = 0x23

SolCmdSetupMfg     = 0x31
SolCmdSetupChip    = 0x34

SolCmdGenM         = 0x51
SolCmdGetV         = 0x52

SolCmdAuthM        = 0x81
SolCmdSetG         = 0x82
SolCmdStartG       = 0x83
SolCmdAddEEPROM    = 0x84
SolCmdAuthG        = 0x85
SolCmdAuthN        = 0x86
SolCmdSetG0        = 0x87

# Status
SolStatusDone = 1
SolStatusBusy = 2

xid = 1

verbose = False

def intArrayToString( intArray ):
   l = []
   for c in intArray:
      if c is 0:
         break
      l.append( chr( c ) )
   return "".join( l )

def dump( data, str="" ):
   for i in range(len(data)):
      if ((i&0x0f) == 0 and i > 0):
         sys.stdout.write("\n")

      sys.stdout.write("%02x" % (data[i] & 0xff))
      if (((i+1)&0x0f) != 0):
         sys.stdout.write(" ")
   sys.stdout.write("\n")
   sys.stdout.flush()

# no newline
def dump1( data, str="" ):
   sys.stdout.write("%s:" % str)
   for i in range(len(data)):
      sys.stdout.write("%02x " % (data[i] & 0xff))
   sys.stdout.write("\n")
   sys.stdout.flush()

# no newline
def format1( data, str="" ):
   s = ""
   if str: s += "%s:" % str
   for i in range(len(data)):
      s += "%02x " % (data[i] & 0xff)
   return s

def dump_m2info( data ):
   return( "M2: 0x%02x%02x (%s)" % 
           ( data[0], data[1], intArrayToString( data[2:10] ) ) )

def dump_swinfo( data, str="" ):
   return( "%s: %c%c%c%c-%02x%02x%02x%02x" % 
           (str, data[0], data[1], data[2], data[3],
                 data[4], data[5], data[6], data[7] ) )

###############################################################################

def resetChip( handle, verbose=False ):
   # BUG6182: See bug text.
   iters = 0
   while iters < 10:
      try:
         reset( handle )
         if call( handle, SolCmdM2Info, maxWait=10 ) == 0:
            buffer = get( handle, 18 )
            if verbose:
               print "reset: %s" % dump_m2info( buffer )
            return
      except SolError, e:
         print "Sol error: %s[iter %d]" % (e, iters)
         iters = iters + 1
   print "ERROR: reset iteration limit reached", iters
   sys.exit( 1 )

###############################################################################

def call( handle, command, data=None, sleep=1, maxWait=180 ):
   """ execute a command on the chip """
   global xid

   if data is not None:
      handle.write8( SolMemDataIn, data )
   handle.write8( SolMemControlXid, xid )
   handle.write8( SolMemControl, command )

   if verbose and sleep > 1:
      sys.stdout.write("begin sleep: %d" % sleep)
      sys.stdout.flush()
   time.sleep( sleep )
   if verbose and sleep > 1:
      sys.stdout.write(" done!\n")
      sys.stdout.flush()
   status = handle.read8( SolMemStatus )
   statusXid = handle.read8( SolMemStatusXid )

   wait = 1
   while ((status == None or statusXid == None or statusXid != xid)
          and wait < maxWait):
      if verbose:
         sys.stdout.write(".")
         sys.stdout.flush()
      status = handle.read8( SolMemStatus )
      statusXid = handle.read8( SolMemStatusXid )
      time.sleep( 1 )
      wait = wait + 1

   def pretty(x):
      hex(x) if x is not None else "None"
      
   if status != SolStatusDone:
      dataout = handle.read8( SolMemDataOut )
      print "txn status error: cmd:%x, status:%s, dataout:%s, xid:%x" % \
            (command, pretty(status), pretty(dataout), xid)
      raise SolError( "call 0x%x" % command, "status error" )
   if statusXid != xid:
      print "txn xid error: cmd:%x, status:%s, statusxid:%s, xid:%x" % \
            (command, pretty(status), pretty(statusXid), xid)
      raise SolError( "call 0x%x" % command, "xid error" )

   xid = (xid + 1) % 0x100
   return 0

def get( handle, length ):  
   """ get some memory from the chip """
   data = array.array('B', [0] * length )

   for x in range(length):
      byte = handle.read8( SolMemBuffer + x)
      if byte == None:
         print "error reading byte", x
         raise SolError( "get", "read error [%d]" % x )
      else:
         data[x] = byte
   return data

def getDataOut( handle ):
   dataout = handle.read8( SolMemDataOut )
   return dataout

def set( handle, data ):
   """ set some memory in the chip """
   size = SolMemBufferEnd - SolMemBuffer + 1
   if len( data ) > size or len( data ) < 0:
      print "ERROR: set: invalid data length", len(data)
      raise SolError( "set", "invalid data length [%d]" % len(data) )
   handle.write8( SolMemBufferLength, len( data ) )
   for x in range( len( data ) ):
      handle.write8( SolMemBuffer + x, data[x] )

def chipFromCmdLine( smbus, reset ):
   chip = SolUtil.chipFromCmdLine( smbus, reset )
   return chip

def chipFromMagic( chipName ):
   chip = SolUtil.chipFromMagic( chipName )
   return chip

def setup( chipName, sysname, fallBack = True ):
   chipMap = SolUtil.chipsFromSysdb( sysname )

   # if no chip is specified, default to the first one
   if chipName is None:
      if len(chipMap) > 0:
         (chipName, chip) = chipMap.items()[0]
         print "No chip specified. Using default [%s]" % chipName
         return chip
      else:
         print "Error: no sol chip found"
         return None
         
   if chipMap.has_key( chipName ):
      return chipMap[ chipName ]
   if fallBack:
      return chipFromMagic( chipName )
   print "Error: chip not found [%s]" % chipName
   return None

def reset( handle ):
   handle.reset()

def cleanup( handle ):
   handle.close()

def chipList( sysname ):
   chipMap = SolUtil.chipsFromSysdb( sysname )
   return chipMap.keys()
