#!/usr/bin/env python
# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import Tracing

traceHandle = Tracing.Handle( 'SnmpConfigMan' )
t0 = traceHandle.trace0
plugin = None

_trapSender = Tac.newInstance( "Snmp::TrapSender" )

class ConfigManMibReactor( Tac.Notifiee ):
   notifierTypeName = "Cli::ConfigHistory"

   def __init__( self, configHistory, configHistoryTrapStatus ):
      self.configHistory_ = configHistory
      self.configHistoryTrapStatus_ = configHistoryTrapStatus
      self.historyEventTable_ = self.configHistory_.historyEventTable
      Tac.Notifiee.__init__( self, configHistory )
      for entry in self.configHistory_.historyEventTable:
         self.handleHistoryEventTable( entry )

   @Tac.handler( 'historyEventTable' )
   def handleHistoryEventTable( self, historyTableIndex ):
      t0( "In handleHistoryEventTable" )
      if historyTableIndex not in self.historyEventTable_:
         t0( "Dequeuing entry at index:", historyTableIndex )
         return

      if self.configHistoryTrapStatus_.lastTrapSentTime >= \
             self.historyEventTable_[ historyTableIndex ].historyEventTime:
         t0( "We already sent traps for these events, let us not resend" )
         return

      t0( "Sending config management trap" )
      historyEvent = self.historyEventTable_[ historyTableIndex ]
      err = _trapSender.doAristaConfigManEventNotify(
         historyEvent.historyEventCommandSource,
         historyEvent.historyEventConfigSource,
         historyEvent.historyEventConfigDest,
         historyEvent.historyEventConfigSourceURLScheme,
         historyEvent.historyEventConfigDestURLScheme,
         historyTableIndex )

      # Setting lastTrapSentTime. Will use this to ensure no resending of traps
      # in case of agent restarts.
      if not err:
         t0( "Sent Config Management trap at ", Tac.now() )
         self.configHistoryTrapStatus_.lastTrapSentTime = \
             self.historyEventTable_[ historyTableIndex ].historyEventTime

class ConfigManMibPlugin( object ):
   def __init__( self, ctx ):
      entityManager = ctx.entityManager()
      mg = entityManager.mountGroup()
      self.configHistoryTrapStatus = mg.mount( "cli/configHistoryTrapStatus",
                                               "Cli::ConfigHistoryTrapStatus", "w" )
      self.configHistory = mg.mount( "cli/configHistory", "Cli::ConfigHistory", "r" )

      def _finish():
         # pylint: disable-msg=W0201
         self.collReactor = ConfigManMibReactor( self.configHistory,
                                                 self.configHistoryTrapStatus )
      mg.close( lambda: None )
      ctx.callbackIs( _finish )

def Plugin( ctx ):
   t0( "SnmpConfigManMib plugin loading..." )
   Tac.dlopen( "libSnmpConfigManMib.so" )
   global plugin
   plugin = ConfigManMibPlugin( ctx )
