# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac, Tracing
import Cell
import SharedMem
import Smash
from Toggles.QosToggleLib import toggleEcnCountersSmashEnabled

traceHandle = Tracing.Handle( 'QosSnmpPlugin' )
t0 = traceHandle.trace0

FapId = Tac.Type( 'FlexCounters::FapId' )

class QosMibConfigReactor( Tac.Notifiee ):
   notifierTypeName = "QosSnmp::QosMibConfig"

   def __init__( self, qosMibConfig, qosSnmpCounterConfig ):
      self.qosSnmpCounterConfig = qosSnmpCounterConfig
      Tac.Notifiee.__init__( self, qosMibConfig )
      self.activity_ = Tac.ClockNotifiee( handler=self.triggerQosCounterUpdate )
      self.activity_.timeMin = Tac.endOfTime
      self.updateInterval = Tac.endOfTime
      self.handleCounterUpdateInterval()

   @Tac.handler( 'counterUpdateInterval' )
   def handleCounterUpdateInterval( self ):
      t0( "In handleCounterUpdateInterval" )
      newInterval = self.notifier_.counterUpdateInterval
      self.updateInterval = newInterval
      self.triggerQosCounterUpdate()

   def triggerQosCounterUpdate( self ):
      t0( 'Updating Qos counters' )
      self.qosSnmpCounterConfig.pmapCounterUpdateRequestTime = Tac.now()
      # trigger counter update after updateInterval.
      self.activity_.timeMin = Tac.now() + self.updateInterval

class QosSnmpPlugin( object ):
   def __init__( self, ctx ):
      entityManager = ctx.entityManager()
      sysdb = entityManager.root()
      snmpAgentRoot = sysdb.parent[ 'Snmp' ]
      snmpRoot = sysdb.parent[ 'snmp' ]
      qosMibDir = snmpRoot.mkdir( 'qos' )
      self.qosMib = qosMibDir.newEntity( 'QosSnmp::QosMib', 'root' )

      mg = entityManager.mountGroup()
      # We only write to pmapCounterUpdateRequestTime in qosSnmpCounterConfig. 
      # The rest is all read-access.
      self.qosConfig = mg.mount( 'qos/config', 'Qos::Config', 'r' )
      self.qosSnmpCounterConfig = mg.mount( 'qos/counterConfig/snmp', 
                                            'Qos::CounterConfig', 'w' )
      self.qosStatus = mg.mount( 'qos/status', 'Qos::Status', 'r' )
      self.qosMibConfig = mg.mount( 'snmp/qos/config', 'QosSnmp::QosMibConfig', 'r' )
      self.qosHwStatus = mg.mount( 'qos/hardware/status/global',
                                   'Qos::HwStatus', 'r' )
      qosSliceHwStatusDirPath = \
            "cell/" + str( Cell.cellId() ) + "/qos/hardware/status/slice"
      self.qosSliceHwStatusDir = mg.mount( qosSliceHwStatusDirPath, 'Tac::Dir', 
                                           'ri' )
      self.qosAclSliceHwStatusDir = mg.mount( 'qos/hardware/acl/status/slice',
                                              'Tac::Dir', 'ri' )
      self.allIntfStatusDir = mg.mount( 'interface/status/all', 
                                        'Interface::AllIntfStatusDir', 'r' )
      self.qosHwCounter = mg.mount( 'qos/hardware/counter', "Qos::HwCounter", 'r' )

      if toggleEcnCountersSmashEnabled():
         # Mount ECN Queue counter smash table.
         shmemEm = SharedMem.entityManager( sysdbEm=entityManager )
         readerInfo = Smash.mountInfo( 'reader' )
         mountPath = 'flexCounters/counterTable/Ecn/%u' % ( FapId.allFapsId )
         self.ecnCounterTable = shmemEm.doMount( mountPath, "Qos::EcnCounterTable",
                                                 readerInfo )
      else:
         self.ecnCounterTable = None

      self.root = None
      self.rootSm = None
      self.qosMibConfigReactor = None
      self.qosSliceHwStatus = None
      self.qosAclSliceHwStatus = None

      def _finishMounts():
         t0( 'QosSnmp: finished mounts' )
         self.root = snmpAgentRoot.newEntity( 'QosSnmp::Root', 'QosSnmp' )
         self.root.rootSm = ( self.qosConfig, self.qosStatus,
                              self.qosHwStatus,
                              self.qosSliceHwStatusDir,
                              self.qosAclSliceHwStatusDir,
                              self.qosMibConfig,
                              self.allIntfStatusDir,
                              self.qosHwCounter,
                              self.qosMib,
                              self.ecnCounterTable )
         self.qosMibConfigReactor = QosMibConfigReactor( self.qosMibConfig,
                                                         self.qosSnmpCounterConfig )
      mg.close( lambda: None )
      ctx.callbackIs( _finishMounts )

qosSnmpPlugin = None

def Plugin( ctx ):
   t0( 'Loading Qos SnmpPlugin' )
   global qosSnmpPlugin
   qosSnmpPlugin = QosSnmpPlugin( ctx )
   Tac.dlopen( 'libQosSnmpAgent.so' )
