# Copyright (c) 2008-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Plugins
import Tac, Tracing
import Cell
import SharedMem
from Snmp import addStage2MountFunction
from SnmpPlugin.XcvrAllStatusDir import xcvrAllStatusDirSysdbPath

# Creates dependency on IntfSnmp
# pkgdeps: import SnmpPlugin.IntfSnmp

traceHandle = Tracing.Handle( 'EthIntfSnmp' )
t0 = traceHandle.trace0

import QuickTrace
qt0 = QuickTrace.trace0
qt1 = QuickTrace.trace1

pfcStatusTypeName = 'Pfc::Status'

_intfDetails = Tac.newInstance( "IntfSnmp::IntfDetails" )
AristaPfcTableEntryKey = Tac.Type( "EthIntfSnmp::AristaPfcPriorityTableEntry::Key" )
AristaPfcWatchdogTxQueueTableIndex = \
      Tac.Type( "EthIntfSnmp::AristaPfcWatchdogTxQueueTableIndex" )
IntfQueueType = Tac.Type( "Interface::PktType" )

class EthPhyIntfStatusReactor( Tac.Notifiee ):
   '''A reactor to handle the delayed set of the portId.'''
   notifierTypeName = "Interface::EthPhyIntfStatus"

   def __init__( self, intfStatus, intfCounter, parent, rmonMib ):
      self.intfCounter_ = intfCounter
      self.parent_ = parent
      self.rmonMib_ = rmonMib
      Tac.Notifiee.__init__( self, intfStatus )
      self.handlePortId()

   @Tac.handler( 'portId' )
   def handlePortId( self ):
      portId = self.notifier_.portId
      intfName = self.notifier_.intfId
      if portId < 1:
         qt0( 'handlePortId: still skipping, portId for', intfName, 'is', portId )
         return
      self.parent_.portIdDict_[ intfName ] = portId
      qt1( 'handlePortId: adding', intfName, 'as ethStatsIndex', portId )
      _intfDetails.status = self.notifier_
      ifIndex = _intfDetails.ifIndex
      self.rmonMib_.etherStat.newMember( portId, ifIndex, self.intfCounter_ )
      # Since the portId is only ever set, and not changed, this reactor
      # doesn't have to hang around.
      del self.parent_.portIdReactor_[ intfName ]

class EthIntfSnmpMibHelper( object ):

   ''' 
   This is the helper object that actually deals with populating all
   the relevant mib objects (internal to the agent). It is driven 
   by the reactors of intf counter and status directory below.
   '''

   def __init__( self, intfConfigDir, intfStatusDir, intfCounterDir,
                 etherLikeMib, rmonMib, ieee8021PfcMib,
                 aristaPfcMib, pfcStatus, qosSliceHwStatusDir ):
      self.intfConfigDir_ = intfConfigDir
      self.intfStatusDir_ = intfStatusDir
      self.intfCounterDir_ = intfCounterDir
      self.mib_ = etherLikeMib
      self.rmonMib_ = rmonMib
      self.ieee8021PfcMib_ = ieee8021PfcMib
      self.aristaPfcMib_ = aristaPfcMib
      self.pfcStatus_ = pfcStatus
      self.qosSliceHwStatusDir_ = qosSliceHwStatusDir

      # We probably don't ever want to remove entries from this dictionary.
      # See https://groups.google.com/a/aristanetworks.com/d/msg/snmp-dev/-
      # BXGaRFfU9W0/z1oKHSwh9G8J [URL split and hyphenated for line length].
      self.ifIndexDict_ = {}
      self.portIdDict_ = {}
      self.portIdReactor_ = {}

      # Create all the reactors that drive the Helper
      self.intfStatusDirReactor_ = EthPhyIntfStatusDirReactor( intfStatusDir,
                                                               self )
      self.counterDirReactor_ = EthIntfCounterDirReactor( intfCounterDir,
                                                          self )
      self.intfConfigDirReactor_ = EthPhyIntfConfigDirReactor( intfConfigDir,
                                                               self )
      self.pfcStatusReactor_ = PfcStatusReactor( pfcStatus,
                                                 intfCounterDir,
                                                 self )

   def addPfcMibEntry( self, ifIndex, intfCounter ):
      if not self.pfcStatus_.pfcClassFramesCounterSupported:
         return
      self.delPfcMibEntry( ifIndex )
      for pri in range( 8 ):
         key = AristaPfcTableEntryKey( ifIndex, pri )
         self.aristaPfcMib_.aristaPfcPriorityTable.newMember( key, intfCounter )

   def delPfcMibEntry( self, ifIndex, forced=False ):
      if not self.pfcStatus_.pfcClassFramesCounterSupported and not forced:
         return
      for pri in range( 8 ):
         key = AristaPfcTableEntryKey( ifIndex, pri )
         del self.aristaPfcMib_.aristaPfcPriorityTable[ key ]

   def addPfcWdTxQueueMibEntry( self, ifIndex, intfName ):
      portWatchdogStatus = None
      pfcSliceHwStatus = None
      for sliceHwStatus in self.qosSliceHwStatusDir_.itervalues():
         if sliceHwStatus is None:
            continue
         portWatchdogStatus = sliceHwStatus.portWatchdogStatus.get( intfName )
         if portWatchdogStatus:
            pfcSliceHwStatus = sliceHwStatus
            break

      if self.pfcStatus_.watchdogSupported and portWatchdogStatus:
         for txId in xrange( 8 ):
            for queueType in [ IntfQueueType.uc, IntfQueueType.mc ]:
               key = AristaPfcWatchdogTxQueueTableIndex( ifIndex, queueType,
                                                         txId )
               self.aristaPfcMib_.aristaPfcWatchdogTxQueueTable.newMember( key,
                                                         intfName, pfcSliceHwStatus )
         
   def delPfcWdTxQueueMibEntry( self, ifIndex ):
      for txId in xrange( 8 ):
         for queueType in [ IntfQueueType.uc, IntfQueueType.mc ]:
            key = AristaPfcWatchdogTxQueueTableIndex( ifIndex, queueType,
                                                      txId )
            del self.aristaPfcMib_.aristaPfcWatchdogTxQueueTable[ key ]
               
   def addEntries( self, intfName ):
      intfStatus = self.intfStatusDir_.intfStatus.get( intfName )
      intfCounter = self.intfCounterDir_.intfCounterDir.get( intfName )
      intfConfig = self.intfConfigDir_.intfConfig.get( intfName )
      assert intfStatus and intfConfig and intfCounter

      _intfDetails.status = intfStatus
      idx = _intfDetails.ifIndex
      if idx < 0:
         qt0( 'addEntries : skipping', intfName, 'because ifIndex is', idx )
         return

      stats = self.mib_.dot3Stat.get( idx )
      if stats:
         qt0( 'addEntries : skipping', intfName, 'entry present' )
         return

      qt1( 'addEntries: adding', intfName, 'as dot3StatsIndex', idx )

      self.ifIndexDict_[ intfName ] = idx
      stats = self.mib_.dot3Stat.newMember( idx )
      stats.status = intfStatus
      stats.counter = intfCounter[ "current" ]
      pause = self.mib_.dot3Pause.newMember( idx )
      pause.status = intfStatus
      pause.counter = stats.counter
      portId = intfStatus.portId
      if portId < 1:
         qt0( 'addEntries : delaying rmon for', intfName, 'because portId is',
              portId )
         self.portIdReactor_[ intfName ] = EthPhyIntfStatusReactor(
            intfStatus, stats.counter, self, self.rmonMib_ )
         return
      self.portIdDict_[ intfName ] = portId
      qt1( 'addEntries: adding', intfName, 'as ethStatsIndex', portId )
      self.rmonMib_.etherStat.newMember( portId, idx, stats.counter )
      self.ieee8021PfcMib_.ieee8021PfcTable.newMember( idx, stats.counter )
      self.addPfcMibEntry( idx, stats.counter )
      self.addPfcWdTxQueueMibEntry( idx, intfName )
      
   def removeEntries( self, intfName ):
      # Interface was removed: remove from mib_.dot3Stat and dot3Pause
      idx = self.ifIndexDict_.get( intfName )
      if idx:
         qt1( 'removeEntries: removing', intfName, 'with dot3StatsIndex', idx )
         del self.mib_.dot3Stat[ idx ]
         del self.mib_.dot3Pause[ idx ]
         del self.ieee8021PfcMib_.ieee8021PfcTable[ idx ]
         self.delPfcMibEntry( idx )
         self.delPfcWdTxQueueMibEntry( idx )
      else:
         qt0( 'removeEntries: no dot3StatsIndex found for', intfName )
      portId = self.portIdDict_.get( intfName )
      if portId:
         qt1( 'removeEntries: removing', intfName, 'with ethStatsIndex', portId )
         del self.rmonMib_.etherStat[ portId ]
      else:
         qt0( 'removeEntries: no ethStatsIndex found for', intfName )

   def intfEntitiesPresent( self, intfName ):
      ''' Helper method to check if all required Intf entities are present.
      This decides whether the interface should appear in the MIB or not 
      '''
      return self.intfConfigDir_.intfConfig.has_key( intfName ) and \
         self.intfStatusDir_.intfStatus.has_key( intfName ) and \
         self.intfCounterDir_.intfCounterDir.has_key( intfName )

   def syncEntries( self ):
      # Synchronize mib_.dot3Stat with intfStatusDir_.intfStatus.  This code
      # makes the reasonable assumption that mib_.dot3Pause will also be kept
      # in sync if we always add and remove from dot3Stat and dot3Pause at the
      # same time.
      toAdd = {}
      toRemove = {}
      srcColl = self.intfStatusDir_.intfStatus
      destColl = self.mib_.dot3Stat
      for name, status in srcColl.iteritems():
         ifIndex = self.ifIndexDict_.get( name, None )
         qt0( 'Looking at', name, ifIndex )
         if self.intfEntitiesPresent( name ) and \
            ( ifIndex is None or \
              not destColl.has_key( ifIndex ) or \
              destColl[ ifIndex ].status != status ):
            toAdd[ name ] = status
      for ifIndex, tableEntry in destColl.iteritems():
         name = tableEntry.status.intfId
         if not self.intfEntitiesPresent( name ):
            toRemove[ name ] = True
      qt0( 'syncEntries: adding', len( toAdd ), ' intfs, removing',
           len( toRemove ), 'intfs' )
      for name in toRemove.keys():
         self.removeEntries( name )
      for name, status in toAdd.iteritems():
         self.addEntries( name )

   def handleIntf( self, intfName ):

      qt0( 'handleIntf(', intfName, ')' )

      if intfName is None:
         self.syncEntries()
         return

      # We only add entries when all related Intf entities are present
      if self.intfEntitiesPresent( intfName ):
         self.addEntries( intfName )
      else:
         self.removeEntries( intfName )


class EthIntfCounterDirReactor( Tac.Notifiee ):

   notifierTypeName = "Interface::AllEthIntfCounterPtrDir"

   def __init__( self, intfCounterDir, mibHelper ):
      self.intfCounterDir_ = intfCounterDir
      self.mibHelper_ = mibHelper
      Tac.Notifiee.__init__( self, intfCounterDir )
      for intfName in intfCounterDir.intfCounterDir:
         self.handleIntfCounter( intfName )

   @Tac.handler( 'intfCounterDir' )
   def handleIntfCounter( self, intfName ):
      qt1( 'handleIntfCounter: ', intfName )
      self.mibHelper_.handleIntf( intfName )

class PfcStatusReactor( Tac.Notifiee ):
   notifierTypeName = pfcStatusTypeName
   def __init__( self, pfcStatus, intfCounterDir, mibHelper ):
      self.pfcStatusDir_ = pfcStatus
      self.mibHelper_ = mibHelper
      self.intfCounterDir_ = intfCounterDir
      Tac.Notifiee.__init__( self, pfcStatus )
      self.handlePfcClassFramesCounters()

   @Tac.handler( "pfcClassFramesCounterSupported" )
   def handlePfcClassFramesCounters( self ):
      for name, ifIndex in self.mibHelper_.ifIndexDict_.iteritems():
         intfCounter = self.intfCounterDir_.intfCounterDir.get( name )
         if intfCounter and self.notifier_.pfcClassFramesCounterSupported:
            counter = intfCounter[ "current" ]
            self.mibHelper_.addPfcMibEntry( ifIndex, counter )
         else:
            self.mibHelper_.delPfcMibEntry( ifIndex, True )

class EthPhyIntfStatusDirReactor( Tac.Notifiee ):

   notifierTypeName = "Interface::AllEthPhyIntfStatusDir"

   def __init__( self, intfStatusDir, mibHelper ):
      self.intfStatusDir_ = intfStatusDir
      self.mibHelper_ = mibHelper
      Tac.Notifiee.__init__( self, intfStatusDir )
      for intfName in intfStatusDir.intfStatus:
         self.handleIntfStatus( intfName )

   @Tac.handler( 'intfStatus' )
   def handleIntfStatus( self, intfName ):
      qt1( 'handleIntfStatus: ', intfName )
      self.mibHelper_.handleIntf( intfName )

class EthPhyIntfConfigDirReactor( Tac.Notifiee ):

   notifierTypeName = "Interface::AllEthPhyIntfConfigDir"

   def __init__( self, intfConfigDir, mibHelper ):
      self.intfConfigDir_ = intfConfigDir
      self.mibHelper_ = mibHelper
      Tac.Notifiee.__init__( self, intfConfigDir )
      for intfName in intfConfigDir.intfConfig:
         self.handleIntfConfig( intfName )

   @Tac.handler( 'intfConfig' )
   def handleIntfConfig( self, intfName ):
      qt1( 'handleIntfConfig: ', intfName )
      self.mibHelper_.handleIntf( intfName )

class PfcPortWatchdogStatusReactor( Tac.Notifiee ):

   notifierTypeName = "Qos::SliceHwStatus"

   def __init__( self, qosSliceHwStatus, mibHelper ):
      self.qosSliceHwStatus_ = qosSliceHwStatus
      self.mibHelper_ = mibHelper
      Tac.Notifiee.__init__( self, qosSliceHwStatus )
      for intfName in qosSliceHwStatus.portWatchdogStatus:
         self.handlePortWatchdogStatus( intfName )

   @Tac.handler( 'portWatchdogStatus' )
   def handlePortWatchdogStatus( self, intfName ):
      qt1( 'handlePortWatchdogStatus: ', intfName )
      if intfName is None:
         return
      portWatchdogStatus = self.qosSliceHwStatus_.portWatchdogStatus.get( intfName )
      idx = self.mibHelper_.ifIndexDict_.get( intfName, None )
      if idx is None:
         return
      elif portWatchdogStatus:
         self.mibHelper_.addPfcWdTxQueueMibEntry( idx, intfName )
      else:
         self.mibHelper_.delPfcWdTxQueueMibEntry( idx )

class QosSliceHwStatusDirReactor( Tac.Notifiee ):

   notifierTypeName = "Tac::Dir"

   def __init__( self, qosSliceHwStatusDir, mibHelper ):
      Tac.Notifiee.__init__( self, qosSliceHwStatusDir )
      self.mibHelper_ = mibHelper
      self.pfcPortWatchdogStatusReactor_ = {}
      for key in self.notifier_:
         self.handleSlice( key )

   @Tac.handler( 'entityPtr' )
   def handleSlice( self, key ):

      sliceInfo = self.notifier_.get( key )
      sliceInfoEntry = self.notifier_.entryState.get( key )
      if sliceInfo and sliceInfoEntry:
         self.pfcPortWatchdogStatusReactor_[ key ] = \
            PfcPortWatchdogStatusReactor( sliceInfo, self.mibHelper_ )
      else:
         if key in self.pfcPortWatchdogStatusReactor_:
            del self.pfcPortWatchdogStatusReactor_[ key ]

class EthIntfSnmpPlugin( object ):
   def __init__( self, entityManager ):
      t0( "Running EthIntfSnmpPlugin" )

      self.entityManager_ = entityManager
      sysdb = entityManager.root()
      snmpRoot = sysdb.parent[ 'snmp' ]
      ethIntfDir = snmpRoot.mkdir( 'ethIntf' )

      etherLikeMib = ethIntfDir.newEntity( 'EthIntfSnmp::EtherLikeMib',
                                           'etherlikemib' )

      rmonMib = ethIntfDir.newEntity( 'EthIntfSnmp::RmonMib',
                                      'rmonmib' )
      mauMib = ethIntfDir.newEntity( 'EthIntfSnmp::MauMib',
                                     'maumib' )
      ieee8021PfcMib = ethIntfDir.newEntity( 'EthIntfSnmp::Ieee8021PfcMib',
                                             'ieee8021pfcmib' )
      aristaPfcMib = ethIntfDir.newEntity( 'EthIntfSnmp::AristaPfcMib',
                                           'aristapfcmib' )

      self.intfStatusDirReactor_ = None     
      self.counterReaderSm_ = None
      self.mibHelper_ = None
      self.counterDirReactor_ = None
      self.xcvrAllStatusDir_ = sysdb.parent[ 'Snmp' ][ 'XcvrAllNewStatusDir' ]

      mg = entityManager.mountGroup()
      intfConfigDir = mg.mount( 'interface/config/eth/phy/all',
                                'Interface::AllEthPhyIntfConfigDir', 'r' )
      intfStatusDir = mg.mount( 'interface/status/eth/phy/all',
                                'Interface::AllEthPhyIntfStatusDir', 'r' )
      ethIntfCounterWriterStatusDir = mg.mount(
         'interface/ethIntfCounter/writerStatus', 'Tac::Dir', 'ri' )
      xcvrFixedConfigDir = mg.mount(
         'hardware/xcvr/config/all', 'Xcvr::AllConfigDir', 'r' )
      pfcStatus = mg.mount( 'dcb/pfc/status', pfcStatusTypeName, 'r' )
      qosSliceHwStatusDirPath = \
            "cell/" + str( Cell.cellId() ) + "/qos/hardware/status/slice"
      qosSliceHwStatusDir = mg.mount( qosSliceHwStatusDirPath, 'Tac::Dir', 'ri' )

      def _finishMounts():
         # pylint: disable-msg=W0201
         smashEm = SharedMem.entityManager( sysdbEm=entityManager )
         self.counterReaderSm_ = Tac.newInstance(
                                       "Interface::EthIntfCounterReaderSm",
                                       smashEm,
                                       ethIntfCounterWriterStatusDir )
         self.counterReaderSm_.enableLegacyShmemManSupport()
         self.counterReaderSm_.handleInitialized()
         intfCounterDir = self.counterReaderSm_.legacyShmemManCounterAccessor

         self.mibHelper_ = EthIntfSnmpMibHelper( intfConfigDir, intfStatusDir,
                                                 intfCounterDir, etherLikeMib,
                                                 rmonMib, ieee8021PfcMib,
                                                 aristaPfcMib, pfcStatus,
                                                 qosSliceHwStatusDir )
         self.mauMibSm_ = Tac.newInstance( "EthIntfSnmp::MauMibSm",
                                           intfConfigDir, intfStatusDir, 
                                           xcvrFixedConfigDir,
                                           self.xcvrAllStatusDir_, mauMib )

         self.qosSliceHwStatusDirReactor_ = QosSliceHwStatusDirReactor( 
                                             qosSliceHwStatusDir,
                                             self.mibHelper_ )

      mg.close( _finishMounts )

ethIntfSnmpPlugin = None

@Plugins.plugin( requires=( 'snmp/intf', xcvrAllStatusDirSysdbPath ) )
def Plugin( ctx ):
   t0( "EthIntfSnmp Plugin Loading..." )
   entityManager = ctx.entityManager()

   def _callback():
      global ethIntfSnmpPlugin
      ethIntfSnmpPlugin = EthIntfSnmpPlugin( entityManager )

   addStage2MountFunction( _callback )
