# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac, Tracing

# MIB limit
MIB_LIMIT = 64
traceHandle = Tracing.Handle( 'DaemonSnmp' )
t0 = traceHandle.trace0
t8 = traceHandle.trace8

OptionTableIndex = Tac.Type( "GenericAgent::OptionTableIndex" )
DataTableIndex = Tac.Type( "GenericAgent::DataTableIndex" )

class AgentConfigReactor( Tac.Notifiee ):
   notifierTypeName = 'GenericAgent::Config'

   def __init__( self, genericAgentConfig ):
      self.name_ = genericAgentConfig.name
      self.config = genericAgentConfig
      Tac.Notifiee.__init__( self, genericAgentConfig )
      t0( "AgentConfigReactor __init__", self.name_ )
      self.handleEnabled()
      for key in genericAgentConfig.option.keys():
         self.handleOption( key )

   @Tac.handler( 'enabled' )
   def handleEnabled( self ):
      plugin.daemonMib.enabledTable.newMember( self.name_, self.config )
      t8( "AgentConfigReactor handleEnabled", self.name_,
            plugin.daemonMib.enabledTable[ self.name_ ].enabled )

   @Tac.handler( 'option' )
   def handleOption( self, optionKey ):
      if len( optionKey ) <= MIB_LIMIT:
         if optionKey in self.notifier_.option:
            t8( optionKey, 'added to option' )
            plugin.daemonMib.optionTable.newMember(
                  OptionTableIndex( self.name_, optionKey ), self.config )
         else:
            del plugin.daemonMib.optionTable[
                  OptionTableIndex( self.name_, optionKey ) ]
            t8( optionKey, 'removed from option' )

   def close( self ):
      t0("AgentConfigReactor cleanup", self.name_ )
      del plugin.daemonMib.enabledTable[ self.name_ ]
      for optionKey in plugin.daemonMib.optionTable:
         if optionKey.agentName == self.name_:
            t8( 'deleting', optionKey.agentName, optionKey.key )
            del plugin.daemonMib.optionTable[ optionKey ]
      Tac.Notifiee.close( self )


class AgentStatusReactor( Tac.Notifiee ):
   notifierTypeName = 'GenericAgent::Status'

   def __init__( self, genericAgentStatus ):
      self.name_ = genericAgentStatus.name
      self.status = genericAgentStatus
      Tac.Notifiee.__init__( self, genericAgentStatus )
      t0( "AgentStatusReactor __init__", self.name_ )
      self.handleRunning()
      for key in genericAgentStatus.data.keys():
         self.handleData( key )

   @Tac.handler( 'running' )
   def handleRunning( self ):
      plugin.daemonMib.runningTable.newMember( self.name_, self.status )
      t8( "AgentStatusReactor handleRunning", self.name_,
            plugin.daemonMib.runningTable[ self.name_ ].running )

   @Tac.handler( 'data' )
   def handleData( self, dataKey ):
      if len( dataKey ) <= MIB_LIMIT:
         if dataKey in self.notifier_.data:
            t8( dataKey, 'added to data' )
            plugin.daemonMib.dataTable.newMember(
                  DataTableIndex( self.name_, dataKey ), self.status )
         else:
            del plugin.daemonMib.dataTable[
                  DataTableIndex( self.name_, dataKey ) ]
            t8( dataKey, 'removed from data' )

   def close( self ):
      t0("AgentStatusReactor cleanup", self.name_ )
      del plugin.daemonMib.runningTable[ self.name_ ]
      for dataKey in plugin.daemonMib.dataTable:
         if dataKey.agentName == self.name_:
            t8( 'deleting', dataKey.agentName, dataKey.key )
            del plugin.daemonMib.dataTable[ dataKey ]
      Tac.Notifiee.close( self )

class DaemonSnmpPlugin( object ):
   def __init__( self, ctx ):
      entityManager = ctx.entityManager()
      self.entityManager_ = entityManager
      sysdb = entityManager.root()
      snmpRoot = sysdb.parent[ 'snmp' ]
      daemonDir = snmpRoot.mkdir( 'launcher' )
      self.daemonMib = daemonDir.newEntity( 'GenericAgent::DaemonMib', 'daemonMIB' )

      self.daemonConfigCollReactor_ = None
      self.daemonStatusCollReactor_ = None

      mg = entityManager.mountGroup()
      self.daemonConfigDir_ = mg.mount( 'daemon/agent/config', 'Tac::Dir', 'ri' )
      self.daemonStatusDir_ = mg.mount( 'daemon/agent/status', 'Tac::Dir', 'ri' )


      def _finishMounts():
         def _filter( key, daemonDir, collectionReactor ):
            if daemonDir.entityPtr.get( key ) and daemonDir.entryState.get( key ):
               # The daemon agent has been successfully added
               return True
            else:
               # Properly remove entity from Tac::Dir.
               # Currently collectionChangeReactor doesn't play well with Tac::Dir
               # collections because of the need to monitor the multiple collections
               # that Tac::Dir objects utilize.
               # This is BUG157366. 
               reactors_ = collectionReactor.reactors_
               if key in reactors_:
                  reactors_[ key ].close()
                  del reactors_[ key ]

         self.daemonConfigCollReactor_ = Tac.collectionChangeReactor(
               self.daemonConfigDir_.entityPtr, AgentConfigReactor,
               reactorFilter = lambda key: _filter( key,
                  self.daemonConfigDir_, self.daemonConfigCollReactor_ ) )
         self.daemonStatusCollReactor_ = Tac.collectionChangeReactor(
               self.daemonStatusDir_.entityPtr, AgentStatusReactor,
               reactorFilter = lambda key: _filter( key,
                  self.daemonStatusDir_, self.daemonStatusCollReactor_ ) )

      mg.close( lambda: None )
      ctx.callbackIs( _finishMounts )


plugin = None
def Plugin( ctx ):
   global plugin
   plugin = DaemonSnmpPlugin( ctx )
   Tac.dlopen( 'libGenericAgentState.so' )
