#!/usr/bin/env python
# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import Tracing

t0 = Tracing.trace0

defaultNotifHelpTextFmt = "Enable %s trap"
defaultTrapTypeDescFmt = "Enable %s traps"

def isMasterSysdb( entityManager ):
   redundancyStatus = entityManager.redundancyStatus()
   if not ( redundancyStatus.protocol == 'sso' and
             redundancyStatus.mode == 'standby' ):
      return True
   return False

def findParentConfig( config, parentPath ):
   notifConfig = config.notificationConfig[ parentPath[ 0 ] ]
   for trapType in parentPath[ 1 : ]:
      notifConfig = notifConfig.subtype[ trapType ]
   return notifConfig

def registerNotificationType( entityManager,
                              name,
                              description=None,
                              notifications=None,
                              strip=None,
                              enabledDefault=True,
                              delays=None,
                              parentPath=None,
                              subtypes=None ):
   """
   Used by DefaultConfig plug-ins to register notifications.

   @entityManager - The entity manager passed from DefaultConfigPlugins
   @name - Type of trap to register
   @description - Description of trap type; if None, the name will be used instead
   @notifications - List of specific trap tuples for the given trap type.
              Each item in this list is either:
              1) a tuple, which must contain the notification name, and either or
                both of the default state of the trap (used when
                globalNotificationEnabled is default and there are no explicit
                settings for the specific trap or notification type), and a
                custom help text. Thus the following forms are expected:
                - ( name, defaultState )
                - ( name, helpText )
                - ( name, defaultState, helpText )
              2) just the trap name string
              If default state is not provided, the default is True, and if
              custom help text is not provided, it defaults to "enable
              <trapName> trap"
   @strip - The prefix of each notification to 'strip' off before converting
            each notification into a Cli token.
            If none, set the strip to the name of the trap type
   @enabledDefault - The default 'enable' status for the trap type. Possible values:
               * True  - NotificationType is enabled by default
               * False - NotificationType is disabled by default
   @delays - A dictionary of (key, value) pairs where key is a notification name
             and value is a delay time in seconds. Delay is the amount of time
             between the event and the generation of the notification, to allow for,
             e.g., reconvergence before sending linkDown
   @parentPath - An ordered list of strings in which each string names a trap type.
            The list represents the path starting from the root/top-level trap type
            to the current trap type's parent trap type. If the current trap type is
            at the root/top-level (i.e. it should become a member of 
            Snmp::Config::notificationConfig), parentPath will be None.
   @subtypes - A list of dictionaries representing all of the given trap type's
            subtypes. Each dictionary holds information about one subtype. Keys
            are strings that match the label of an argument (e.g., 'name' or 
            'description'). Values are the content of the argument needed to
            register the given trap type in registerNotificationType().
   """

   if not isMasterSysdb( entityManager ):
      return

   NotificationEnabled = Tac.Type( "Snmp::NotificationEnabled" )
   config = entityManager.lookup( 'snmp/config' )
   description = defaultTrapTypeDescFmt % ( description or name )

   if not parentPath:
      # trap type
      notifConfig = config.newNotificationConfig( name, description )
   else:
      # subtype
      parentConfig = findParentConfig( config, parentPath )
      notifConfig = parentConfig.newSubtype( name, description )

   notifConfig.snmpConfig = config

   strip = strip or name
   notifConfig.strip = strip or name

   notifConfig.enabledDefault = enabledDefault
   notifConfig.enabled = NotificationEnabled.notifDefault

   if notifications:
      registerNotifications( entityManager, name, notifications, delays, parentPath )

   if subtypes:
      for subtype in subtypes:
         registerNotificationType( entityManager,
                                   subtype.get( 'name' ),
                                   subtype.get( 'description' ),
                                   subtype.get( 'notifications' ),
                                   subtype.get( 'strip' ),
                                   subtype.get( 'enabledDefault', True ),
                                   subtype.get( 'delays' ),
                                   subtype.get( 'parentPath' ),
                                   subtype.get( 'subtypes' ) )

def registerNotifications( entityManager,
                           name,
                           notifications,
                           delays,
                           parentPath=None ):
   if not isMasterSysdb( entityManager ):
      return

   config = entityManager.lookup( 'snmp/config' )
   if not parentPath:
      notifConfig = config.notificationConfig.get( name )
   else:
      parentConfig = findParentConfig( config, parentPath )
      notifConfig = parentConfig.subtype[ name ]

   if notifConfig:
      # Each item in the list of notifications is either a string representing
      # the notification name, or else a tuple of the following forms:
      # ( name, defaultState ), ( name, helpText ), or
      # ( name, defaultState, helpText )
      for notificationIter in notifications:
         enabled = True
         helpText = None
         if isinstance( notificationIter, tuple ):
            if len( notificationIter ) == 3:
               # Must be ( name, defaultState, helpText )
               notification, enabled, helpText = notificationIter
               assert isinstance( enabled, bool )
               assert isinstance( helpText, str )
            elif len( notificationIter ) == 2:
               # Could be ( name, defaultState ) or ( name, helpText )
               notification, second = notificationIter
               if isinstance( second, bool ):
                  enabled = second
               elif isinstance( second, str ):
                  helpText = second
               else:
                  assert False, "Second item must be bool or str in 2-tuple"
            else:
               assert False, "Notification tuple must be length 2 or 3"
         else:
            notification = notificationIter

         if not helpText:
            helpText = defaultNotifHelpTextFmt % notification

         specNotif = notifConfig.newNotification( notification, helpText )
         specNotif.enabledDefault = enabled
         if delays and notification in delays:
            notifConfig.delay[ notification ] = delays[ notification ]
   else:
      t0( "NotificationType %s is not registered. registerNotifications Failed",
          name )
