#!/usr/bin/env python
# Copyright (c) 2008-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import hashlib
import re

# Contains number of hex digits required for each key type supported
# for localized users
_userKeyLenMap = {
      'md5' : 32,
      'sha' : 40,
      'sha224' : 56,
      'sha256' : 64,
      'sha384' : 96,
      'sha512' : 128,
      'des' : 32,
      'aes' : 32,
      'aes192' : 48,
      'aes256' : 64
}
def userKeyLen( keyType ):
   """Returns number of hex digits required for the specified user key type.
      Examples of key type: 'md5', 'sha', 'sha(224|256|384|512)', etc."""
   return _userKeyLenMap[ keyType ]

def _fromhex( s ):
   r = []
   for i in xrange( 0, len( s ), 2 ):
      substr = s[ i : i + 2 ]
      r.append( chr( int( substr, 16 ) ) )
   return "".join( r )

def _hasher( authType ):
   if authType == 'md5':
      hasher = hashlib.md5           # pylint: disable-msg=E1101
   elif authType == 'sha':
      hasher = hashlib.sha1          # pylint: disable-msg=E1101
   elif authType == 'sha224':
      hasher = hashlib.sha224        # pylint: disable-msg=E1101
   elif authType == 'sha256':
      hasher = hashlib.sha256        # pylint: disable-msg=E1101
   elif authType == 'sha384':
      hasher = hashlib.sha384        # pylint: disable-msg=E1101
   elif authType == 'sha512':
      hasher = hashlib.sha512        # pylint: disable-msg=E1101
   else:
      assert not "Unknown authType"
      hasher = None
   return hasher

def keyFromPassphrase( phrase, authType ):
   """Generates a key from a passphrase as describe in RFC 2574 section A.2.
   authType should be 'md5', 'sha', or 'sha(224|256|384|512)'."""
   hasher = _hasher( authType )
   hash = hasher()            # pylint: disable-msg=W0622
   bytesLeft = 1048576
   concat = phrase
   while len( concat ) < 64:
      concat += phrase
   concatLen = len( concat )
   while bytesLeft > concatLen:
      hash.update( concat )
      bytesLeft -= concatLen
   if bytesLeft > 0:
      hash.update( concat[ 0 : bytesLeft ] )
   ku = hash.hexdigest()
   return ku

def localizePassphrase( phrase, authType, engineId, privType=None ):
   """Performs a key localization as described in RFC 2574 section 2.6.
   authType should be 'md5', 'sha', or 'sha(224|256|384|512)'.
   privType specifies the privacy protocol, like 'des', 'aes',
   'aes192' or 'aes256'.  This is needed when we need to create
   a longer kul for a stronger privacy protocol."""
   ku = _fromhex( keyFromPassphrase( phrase, authType ) )
   hasher = _hasher( authType )
   localHash = hasher()
   localHash.update( ku + _fromhex( engineId ) + ku )
   kul = localHash.hexdigest()
   if privType is not None:
      bitsRequired = {
            'des': 128,
            'aes': 128,
            'aes192': 192,
            'aes256': 256,
            }[ privType.lower() ]
      # From draft-blumenthal-aes-usm-04 section 3.1.2.1
      while len( kul ) * 4 < bitsRequired:
         localHash = hasher()
         localHash.update( _fromhex( kul ) )
         kul = kul + localHash.hexdigest()
      # Truncate in case we have too much.
      kul = kul[ : bitsRequired / 4 ]
   return kul

firstCapRe = re.compile( '(.)([A-Z][a-z]+)' )
allCapRe = re.compile( '([a-z0-9])([A-Z])' )

def convert( name ):
   s1 = firstCapRe.sub( r'\1-\2', name )
   return allCapRe.sub( r'\1-\2', s1 ).lower()

def trapToToken( trap, strip ):
   '''
   Converts stripCamelCaseString to camel-case-string
   Where we remove the regexp 'strip' and convert the remaining
   camel case string
   '''
   trap = re.sub( '^(%s)(.*)' % strip, r'\2', trap )
   return convert( trap )
