# Copyright (c) 2007-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import Plugins
import os
import Agent
import Tracing
import PySnmp # pylint: disable-msg=F0401
import QuickTrace
from SnmpDebugUtils import enableNetSnmpDebug

traceHandle = Tracing.Handle( 'Snmp' )
t0 = traceHandle.trace0
t9 = traceHandle.trace9
config = None
status = None
nsConfig = None
debugConfig = None


class SnmpPluginContext( object ):
   def __init__( self, entityManager, snmpConfig, mountCallbacks ):
      self.entityManager_ = entityManager
      self.snmpConfig_ = snmpConfig
      self.mountCallbacks_ = mountCallbacks

   def entityManager( self ):
      return self.entityManager_

   def objectsDisabled( self, objects ):
      '''See SysdbUtil.cpp:SnmpUtil::objectsDisabled'''
      disableConfig = self.snmpConfig_.disableObjectsConfig.get( objects )
      assert disableConfig, "You need to register the %s disableConfig from your " \
            "DefaultConfigPlugin" % objects
      return not disableConfig.enabled

   def callbackIs( self, callback ):
      # An ordered list of callbacks - each plugin registers
      # ctx.callbackIs( func ), and they are called in order
      # of registration.
      self.mountCallbacks_.append( callback )

# Mounts that need to happen after the first stage of mounting of another plugin
_stage2Functions = []

def addStage2MountFunction( function ):
   _stage2Functions.append( function )

class Snmp( Agent.Agent ):
   """The Snmp agent"""
   def __init__( self, entityManager, configfile=None, agentxsocket=None,
                 snmpplugindir=None ):
      t0( "Snmp.__init__" )
      self.rootObject_ = None
      Agent.Agent.__init__( self, entityManager )
      Tac.activityManager.useEpoll = True

      if "QUICKTRACE_DISABLE" not in os.environ:
         qtfile = "%s%s.qt" % ( self.agentName, "-%d" if "QUICKTRACEDIR"
                                not in os.environ else "" )
         # Snmp and its plugins use level 0 for profiling
         # SnmpImpl uses level 2 for select timeout
         QuickTrace.initialize( filename=qtfile,
                                sizes="128,0,32,0,0,0,128,0,0,0",
                                maxStringLen=48 )

      assert ( configfile is None ) or ( agentxsocket is None )
      self.configfile_ = configfile
      self.agentxsocket_ = agentxsocket
      self.snmpplugindir_ = snmpplugindir

      t0( "Creating a new snmpDir named \"snmp\"." )
      self.snmpDir_ = entityManager.root().parent.mkdir( 'snmp' )

      # SnmpPlugins register their "mounts complete" callbacks here.
      # Unlike the callback provided to mg.close(), this list is
      # ordered, so that dependencies that are expressed as
      # part of @Plugin.plugin( requires= ) are respected in
      # these callbacks.
      self.mountCallbacks_ = []

   def getDefaultTrace( self ):
      # Enable smash lifecycle tracing and IpSnmp to detect BUG256235
      return 'Smash*/01,IpSnmp/0,StrataSnmp/0'

   def initSnmpAgent( self ):
      t0( "Entering Snmp initSnmpAgent." )
      if self.configfile_:
         PySnmp.initSnmpMasterAgent( self.configfile_ )
      else:
         if nsConfig.clientRecvBuf > 0:
            PySnmp.configureSnmpBufferSize( True, True, nsConfig.clientRecvBuf )
         if nsConfig.clientSendBuf > 0:
            PySnmp.configureSnmpBufferSize( True, False, nsConfig.clientSendBuf )
         if nsConfig.serverRecvBuf > 0:
            PySnmp.configureSnmpBufferSize( False, True, nsConfig.serverRecvBuf )
         if nsConfig.serverSendBuf > 0:
            PySnmp.configureSnmpBufferSize( False, False, nsConfig.serverSendBuf )
         PySnmp.initSnmpAgentxSubAgent( self.agentxsocket_ )
         PySnmp.initSnmp()

   def initDebug( self ):
      t0( "Entering Snmp initDebug" )
      self.agentRoot_.newEntity( 'Snmp::NetSnmpDebug', 'netSnmpDebug' )
      snmpDebug = debugConfig.subcategory[ 'snmp' ].subcategory[ 'agent' ]
      debug = snmpDebug.messageType[ 'enable' ].enabled
      if 'NETSNMP_DEBUG' in os.environ:
         # NETSNMP_DEBUG is equivalent to the -D command-line option to snmpd; set it
         # to the empty string to enable all net-snmp debugging, or to a
         # comma-separated list of tokens (where a "token" is the first argument to
         # DEBUGMSGTL) to enable debugging for just those tokens.
         enableNetSnmpDebug( os.environ[ 'NETSNMP_DEBUG' ], self.agentRoot_ )
      elif debug:
         tokens = snmpDebug.subcategory[ 'tokens' ].messageType
         debugTokens = ','.join( [ key for key in tokens if tokens[ key ].enabled ] )
         enableNetSnmpDebug( debugTokens, self.agentRoot_ )

   def doInit( self, entityManager ):
      t0( "Entering Snmp doInit." )
      mg = entityManager.mountGroup()
      global config
      config = mg.mount( "snmp/config", "Snmp::Config", "r" )
      global status
      status = mg.mount( "snmp/status", "Snmp::Status", "w" )
      global nsConfig
      nsConfig = mg.mount( "snmp/netSnmpConfig", "Snmp::NetSnmpConfig", "w" )
      global debugConfig
      debugConfig = mg.mount( 'debug/config', 'Debug::Config', 'r' )

      t0( "Creating snmp agent root object" )
      self.rootObject_ = self.agentRoot_.newEntity( 'Snmp::Root', 'root' )
      self.rootObject_.scheduler = self.agentScheduler()
      self.rootObject_.cliRequestDirSm = self.agentCmdRequestDirSm()

      # Creating interface local entities used by SnmpPlugins here.
      # We need to have better infra support to do this in a cleaner way
      # BUG198656 has been filed to track the issue.
      self.createLocalEntity( "AllIntfConfigDir",
            "Interface::AllIntfConfigDir",
            "interface/config/all" )
      self.createLocalEntity( "AllIntfStatusDir",
            "Interface::AllIntfStatusDir",
            "interface/status/all" )
      self.createLocalEntity( "AllEthPhyIntfConfigDir",
            "Interface::AllEthPhyIntfConfigDir",
            "interface/config/eth/phy/all" )
      self.createLocalEntity( "AllEthPhyIntfStatusDir",
            "Interface::AllEthPhyIntfStatusDir",
            "interface/status/eth/phy/all" )
      self.createLocalEntity( "EthLagIntfStatusDir",
            "Interface::EthLagIntfStatusDir",
            "interface/status/eth/lag" )
      self.createLocalEntity( "EthIntfStatusDir",
            "Interface::EthIntfStatusDir",
            "interface/status/eth/intf" )

      self.localEntitiesCreatedIs( True )

      mg.close( self._doneMountingStage1 )

   def _doneMountingStage1( self ):
      self.initDebug()
      status.initialized = False

      t0( "Entering mounting stage 1." )
      mg = self.entityManager.mountGroup()

      pluginPath = None

      if self.snmpplugindir_ is not None:
         t0( "Loading plugins from directory:", self.snmpplugindir_ )
         pluginPath = [ self.snmpplugindir_ ]

      Plugins.loadPlugins( 'SnmpPlugin',
                           context=SnmpPluginContext( self.entityManager, config,
                                                      self.mountCallbacks_ ),
                           pluginPath=pluginPath )
      mg.close( self._doneMountingStage2 )

   def _doneMountingStage2( self ):
      t0( "All stage 1 mounts complete, now calling callbacks" )
      t9( '%s.%s' % ( f.__module__, f.__name__ ) for f in self.mountCallbacks_ )

      for cb in self.mountCallbacks_:
         cb()

      t0( "Entering mounting stage 2" )

      mg = self.entityManager.mountGroup()

      for func in _stage2Functions:
         func()

      mg.close( self._doneMountingStage3 )

   def _doneMountingStage3( self ):
      t0( "Entering mounting stage 3." )
      # Now that all of the mounts have completed, initialize
      # the agent and register all of the scalars and tables.
      self.initSnmpAgent()

      # The DescriptorManagerSm is the means by
      # which the NET-SNMP subagent is integrated into the libfwk
      # runActivities loop.
      self.rootObject_.descriptorManagerSm = ( self.rootObject_.scheduler, )
      # The TrapManagerSm handles the transmission of traps
      # from the queue.
      trapMgr = Tac.singleton( 'Snmp::TrapManager' )
      self.rootObject_.trapManager = trapMgr
      self.rootObject_.trapManagerSm = ( self.rootObject_.scheduler,
                                         trapMgr )
      # AgentCommandCallback setup
      self.rootObject_.snmpShowSchedCallback = ( "", "ShowSched",
                                                 self.rootObject_.scheduler )
      self.rootObject_.cliRequestDirSm.addAgentCommandCallback(
            self.rootObject_.snmpShowSchedCallback )

      contexts = Tac.singleton( 'Snmp::Contexts' )
      self.rootObject_.contexts = contexts
      self.rootObject_.contextSm = ( contexts, )

      if status.coldStart:
         t0( "Variable coldStart is True, handle Snmp agent coldStart." )
         PySnmp.handleSnmpColdStart()
         t0( "Setting variable coldStart to False." )
         status.coldStart = False

      status.initialized = True

   def warm( self ):
      """We are only warm if the AgentX connection is established, meaning
      we actually have a channel for serving requests."""
      r = PySnmp.agentxConnectionEstablished()
      if not r:
         # Allow tests to override the requirement for an
         # AgentX connection.
         r = bool( os.environ.get( 'NO_AGENTX' ) )
      t0( "Snmp.warm() returning %s" % r )
      return r

def main():
   container = Agent.AgentContainer( [ Snmp ], passiveMount=True )
   container.addOption( "-x", "--agentxsocket", action="store",
      default="/var/run/agentx/master",
      help="path for agentx socket (default: %default)",
      agentClass=Snmp )
   container.addOption( "--snmpplugindir", action="store",
                        default=None, agentClass=Snmp )
   container.runAgents()
