#!/usr/bin/env python
# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# This module creates a 'snapshot' (fork/core) core file for the 
# specified Rib/IGP agents and is run within the context of 
# a VRF namespace by connecting to the Rib/IGP agent in that network 
# namespace

import socket, errno
import sys
import RibAgent
import IsisAgt
import OspfAgt
import RipAgt

AGENT_SNAPSHOT_RECV_BUF = 1024

def doSnapshot( agentName ):
   if ( agentName == "ribd" ):
      AGENT_SNAPSHOT_PORT = RibAgent.RIB_SNAPSHOT_PORT
   elif ( agentName == "isis" ):
      AGENT_SNAPSHOT_PORT = IsisAgt.ISIS_SNAPSHOT_PORT
   elif ( agentName == "ospf" ):
      AGENT_SNAPSHOT_PORT = OspfAgt.OSPF_SNAPSHOT_PORT
   elif ( agentName == "rip" ):
      AGENT_SNAPSHOT_PORT = RipAgt.RIP_SNAPSHOT_PORT
   else:
      exit( 1 )

   sockFd = None
   try:
      sockFd = socket.socket()
   except socket.error as e:
      exit( 1 )

   try:
      # a blocking connect socket with low time-out
      sockFd.settimeout( 1.0 )
      sockFd.connect( ( 'localhost', AGENT_SNAPSHOT_PORT ) )
   except socket.error as e:
      sockFd.close()
      exit( 2 )

   sockFd.settimeout( 5.0 )
   while True:
      try:
         data = sockFd.recv( AGENT_SNAPSHOT_RECV_BUF )
         if not data:
            # reached EOF, all data has been received
            sys.stdout.flush()
            break
         sys.stdout.write( getSnapshotCoreFileName( data ) )
         sys.stdout.flush()
      except socket.error as e:
         if e.errno == errno.ETIMEDOUT:
            # we encountered an err.ETIMEDOUT
            # not all data from the sender could be received in time
            sockFd.close()
            exit( 3 )
      except IOError as e:
         # If someone pipes the output of command closes it without reading the
         # whole output, it's fine
         if e.errno == errno.EPIPE:
            sockFd.close()
            exit(0)
   sockFd.close()
   exit( 0 )

def getSnapshotCoreFileName( pid ):
   if pid:
      fileNamePrefix = 'core.%s' % ( pid )
   else:
      return 'Forked process creation error'
   return "Core file with file name prefix '/var/core/%s'" % fileNamePrefix
