#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
#
# This file provides a library for developers to use when writing scripts and other
# tools which need transceiver SMBus device information.
#
# There have been many instances where developers have needed to quickly prototype
# some code which issues SMBus requests to transeivers.  This library exists to
# assist the speed and quality in which we can develop SMBus scripts either for
# customers, for transceiver bringup, or even for transceiver qualification.
#
# This file is installed in EOS and therefore present on any switch running EOS.

from __future__ import absolute_import, division, print_function
import re
import Tac


class TransceiverSmbusDevice( object ):
   """Data object which represents a transceiver SMBus device"""

   def __init__( self, accelId, busId, deviceId,
                 deviceName, linecardNumber=None ):
      """accelId : int
         busId : int
         deviceId : int
         deviceName : string of the form 'EthernetX'
         linecardNumber : int or None
      """
      self.accelId = accelId
      self.busId = busId
      self.deviceId = deviceId
      self.linecardNumber = linecardNumber
      self.deviceName = deviceName


class TransceiverSmbusDeviceMap( object ):
   """Represents a mapping from interface name to SMBus device data

   The class variable 'mapping' is a map of { str : TransceiverSmbusDevice }.
   A key of 'mapping' will be of the form 'EthernetX' on a fixed system, or
   'EthernetX/X' on a modular system.
   """

   def __init__( self ):
      self.mapping = self.__createMapping()

   def __runSmbusShowCmd( self ):
      """Runs 'show platform smbus counters' by invoking the CLI"""
      cmd = [ 'FastCli', '-p', '15' ]
      output = Tac.run( cmd, input='show platform smbus counters',
                        shell=True, stdout=Tac.CAPTURE )
      return output.split( '\n' )

   def __parseLine( self, line ):
      """Parses single line of 'show platfrom smbus counters' output.

      Returns None if the line does not contain a transceiver SMBus path,
      returns a TransceiverSmbusDevice otherwise."""
      # Pre-process the line to make it easier to work with
      line = line.replace( 'Xcvr', 'Ethernet' )

      # Search for 'Ethernet'
      ethSearch = re.search( r'Ethernet?\d+(/\d+)?', line )
      if ethSearch is None:
         return None

      ethName = ethSearch.group( 0 )
      line = line[ len( ethName ): ].strip()

      linecardNum = None
      if line.startswith( 'Linecard' ):
         linecardSearch = re.search( r'Linecard(\d+)/', line )
         linecardNum = int( linecardSearch.group( 1 ) )

      # Search for Smbus address path
      pathSearch = re.search( r'(\d{2})/(\d{2})/(0x50)', line )
      if pathSearch is None:
         # We should always match, but we gate with a 'None' check anyways
         # because there might be some cases I didn't consider where the
         # Smbus address path does not exist and we would rather not crash.
         return None

      pathTuple = pathSearch.groups()
      if pathTuple is not None:
         accelId, busId, deviceId = pathTuple
         accelId = int( accelId )
         busId = int( busId )
         deviceId = int( deviceId, 16 )  # hex
         return TransceiverSmbusDevice( accelId, busId, deviceId,
                                        ethName, linecardNum )
      else:
         return None

   def __createMapping( self ):
      """Returns a mapping from interface name to SMBus device address

      The keys of the returned dictionary will be strings of the format
      'Ethernet1' (or 'Ethernet3/1' or if the system is modular).  The values
      of the dictionary will be objects of type TransceiverSmbusDevice.
      Return type -> { str : TransceiverSmbusDevice }
      """
      smbusDeviceMap = {}
      output = self.__runSmbusShowCmd()

      for line in output:
         xcvr = self.__parseLine( line )
         if xcvr:
            smbusDeviceMap[ xcvr.deviceName ] = xcvr

      return smbusDeviceMap
