#!/usr/bin/env python
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import Tracing
import TacMarco
import SharedMem
import threading

# To allow users to do SmashLazyMount.mountInfo()
# pylint: disable-msg=W0611
from Smash import mountInfo

t3 = Tracing.trace3

# Dictionary of SmashProxies, keys are Smash entity paths
smashProxies = {}
proxyLock = threading.Lock()

class SmashProxy:
   ''' Wraps a Smash entity, mounting that entity upon proxy access. '''
   def __init__( self, shmemEm, entityPath, entityType, entityInfo ):
      self.__dict__[ 'shmemEm_' ] = shmemEm
      self.__dict__[ 'entityPath_' ] = entityPath
      self.__dict__[ 'entityType_' ] = entityType
      self.__dict__[ 'entityInfo_' ] = entityInfo
      self.__dict__[ 'entity_' ] = None

   def __getattr__( self, attr ):
      self.mount_()
      return getattr( self.entity_, attr )

   def __cmp__( self, other ):
      self.mount_()
      return cmp( self.entity_, other )

   def __repr__( self ):
      return '<%s for %s>' % ( 'SmashLazyMount.SmashProxy',
                               '%r' % self.entity_ if self.entity_ else
                               '%s at %s' % ( self.entityType_,
                                              self.entityPath_ ) )

   def __nonzero__( self ):
      self.mount_()
      return bool( self.entity_ )

   def __tac_object__( self ):
      self.mount_()
      return self.entity_

   def mount_( self ):
      if not self.entity_:
         # Perform the real mount
         t3( 'SmashLazyMount.SmashProxy.mount_(): mounting', self.entityPath_ )
         self.__dict__[ 'entity_' ] = self.shmemEm_.doMount( self.entityPath_,
                                                             self.entityType_,
                                                             self.entityInfo_ )
         # Delete the EntityManager reference
         self.__dict__[ 'shmemEm_' ] = None

def mount( entityManager, entityPath, entityType, entityInfo ):
   ''' Returns a Smash proxy for the given entity path, type and info. Any attempt
   to access the proxy will trigger the mount of the entity. If the entity was
   already mounted, it is returned directly, instead of returning a proxy.'''

   # Check if a proxy was already created for this entity.
   # NOTE: We cannot invoke the __nonzero__ operator on the potentially existing
   # proxy, hence the two lookups.
   proxy = None
   fullPath = "%s/%s" % ( entityManager.sysname(), entityPath )
   with proxyLock:
      if fullPath in smashProxies.keys():
         proxy = smashProxies.get( fullPath )
         t3( 'SmashLazyMount.mount(): found _Proxy at ', fullPath )
         # Make sure that the existing proxy has the requested entity type
         assert proxy.entityType_ == entityType, \
            'Trying to lazy-mount %s with type %s but proxy has type %s' % \
            ( entityPath, entityType, proxy.entityType_ )
         # In non-cohab mode, make sure that the existing proxy has
         # the requested mode
         if not entityManager.local():
            assert proxy.entityInfo_.mode == entityInfo.mode, \
               'Trying to lazy-mount %s with mode %s but proxy has mode %s' % \
               ( entityPath, entityInfo.mode, proxy.entityInfo_.mode )
         # If both the existing proxy's mode and the requested mode are 'writer',
         # make sure that the collection infos match
         if proxy.entityInfo_.mode == 'writer' and entityInfo.mode == 'writer':
            assert proxy.entityInfo_.collectionInfo == entityInfo.collectionInfo, \
               'Trying to lazy-mount %s with collection info %s and mode writer' \
               ' but proxy has collection info %s and mode writer' % \
               ( entityPath, entityInfo.collectionInfo,
                 proxy.entityInfo_.collectionInfo )

      # Check if the entity is already mounted
      shmemEm = SharedMem.entityManager( sysdbEm=entityManager )
      entity = shmemEm.getTacEntity( entityPath )
      if entity:
         t3( 'SmashLazyMount.mount(): entity already mounted' )
         # Make sure that the mounted entity has the requested type
         # and its collections are initialized with the requested info
         assert entity.tacType.fullTypeName == entityType, \
            'Trying to lazy-mount %s with type %s but mounted entity has type %s' % \
            ( entityPath, entityType, entity.tacType.fullTypeName )
         if SharedMem.cohabMultithreadMounts():
            # Don't check initialization parameters in multithreading mode
            # See doMount() in TacSharedMem/EntityManager.tin for details.
            return entity
         for attr in entity.attributes:
            if attr.endswith( 'Control' ):
               try:
                  control = getattr( entity, attr )
               except AttributeError:
                  continue
               if control.tacType.fullTypeName == 'TacSmash::CollectionControl':
                  collectionName = attr[:-( len( 'Control' ) )]
                  assert collectionName in entity.attributes
                  if entityManager.local():
                     # In cohab mode, the actual mode should always be 'writer'
                     assert control.mode() == 'writer', 'Trying to lazy-mount %s' \
                        ' in cohab mode but %s in mounted entity has mode %s' % \
                        ( entityPath, collectionName, control.mode() )
                  else:
                     # In non-cohab mode, make sure the modes match
                     assert control.mode() == entityInfo.mode, 'Trying to lazy-' \
                        'mount %s with mode %s but %s in mounted entity has mode ' \
                        '%s' % ( entityPath, entityInfo.mode, collectionName,
                                 control.mode() )
                  if entityInfo.mode == 'writer':
                     assert entityInfo.collectionInfo, 'Trying to lazy-mount %s as' \
                        ' writer but collection info is missing'
                     if control.maxSize() != 0:
                        reqSize = entityInfo.collectionInfo[ collectionName ].size
                        assert control.maxSize() == reqSize, 'Trying to' \
                           ' lazy-mount %s with size %s for %s but mounted entity' \
                           ' has size %s' % ( entityPath, reqSize, collectionName,
                                              control.maxSize() )
         return entity

      # The entity is not mounted yet (modulo a concurrent thread
      # mounting it somewhere)
      t3( 'SmashLazyMount.mount(): entity not mounted yet' )
      if proxy is None:
         # Create a proxy
         t3( 'SmashLazyMount.mount(): creating SmashProxy for type:', entityType,
             'path:', entityPath, 'info:', entityInfo,
             'local:', entityManager.local() )
         proxy = SmashProxy( shmemEm, entityPath, entityType, entityInfo )
         smashProxies[ fullPath ] = proxy

   return proxy
