# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# A library for finding all SMART capable storage devices and querying those devices
# for various SMART health statistics.

import re
import Tac

REALLOCATIONS = 'Reallocations remaining'
WEAR_LIFE = 'Wear life remaining'
HEALTH = 'Health'

# Not all SMART devices supply the same attributes, even though several have similar
# meanings.
# Tuples ( name, negative ) where negative=True means the measurement is "life used".
# negative=False means the measurement is "life left".
reallocationAttributes = [ ('Reallocate_NAND_Blk_Cnt', False),
                           ('Reallocated_Sector_Ct', False ),
                         ]
wearLifeAttributes = [ ( 'Percent_Lifetime_Used', False ),
                       ( 'Perc_Rated_Life_Used', False ),
                     ]

def parse( mount ):
   matches = re.search( '/mnt/([a-z0-9]*) .*?/dev/([a-z0-9]*)', mount )
   if matches is None:
      return ( None, None )

   name = matches.group( 1 )
   device = matches.group( 2 )

   return ( name, device )

def smartHealthStatus():
   '''Returns a dict of dicts like so:
   {
      'drive': {
                  'Wear life remaining': 0.8,
                  'Reallocations': 1.0,
                  'Health': 'OK',
               },
      'usb1':  {
                  'Health': 'OK',
                },
      ...
   }
   '''

   healths = {}

   for mount in open( '/proc/self/mountinfo' ).readlines():
      if '/mnt/' not in mount:
         continue

      name, device = parse( mount )
      if name is None:
         continue # Failed to parse

      healths[ name ] = {}

      smartArgs = [ '/usr/sbin/smartctl', '--offlineauto=on', '-HA' ]
      if 'usb' in name:
         smartArgs.append( '-d' )
         smartArgs.append( 'scsi' )

      smartArgs.append( '/dev/%s' % device )

      smart = Tac.run( smartArgs, stdout=Tac.CAPTURE, ignoreReturnCode=True,
                       asRoot=True )
      
      for line in smart.split( '\n' ):
         if 'SMART overall-health self-assessment test result' in line:
            # SSDs
            
            smartHealth = line.split( ': ' )[ 1 ]
            if smartHealth == 'PASSED':
               healths[ name ][ HEALTH ] = 'OK'
            else:
               healths[ name ][ HEALTH ] = smartHealth

            continue

         if 'SMART Health Status' in line:
            # The USB drives we have in the lab

            smartHealth = line.split( ': ' )[ 1 ]
            healths[ name ][ HEALTH ] = smartHealth

            continue

         def findAttribute( nameList, line ):
            for attribute in nameList:
               if attribute[ 0 ] in line:
                  # 206 Write_Error_Rate        0x000e   100   100   ...
                  field = float( line.split()[ 4 ] )

                  if attribute[ 1 ]:
                     field = 100 - field

                  return field / 100
            return None

         field = findAttribute( reallocationAttributes, line )
         if field is not None:
            healths[ name ][ REALLOCATIONS ] = field
         field = findAttribute( wearLifeAttributes, line )
         if field is not None:
            healths[ name ][ WEAR_LIFE ] = field

   return healths
