# Copyright (c) 2005-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

"""A parser for a very simple configuration file format.  Each line of the file must
either be empty (except for whitespace) or else contain a binding of the form:

   NAME=VALUE

NAME must contain only letters, numbers and underscores.  VALUE may contain any
characters.  Leading whitespace is ignored, but there must be no space before the
equals sign."""

from __future__ import absolute_import, division, print_function
import re, os

class ParseError( Exception ):
   pass

class SimpleConfigFileDict( object ):
   def __init__( self, filename, createIfMissing=False, autoSync=False ):
      self.filename_ = filename
      self.autoSync_ = autoSync
      if createIfMissing:
         dirPath = os.path.dirname( filename )
         if not os.path.exists( dirPath ):
            os.makedirs( dirPath )
         open( self.filename_, 'a' )

   def __getitem__( self, key ):
      config = self._readConfig()
      return config[ key ]

   def __setitem__( self, key, value ):
      config = self._readConfig()
      config[ key ] = value
      self._writeConfig( config )

   def __delitem__( self, key ):
      config = self._readConfig()
      del config[ key ]
      self._writeConfig( config )

   def keys( self ):
      config = self._readConfig()
      return config.keys()

   def items( self ):
      config = self._readConfig()
      return config.items()

   def __contains__(self, key):
      config = self._readConfig()
      return key in config

   def __iter__(self):
      return iter( self.keys() )

   def get( self, key, default=None ):
      try:
         return self[ key ]
      except KeyError:
         return default

   def has_key( self, key ):
      return key in self

   def pop( self, key, *args ):
      assert len( args ) <= 1
      config = self._readConfig()
      try:
         value = config[ key ]
      except KeyError:
         if args:
            return args[ 0 ]
         raise
      del config[ key ]
      self._writeConfig( config )
      return value

   def _readConfig( self ):
      d = {}
      n = 0
      if not os.path.exists( self.filename_ ):
         return d
      for line in open( self.filename_ ):
         n += 1
         line = line.lstrip()
         if not line or line.startswith( "#" ):
            continue
         m = re.match( "([A-Za-z0-9_]+)=([^\n\r]*)", line )
         if not m:
            raise ParseError( "Syntax error in %s, line %s" % ( self.filename_, n ) )
         ( var, value ) = m.groups()
         d[ var ] = value
      return d

   def _writeConfig( self, d ):
      with open( self.filename_, 'w' ) as f:
         for var in sorted( d.keys() ):
            value = d[ var ]
            f.write( "%s=%s\n" % ( var, value ) )
         if self.autoSync_:
            f.flush()
            os.fsync( f.fileno() )
