# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

'''
Contains functions for adding a signature file to a zip file
such as a SWI or a SWIX, particularly preparing the SWI/SWIX
with a null signature and adding the final signature file to 
the SWI/SWIX. The signature is generated from functions used
in SignatureRequest.py.
'''

from __future__ import absolute_import, division, print_function
import base64
from binascii import crc32
import hashlib
import zipfile

import CRC32Collision
import SignatureRequest
import SwiSignLib
import Tracing

traceHandle = Tracing.Handle( 'SwiSign' )
t0 = traceHandle.trace0

# Constants for signing
SIGN_VERSION = 1
SIGN_HASH = 'SHA-256'
SIGNATURE_MAX_SIZE = 8192

class Signature( object ):
   ''' The contents of the signature file that will go into the zip file
   being signed. '''
   def __init__( self ):
      self.version = SIGN_VERSION
      self.hash = ""
      self.cert = ""
      self.signature = ""
      self.crcpadding = [ 0, 0, 0, 0 ]
      self.offset = 0
      self.nullcrc32 = 0

   def __repr__( self ):
      data = r''
      data += "HashAlgorithm:" + self.hash + "\n"
      data += "IssuerCert:" + self.cert + "\n"
      data += "Signature:" + self.signature + "\n"
      data += "Version:" + str( self.version ) + "\n"
      crcPadding = "CRCPadding:"
      for byte in self.crcpadding:
         crcPadding += "%c" % ( byte & 0xff )
      data += "Padding:"
      # We need to add padding to make the null signature the same length as
      # the actual signature so they will generate the same hash.
      # The padding amount factors in the length of the current signature data,
      # the CRC padding, and the newline character for the padding field.
      paddingAmt = SIGNATURE_MAX_SIZE - len( data ) - len( crcPadding ) - 1
      data += "*" * paddingAmt + "\n"
      data += crcPadding
      assert len( data ) == SIGNATURE_MAX_SIZE
      return data

def addNullSig( swiFile, size ):
   ''' Add a null signature to the zip file, noting the location of the 
   signature and the CRC32 of the resulting file, which will be used to replace
   the signature with the real one later. '''
   data = '\000' * size # pylint: disable-msg=anomalous-backslash-in-string
   nullcrc32 = crc32( data ) & 0xffffffff
   with zipfile.ZipFile( swiFile, 'a', zipfile.ZIP_STORED ) as swi:
      sigFileName = SwiSignLib.signatureFileName( swiFile )
      swi.writestr( sigFileName, data )
      with swi.open( swi.getinfo( sigFileName ) ) as sigFile:
         # pylint: disable-msg=protected-access
         offset = sigFile._fileobj.tell()
   return offset, nullcrc32

def generateHash( swi, hashAlgo, blockSize=65536 ):
   # For now, we always use SHA-256. This is not a user input (we hardcode it in)
   assert hashAlgo == 'SHA-256'
   # pylint: disable-msg=no-member
   sha256sum = hashlib.sha256()
   with open( swi, 'rb' ) as swiFile:
      for block in iter( lambda: swiFile.read( blockSize ), b'' ):
         sha256sum.update( block )
   return sha256sum.hexdigest()

def prepareDataForServer( swiFile, version, swiSignature, product='EOS' ):
   ''' Adds a null signature to the zip file and returns the data to send to a
   signing server to get the real signature. '''
   body = {}
   hashAlgo = SIGN_HASH
   # Add null signature to swiFile (swiSignature is a null signature at this point)
   offset, nullcrc32 = addNullSig( swiFile, len( str( swiSignature ) ) )

   # Update swiSignature with the offset and nullcrc32
   swiSignature.offset = offset
   swiSignature.nullcrc32 = nullcrc32

   # Generate hash of the swi file with the null signature
   hashStr = generateHash( swiFile, hashAlgo )
   hashStr = base64.standard_b64encode( hashStr.decode( 'hex' ) )
   t0( "hashStr is: ", hashStr )
   body = { "version": version,
            "product": product,
            "hash_algorithm": hashAlgo,
            "input_hash": hashStr
          }
   return body

def generateSigFileFromServer( signatureData, swiFile, swiSignature ):
   ''' Replaces the null signature of the zip file with a signature obtained from 
   an external source such as a signing server. '''
   data = SignatureRequest.extractServerData( signatureData )

   # Update the certificate and signature, and hash fields of swiSignature.
   # Fields from the server are returned in unicode and must be converted to
   # byte string for crc padding
   swiSignature.cert = base64.standard_b64encode( data.certificate )
   swiSignature.hash = str( data.hashAlgorithm )
   swiSignature.signature = data.signature

   # Update crc padding for swiSignature to match the null signature.
   # The last 4 bytes of swiSignature is the crc padding of swiSignature.
   nullcrc32 = swiSignature.nullcrc32
   swiSigCrc32 = crc32( str( swiSignature )[ : -4 ] ) & 0xffffffff
   swiSignature.crcpadding = CRC32Collision.matchingBytes( nullcrc32, swiSigCrc32 )

   # Rewrite the swi-signature in the right place, replacing the null signature
   offset = swiSignature.offset
   with open( swiFile, 'r+' ) as outfile:
      outfile.seek( offset )
      outfile.write( str( swiSignature ) )
