# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

""" Implements parse of cli show commands.
Supported in unpriviledged mode:
   - show processes
   - show processes top
   - show uptime
"""
import re, time
import collections
import LinuxCmdConstants
import CliPlugin.ShowProcessesModel

def parseShowUptime( output ):
   model = CliPlugin.ShowProcessesModel.SystemTimeInfo()
   model.currentTime = time.time()
   with open( '/proc/uptime', 'r' ) as f:
      model.upTime = float( f.readline().split()[ 0 ] )
   # split the output string into load info and other info
   fields = output.splitlines()[ 0 ].split( 'load average: ' )
   assert len( fields ) == 2
   # get the number of user logging on
   res = re.search( r'(\d+) user(s)?', fields[ 0 ] )
   assert res
   model.users = int( res.group( 1 ) )
   # get the load average
   model.loadAvg = map( float, fields[ 1 ].split( ', ' ) )
   return model

def convertToSec( timeStr ):
   res = map( int, re.split( r"[-:]", timeStr ) )
   if len( res ) == 4:
      # cumulative CPU time in the format of "DD-HH:MM:SS"
      return ( res[ 0 ] * 3600 * 24 + res[ 1 ] * 3600 + res[ 2 ] * 60 + res[ 3 ] )
   else:
      # cumulative CPU time in the format of "HH:MM:SS"
      return res[ 0 ] * 3600 + res[ 1 ] * 60 + res[ 2 ]

def parseShowProcLine( procLine ):
   processInfo = CliPlugin.ShowProcessesModel.ProcessInfo()
   # devide the line into field of PID, %CPU, %MEM, TT, STAT and the rest
   fields = re.split( r'\s+', procLine.lstrip(), 5 )
   assert len( fields ) == 6
   pid = int( fields[ 0 ] )
   processInfo.cpuPct = float( fields[ 1 ] )
   processInfo.memPct = float( fields[ 2 ] )
   processInfo.ttyName = fields[ 3 ]
   processInfo.state = fields[ 4 ]
   # the rest line starts with STARTED, 
   # in the format of 'weekday month day HH:MM:SS year'
   startTimeStr = fields[ 5 ][ : LinuxCmdConstants.LSTARTED_LENGTH ]
   time_tuple = time.strptime( startTimeStr, "%a %b %d %H:%M:%S %Y" )
   processInfo.startTime = time.mktime( time_tuple )
   # skip started time and get the cumulative CPU time
   restFields = fields[ 5 ][ ( LinuxCmdConstants.LSTARTED_LENGTH + 
                  LinuxCmdConstants.STARTED_LENGTH + 2 ) : ].split( ' ', 1 )
   processInfo.totalActiveTime = convertToSec( restFields[ 0 ] )
   processInfo.cmd = restFields[ 1 ]
   return ( pid, processInfo )

def parseShowProcesses( output ):
   processes = collections.OrderedDict() 
   procList = output.splitlines()
   if len( procList ) > 1:
      for p in procList[ 1: ]:
         pid, pInfo = parseShowProcLine( p )
         processes[ pid ] = pInfo
   return processes

def parseShowTopOnceTaskSum( taskSum ):
   taskSummary = CliPlugin.ShowProcessesModel.ThreadsInfo()
   values = re.findall( r'\d+', taskSum )
   assert len( values ) == 5
   taskSummary.total = int( values[ 0 ] )
   taskSummary.run = int( values[ 1 ] )
   taskSummary.sleep = int( values[ 2 ] )
   taskSummary.stop = int( values[ 3 ] )
   taskSummary.zombie = int( values[ 4 ] )
   return taskSummary

def parseShowTopOnceCpuSum( cpuSumList ):
   cpuInfoDict = collections.OrderedDict()
   for line in cpuSumList:
      cpuSum = CliPlugin.ShowProcessesModel.CpuInfo()
      cpuName, cpuInfo = line.split( ':', 1 )
      values = re.findall( r"\d*\.\d+|\d+", cpuInfo )
      assert len( values ) == 8
      cpuSum.user = float( values[ 0 ] )
      cpuSum.system = float( values[ 1 ] )
      cpuSum.nice = float( values[ 2 ] )
      cpuSum.idle = float( values[ 3 ] )
      cpuSum.ioWait = float( values[ 4 ] )
      cpuSum.hwIrq = float( values[ 5 ] )
      cpuSum.swIrq = float( values[ 6 ] )
      cpuSum.stolen = float( values[ 7 ] )
      cpuInfoDict[ cpuName ] = cpuSum
   return cpuInfoDict

def parseShowTopOnceMemSum( memSum ):
   def getMemInfo( info, scale ):
      memInfo = CliPlugin.ShowProcessesModel.MemInfo()
      values = re.findall(r'\d+', info )
      assert len( values ) == 4
      memInfo.memTotal = int( values[ 0 ] ) * scale
      memInfo.memUsed = int( values[ 1 ] ) * scale
      memInfo.memFree = int( values[ 2 ] ) * scale
      memInfo.memBuffer = int( values[ 3 ] ) * scale
      return memInfo

   memSummary = CliPlugin.ShowProcessesModel.SystemMemSum()
   unitType = memSum[ 0 ].split( ' ', 1 )[ 0 ]
   scale = LinuxCmdConstants.UNIT_TABLE[ unitType ]
   # get physical memory info
   memSummary.physicalMem = getMemInfo( memSum[ 0 ], scale )
   # get virtual memory info
   memSummary.swapMem = getMemInfo( memSum[ 1 ], scale )
   return memSummary

class reWrapper( object ):
   def __init__( self ):
      self._match = None
      self._groups = None
   def match( self, pattern, string ):
      self._match = re.match( pattern, string, re.IGNORECASE )
      return self._match
   def groups( self, index ):
      if self._groups:
         return self._groups[ index ]
      if self._match:
         self._groups = map( int, self._match.groups() )
         return self._groups[ index ]
      return None

def convertToCentiseconds( timeStr ):
   # determin the format of cpuTime
   timeStrFt = reWrapper()
   if timeStrFt.match( r'(\d+):(\d+).(\d+)', timeStr ):
      # timeStr is in the format of mimute:second.centisecond
      return ( 100 * ( timeStrFt.groups( 0 ) * 60 + timeStrFt.groups( 1 ) ) 
               + timeStrFt.groups( 2 ) )
   if timeStrFt.match( r'(\d+):(\d+)', timeStr ):
      # timeStr is in the format of mimute:second
      return 100 * ( timeStrFt.groups( 0 ) * 60 + timeStrFt.groups( 1 ) )

   if timeStrFt.match( r'(\d+),(\d+)', timeStr ):
      # timeStr is in the format of hour,mimute
      return 100 * ( timeStrFt.groups( 0 ) * 3600 + timeStrFt.groups( 1 ) * 60 )

   if timeStrFt.match( r'(\d+)h', timeStr ):
      # timeStr is in the format of hours
      return 100 * timeStrFt.groups( 0 ) * 3600

   if timeStrFt.match( r'(\d+)d', timeStr ):
      # timeStr is in the format of days
      return 100 * timeStrFt.groups( 0 ) * 3600 * 24

   if timeStrFt.match( r'(\d+)w', timeStr ):
   # timeStr is in the format of weeks
      return 100 * timeStrFt.groups( 0 ) * 3600 * 24 * 7
   
   # the format can not be determined
   return -1

def parseShowTopOnceActivity( line ):
   activityInfo = CliPlugin.ShowProcessesModel.ProcessActivityInfo()
   fields = line.split()
   assert len( fields ) > 11
   activityInfo.userName = fields[ 1 ]
   activityInfo.priority = fields[ 2 ]
   activityInfo.niceValue = int( fields[ 3 ] )
   activityInfo.virtMem = fields[ 4 ]
   activityInfo.residentMem = fields[ 5 ]
   activityInfo.sharedMem = fields[ 6 ]
   activityInfo.status = fields[ 7 ]
   activityInfo.cpuPctType = ( '{:.1f}' if re.search( r'\.', fields[ 8 ] )
                                        else '{:5.0f}' )
   activityInfo.cpuPct = float( fields[ 8 ] )
   activityInfo.memPct = float( fields[ 9 ] )
   activityInfo.activeTime = convertToCentiseconds( fields[ 10 ] )
   cmdIndex = line.find( fields[ 11 ] )
   activityInfo.cmd = line[ cmdIndex: ].rstrip()
   return int( fields[ 0 ] ), activityInfo

def parseShowTopOnce( output ):
   model = CliPlugin.ShowProcessesModel.SystemTopInfo()
   # split output into summary and process info
   parts = output.split( '\n\n' )
   assert len( parts ) == 2
   summary = parts[ 0 ].splitlines()
   assert len( summary ) > 4
   model.timeInfo = parseShowUptime( summary[ 0 ] )
   model.threadsStateInfo = parseShowTopOnceTaskSum( summary[ 1 ] )
   # In an SMP environment, each line can reflect individual CPU state percentages.
   model.cpuInfo = parseShowTopOnceCpuSum( summary[ 2 : -2 ] )
   model.memInfo = parseShowTopOnceMemSum( summary[ -2: ] )

   activitiesInfo = collections.OrderedDict()
   activities = parts[ 1 ].splitlines()[ 1: ]
   for act in activities:
      pid, actInfo = parseShowTopOnceActivity( act )
      activitiesInfo[ pid ] = actInfo
   model.processes = activitiesInfo
   return model
