# Copyright (c) 2005-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import collections
import errno
import os
import signal
import simplejson
import subprocess
import sys
import tempfile

import ArPyUtils
import BasicCliUtil
import CapiCliCommon
import CliCommand
import CliCommon
import CliAaa
import CliMatcher
import CliModel
import CliParser
import EbnfParser
import FileCliUtil
import Tac
import TacSigint
import Tracing
import Url

th = Tracing.defaultTraceHandle()
t0 = th.trace0
t1 = th.trace1
t2 = th.trace2
t3 = th.trace3
t7 = th.trace7
t8 = th.trace8

#-------------------------------------------------------------------------------
# 'show' commands and output filtering.
#-------------------------------------------------------------------------------
# We use 'sh' as an alternate for 'show' so people can still type 'sh xxx'
# even when there are other commands starting with 'sh', such as 'shutdown'.
showKwMatcher = CliMatcher.KeywordMatcher( 'show',
      helpdesc='Display details of switch operation',
      alternates=[ 'sh' ], common=True )
pipeKwMatcher = CliMatcher.KeywordMatcher( '|',
      helpdesc='Command output pipe filters' )
noMoreKwMatcher = CliMatcher.KeywordMatcher( 'no-more',
      helpdesc='Disable pagination for this command' )

# redirect related matchers
redirectKwMatcher = CliMatcher.KeywordMatcher( 'redirect',
      helpdesc='Redirect output to URL' )
redirectPlusKwMatcher = CliMatcher.KeywordMatcher( '>',
      helpdesc='Redirect output to URL' )
redirectUrlMatcher = Url.UrlMatcher( lambda fs: fs.supportsWrite(),
      'Destination file name' )

# append related matchers
appendKwMatcher = CliMatcher.KeywordMatcher( 'append',
      helpdesc='Append redirected output to URL' )
appendPlusKwMatcher = CliMatcher.KeywordMatcher( '>>',
      helpdesc='Append redirected output to URL' )
slashAppendKwMatcher = CliMatcher.KeywordMatcher( '/append',
      helpdesc='Copy and append output to URL' )
appendUrlMatcher = Url.UrlMatcher( lambda fs: fs.supportsAppend(),
      'Destination file name' )

# tee related matchers
teeKwMatcher = CliMatcher.KeywordMatcher( 'tee',
      helpdesc='Copy output to URL' )
teeUrlMatcher = Url.UrlMatcher( lambda fs: fs.supportsWrite(),
      'Destination file name',
      notAllowed=[ '/append' ] )

# The following is done so we can use partial built-in filter commands
# (such as 'inc').
# Note, we purposely do not put 'nz' here so people can use
# 'show xxx | nz | ...' like a non-builtin command (BUG14184).
# nz has no parameters so this is unambiguous, plus it's a short
# command so there is not much of an advantage in excluding anything
# as its prefix.
excludedKeywords = (
   'include', 'exclude', 'begin', 'redirect', 'append', 'no-more', 'tee', 'section',
   'json' )
progNamePattern = ''.join( BasicCliUtil.notAPrefixOf( keyword )
                           for keyword in excludedKeywords )

progNameMatcher = CliMatcher.PatternMatcher(
   progNamePattern + '.+',
   helpname='LINE',
   partialPattern=progNamePattern + '.*',
   helpdesc="Filter command by common Linux tools such as grep/awk/sed/wc",
   priority=CliParser.PRIO_LOW )

# " | json [ revision <num> | version <num> ]": whether to display the output in
# json and if so according to which revision (default is latest). Originally there
# was no revision, so displayJson was True or None, now it can also be an int
# or a string (which is a covert way of indicating it is a version, not a rev).
# standalone 'no-more' also supported
class PipeJson( CliCommand.CliExpression ):
   expression = ( '<internal_jsonKw> '
                        '[ ( <internal_revisionKw> <internal_revision> ) | '
                          '( <internal_versionKw> <internal_version> ) ]' )
   data = {
            '<internal_jsonKw>': CliMatcher.KeywordMatcher( 'json',
                                   helpdesc='Produce JSON output for this command' ),
            '<internal_revisionKw>': CliMatcher.KeywordMatcher( 'revision',
                                   helpdesc='specify revision to format output in' ),
            '<internal_revision>': CliMatcher.IntegerMatcher( 1, 100,
                                       helpdesc='the revision to format output in' ),
            '<internal_versionKw>': CliMatcher.KeywordMatcher( 'version',
                                    helpdesc='specify version to format output in' ),
            '<internal_version>': CliMatcher.IntegerMatcher( 1, 100,
                                       helpdesc='the version to format output in' ),
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      if '<internal_jsonKw>' in args:
         revision = None
         version = None
         if '<internal_revision>' in args:
            revision = int( args[ '<internal_revision>' ] )
         if '<internal_version>' in args:
            version = int( args[ '<internal_version>' ] )
         args[ '<internal_jsonValue>' ] = JsonValue( True, revision, version )
      else:
         args[ '<internal_jsonValue>' ] = JsonValue( False, None, None )

unprivPipeOptions = {
   'include': 'Print lines matching the given pattern',
   'exclude': 'Do not print lines matching the given pattern',
   'begin': 'Start output at the first matching line',
   'section': 'Include sections that match'
}

class UniversalPipeExp( CliCommand.CliExpression ):
   expression = '( <internal_unprivPipe> <internal_pipePattern> ) | <internal_nz>'
   data = {
            '<internal_unprivPipe>': CliMatcher.EnumMatcher( unprivPipeOptions ),
            '<internal_pipePattern>': CliMatcher.StringMatcher( helpname='WORD',
                                                     helpdesc='Regular Expression' ),
            '<internal_nz>': CliCommand.Node( # TODO: dasturias@ regress of BUG14184
                                       matcher=CliMatcher.KeywordMatcher( 'nz',
                                         helpdesc='Include only non-zero counters' ),
                                       alias='<internal_unprivPipe>' )
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      unprivPipeType = args.get( '<internal_unprivPipe>' )
      if not unprivPipeType:
         return

      unprivPipeCmd = args.get( '<internal_pipePattern>' )
      if unprivPipeType == 'include':
         filterCommand = [ 'grep', '-aEe', unprivPipeCmd ]
      elif unprivPipeType == 'exclude':
         filterCommand = [ 'grep', '-aEve', unprivPipeCmd ]
      elif unprivPipeType == 'begin':
         filterCommand = [ 'grep', '-aE', '-A', '1000000', '-e', unprivPipeCmd ]
      elif unprivPipeType == 'section':
         filterCommand = [ 'section', unprivPipeCmd ]
      elif unprivPipeType == 'nz':
         filterCommand = [ 'nz' ]
      else:
         assert False, 'Unknown pipe %s' % unprivPipeType

      assert '<internal_filterCommand>' not in args
      args[ '<internal_filterCommand>' ] = filterCommand

class EnablePipeExp( CliCommand.CliExpression ):
   expression = ( '<internal_unprivPipeExpr> | '
                  '( <internal_progName> [ <internal_progArgs> ] ) |'
                  '( <internal_append> <internal_appendUrl> ) | '
                  '( <internal_tee> <internal_slashAppend> <internal_appendUrl>  ) |'
                  '( <internal_redirect> <internal_redirectUrl> ) | '
                  '( <internal_tee> <internal_teeUrl> )' )
   data = {
            '<internal_unprivPipeExpr>': UniversalPipeExp,
            '<internal_progName>': progNameMatcher,
            '<internal_progArgs>': CliMatcher.StringMatcher( helpname='LINE',
                                                helpdesc='Filter command pipeline' ),
            '<internal_append>': appendKwMatcher,
            '<internal_appendUrl>': appendUrlMatcher,
            '<internal_slashAppend>': slashAppendKwMatcher,
            '<internal_redirect>': redirectKwMatcher,
            '<internal_redirectUrl>': redirectUrlMatcher,
            '<internal_tee>': teeKwMatcher,
            '<internal_teeUrl>': teeUrlMatcher
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      progName = args.get( '<internal_progName>' )
      if not progName:
         return

      assert '<internal_filterCommand>' not in args
      cmd = progName + ' ' + args.get( '<internal_progArgs>', '' )
      args[ '<internal_filterCommand>' ] = [ '/bin/sh', '-c', cmd ]

# 'unprivPipeExpr' contains the filter commands that are allowed from
# unprivileged mode. Every token added here must be included in the
# progNamePattern variable above so that it will not match in both
# the unprivExpr rule and the expression rule (notice the '!' (negation)
# characters in those regular expressions).
#
class UnprivExpr( CliCommand.CliExpression ):
   # <pipe> [ json [ <rev> ] <pipe> ] [ no-more <pipe> ] <unprivPipeExpr>
   # <pipe> json [ <rev> ] [ <pipe> no-more ]
   # <pipe> no-more
   expression = ( '( <internal_pipe1> '
                     '[ <internal_jsonPipe> <internal_pipe2> ]'
                     '[ <internal_noMore> <internal_pipe3> ] '
                     '<internal_unprivPipeExpr> ) |'
                  '( <internal_pipe1> <internal_jsonPipe> '
                     '[ <internal_pipe2> <internal_noMore> ] ) | '
                  '( <internal_pipe1> <internal_noMore> )' )
   data = {
            '<internal_pipe1>': pipeKwMatcher,
            '<internal_pipe2>': pipeKwMatcher,
            '<internal_pipe3>': pipeKwMatcher,
            '<internal_jsonPipe>': PipeJson,
            '<internal_noMore>': noMoreKwMatcher,
            '<internal_unprivPipeExpr>': UniversalPipeExp
          }

#####################################################################################
# Rules for redirecting the output of show commands to a file.  Note that the
# implementation of this behavior is contained in the 'runShowCommand' function in
# BasicCli.py.
# show ... | redirect <url>
# show ... | append <url>
# show ... | tee [/append] <url>
# redirect: show ... >  <url>
# append:   show ... >> <url>
#####################################################################################
class PrivExpr( CliCommand.CliExpression ):
   # <pipe> [ json [ <rev> ] <pipe> ] [ no-more <pipe> ] <privPipeExpr>
   # <pipe> json [ <rev> ] [ <pipe> no-more ]
   # <pipe> no-more
   # >> <url>
   # > <url>
   expression = ( '( <internal_pipe1> '
                     '[ <internal_jsonPipe> <internal_pipe2> ]'
                     '[ <internal_noMore> <internal_pipe3> ] '
                     '<internal_privPipeExpr> ) |'
                  '( <internal_pipe1> <internal_jsonPipe> '
                     '[ <internal_pipe2> <internal_noMore> ] ) | '
                  '( <internal_pipe1> <internal_noMore> ) | '
                  '( <internal_redirectPlus> <internal_redirectUrl> ) |'
                  '( <internal_appendPlus> <internal_appendUrl> )' )
   data = {
            '<internal_pipe1>': pipeKwMatcher,
            '<internal_pipe2>': pipeKwMatcher,
            '<internal_pipe3>': pipeKwMatcher,
            '<internal_jsonPipe>': PipeJson,
            '<internal_noMore>': noMoreKwMatcher,
            '<internal_privPipeExpr>': EnablePipeExp,
            '<internal_redirectPlus>': redirectPlusKwMatcher,
            '<internal_redirectUrl>': redirectUrlMatcher,
            '<internal_appendPlus>': appendPlusKwMatcher,
            '<internal_appendUrl>': appendUrlMatcher,
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      if '<internal_redirectUrl>' in args:
         url = args[ '<internal_redirectUrl>' ]
         assert '<internal_filterCommand>' not in args
         args[ '<internal_filterCommand>' ] = RedirectValue( False, False, url )
      elif '<internal_appendUrl>' in args:
         url = args[ '<internal_appendUrl>' ]
         assert '<internal_filterCommand>' not in args
         tee = '<internal_tee>' in args
         args[ '<internal_filterCommand>' ] = RedirectValue( True, tee, url )
      elif '<internal_teeUrl>' in args:
         url = args[ '<internal_teeUrl>' ]
         assert '<internal_filterCommand>' not in args
         args[ '<internal_filterCommand>' ] = RedirectValue( False, True, url )

PipeValue = collections.namedtuple( 'PipeValue', 'noMore jsonValue filterCommand' )
JsonValue = collections.namedtuple( 'JsonValue', 'displayJson revision version' )
RedirectValue = collections.namedtuple( 'RedirectValue', 'append tee url' )

def _popenPagerProc():
   """Start pager as a separate subprocess"""
   # Fix for BUG24334. Flush stdout before creating a subprocess
   # that could print to stdout first.
   sys.stdout.flush()

   pagerEnviron = os.environ.copy()
   pagerEnviron[ 'LESSSECURE' ] = '1'
   # To allow printing of unicode characters.
   pagerEnviron[ 'LESSCHARSET' ] = 'utf-8'
   pagerCommand = [ 'less', '-d', '-X', '-E', '-K', '-Ps --More-- ' ]
   # We need -X because "LESS_IS_MORE=1 less" has a bug where it
   # emits the rmcup terminfo sequence, even though it never
   # emitted the smcup sequence.  This causes various vt100
   # emulators to put the cursor in an undesirable place,
   # particularly if the alternate screen has not been used
   # since the window was resized.

   pagerProc = subprocess.Popen( pagerCommand,
                                 env=pagerEnviron,
                                 stdin=subprocess.PIPE,
                                 stdout=sys.stdout,
                                 stderr=sys.stderr,
                                 close_fds=True )
   return pagerProc

def _reset_sigpipe_handler():
   # Reset SIGPIPE handler to default (terminate), so the filter command
   # won't print "Broken pipe" when the pager or a downstream process exits
   # prematurely.
   # Without this, 'show xxx | grep yyy | no_such_app' would generate a lot
   # of such error messages.
   signal.signal( signal.SIGPIPE, signal.SIG_DFL )

def _popenFilterProc( filterCommand, downStreamProc ):
   """Start filter as a separate subprocess"""
   # Fix for BUG24334. Flush stdout before creating a subprocess
   # that could print to stdout first.
   sys.stdout.flush()

   filterEnviron = os.environ.copy()
   filterArgs = dict( args=filterCommand,
                      env=filterEnviron,
                      preexec_fn=_reset_sigpipe_handler,
                      stdin=subprocess.PIPE,
                      stdout=sys.stdout,
                      stderr=sys.stderr,
                      close_fds=True )
   if downStreamProc:
      # redirect output down the pipeline
      filterArgs[ 'stdout' ] = downStreamProc.stdin
   filterProc = subprocess.Popen( **filterArgs )
   return filterProc

def _closePipedProc( subProc ):
   """Close subprocess that was started as a part of 'show' command pipeline."""
   returnCode = None
   if not subProc:
      return None
   try:
      # first, close stdin
      subProc.stdin.close()
   except ( OSError, IOError ), e:
      if e.errno == errno.EPIPE:
         # This happens when the filter program closes its stdout before
         # I close my end of the pipe, which I can safely ignore.
         pass
      else:
         raise
   finally:
      subProc.wait()
      returnCode = subProc.returncode

   return returnCode

def checkCliModel( mode, valueFuncResult, modelExpected ):
   """Check the result from the value function to ensure that it is a correct model.

   We expect the following valid cases:
     1) The result is None and either of
       a) the cliModel are both None
      -or-
       b) we logged an error
     2) The result is an instance of cliModel

   Args:
     - valueFuncResult: The result of the value function
     - modelExpected: The type of model that we expect the value function to return

   Returns:
     True if this is the model we expect, False otherwise.
   """
   return ( ( valueFuncResult is None and ( modelExpected is None
                                            or mode.session.hasError() ) )
            or ( valueFuncResult is not None and modelExpected is not None
                 and isinstance( valueFuncResult, modelExpected ) ) )

def _shouldCheckModel( valueFuncResult, cliModel ):
   """ We should type check the models if a person didn't register a cliModel or the
   model is a subclass of CliModel.UncheckedModel

   Keyword Arguments:
   valueFuncResult -- The result of the value function
   cliModel -- The cli model class that was registered with the show command

   Returns -- bool, should we typecheck this model
   """
   return not ( cliModel and issubclass( cliModel, CliModel.UncheckedModel ) )

class JsonFormat( object ):
   """
   Set the session's output format to "json" and restore to "text" on exit.
   """
   def __init__( self, session ):
      self.session = session

   def __enter__( self ):
      self.session.outputFormat_ = "json"
      self.session.shouldPrintIs( False )

   def __exit__( self, _type, _value, _traceback ):
      self.session.outputFormat_ = "text"
      self.session.shouldPrintIs( True )

def _setRequestedModelRevision( mode, jsonValue, cliModel ):
   """
   Store requested revision into session, where the command handler can later pick
   it up via mode.modelRevision(). In case of input through Capi, the session's
   revision and version are set from the jsonrpc in CliApi; in case of input through
   cli we get that info via the jsonValue variable which is a named tuple
   Version 0 is interpreted as 'latest rev', and is set as
   such in CliApi already by capi, and revision will be None if not set (in which
   case the version is used to determine the rev).
   Any version number will be translated into its corresponding revision (based on
   the model) and thereafter we only care about the revision.
   Invalid revision numbers will assert.
   """
   if not cliModel:
      return
   revision = 1
   if not mode.session.shouldPrint(): # meaning it is capi
      revision = cliModel.getRevision( mode.session.requestedModelVersion_,
                                       mode.session.requestedModelRevision_ )
   else: # this is cli
      if jsonValue.displayJson:
         if jsonValue.version is not None:
            revision = cliModel.getRevision( jsonValue.version )
         elif jsonValue.revision is not None:
            revision = cliModel.getRevision( 1, jsonValue.revision )
         else:
            revision = cliModel.getRevision( CliCommon.JSONRPC_VERSION_LATEST )
   mode.session.requestedModelRevision_ = revision
   mode.session.requestedModelRevisionIsLatest_ = revision == cliModel.__revision__

def _makeCapiResponse( mode, result=None, revision=None ):
   # 400 is same as httplib.BAD_REQUEST but it can't be used here as it requires imp-
   # ort of httplib which imports ssl. This causes Eos/test/CliTests.py to fail in
   # CliWeightWatcher test cause ssl is a memory hog verboten of import in Cli.
   bad_request = 400 # same as httplib.BAD_REQUEST
   code = bad_request if not result else None
   response = CapiCliCommon.CliCommandResponse( code, model=result,
                                                revision=revision,
                                                errors=mode.session_.errors_,
                                                warnings=mode.session_.warnings_,
                                                messages=mode.session_.messages_ )
   response.computeResultDict( streaming=True )
   return response.result

def _runShowCommandFunction( mode, valueFunction, prepareFunction, kwargs ):
   if prepareFunction:
      # If prepare fails, the function should do addError/Warning
      # and raise AlreadyHandledError.
      prepareFunction( mode, **kwargs )
      TacSigint.check()
   return valueFunction( mode, **kwargs )

def runShowCommandFunction( mode, fnWithArgs ):
   valueFunction, prepareFunction, kwargs, _, _ = fnWithArgs
   return _runShowCommandFunction( mode, valueFunction, prepareFunction, kwargs )

class ShowCommandContext( object ):
   """Context manager for running show commands"""

   def __init__( self, mode, filterCommand, noMore, jsonValue, cliModel ):
      self.mode_ = mode
      self.filterCommand_ = filterCommand
      self.noMore_ = noMore
      self.jsonValue_ = jsonValue
      self.cliModel_ = cliModel
      self.showDepth_ = None
      self.filterDepth_ = None
      self.oldStdoutFd_ = None
      self.oldStderrFd_ = None
      self.pagerProc_ = None
      self.filterProc_ = None
      self.redirectFile_ = None
      self.redirectCommand_ = []
      self.shouldPrint_ = mode.session_.shouldPrint()

   def _redirectTempFile( self ):
      try:
         return tempfile.NamedTemporaryFile( dir=CliCommon.CLI_TMP_DIR )
      except ( OSError, IOError ):
         # this is a fallback if we can't use /var/tmp/cli for some reason
         return tempfile.NamedTemporaryFile()

   def _redirectCommand( self, filterCommand ):
      # generate redirect command line
      # returns: redirectCommand, redirectFile
      url = filterCommand.url
      FileCliUtil.checkUrl( url )

      redirectCommand = [ 'redirect' ]
      redirectFile = None

      if filterCommand.tee:
         redirectCommand.append( '--tee' )

      if filterCommand.append:
         redirectCommand.append( '--append' )

      if url.localFilename():
         redirectCommand += [ url.localFilename(), url.url ]
      else:
         # create a temporary file
         try:
            redirectFile = self._redirectTempFile()
            redirectCommand += [ redirectFile.name, "temporary file" ]
         except ( OSError, IOError ), e:
            self.mode_.addError(
               "Error redirecting output to temporary file (%s)"
               % e.strerror )
            if not filterCommand.tee:
               # For tee, we still need to go ahead and print to stdout
               raise CliParser.AlreadyHandledError()
      return redirectCommand, redirectFile

   def replaceFd( self, new, original ):
      saved = None
      if new != original:
         saved = os.dup( original )
         os.dup2( new, original )
      return saved

   def restoreFd( self, saved, original ):
      if saved:
         os.dup2( saved, original )
         os.close( saved )

   def __enter__( self ):
      mode = self.mode_
      session = mode.session_

      # Note that CliPlugin/FileCli.py (in the SysMgr package) augments 'enableExpr'
      # (above) with rules that allow you to redirect or duplicate the output of a
      # command to a file.  In this case, 'filterCommand' will be a tuple, where the
      # first component is a dict with keys 'append' and 'tee', and the second
      # component is a Url object.
      self.showDepth_ = session.showCommandDepth_
      self.filterDepth_ = session.filterCommandDepth_

      # As of now, the only way to distinguish between stdout and
      # file/url filters is by the filterCommand type.
      #
      # There are 3 ways we handle filters
      # 1. filterUsesStdout: Popen the filter process as stdout
      # 2. filterUsesFileOrUrl with localFilename(): Popen 'redirect'
      #    with url.localFileName()
      # 3. filterUsesFileOrUrl with remote URL: Popen 'redirect' with
      #    a temporary file
      filterUsesStdout = ( self.filterCommand_ and
                           isinstance( self.filterCommand_, list ) )
      filterUsesFileOrUrl = ( self.filterCommand_ and
                              isinstance( self.filterCommand_, RedirectValue ) )

      # We output to pager iff:
      # 1. Paging is enabled, and
      # 2. There is no filter, or filter is one of these:
      #    'include', 'exclude', 'begin', 'tee', 'nz', shell command.
      #    For each of these, with the exception of 'tee', filterCommand
      #    is a list, while for 'tee' filterCommand is a tuple, which
      #    makes 'tee' a special case.

      # First check if paging is enabled
      outputToPager = not ( self.noMore_ or
                            session.terminalCtx_.pagerDisabled() or
                            session.disableAutoMore_ or
                            self.showDepth_ )

      # Special check for a tuple filterCommand.
      # Tuple commands are for file/URL output,
      # so they should not write to stdout, unless
      # it is 'tee', which is an exception.
      if filterUsesFileOrUrl:
         ( self.redirectCommand_,
           self.redirectFile_ ) = self._redirectCommand( self.filterCommand_ )
         if not self.filterCommand_.tee:
            outputToPager = False

      if outputToPager:
         # run pager as a separate subprocess
         self.pagerProc_ = _popenPagerProc()

      if filterUsesStdout or filterUsesFileOrUrl:
         try:
            cmdline = self.redirectCommand_ or self.filterCommand_
            # Run filter command as a subprocess, and
            # redirect sys.stdout to its stdin
            self.filterProc_ = _popenFilterProc( cmdline, self.pagerProc_ )
         except EnvironmentError, e:
            mode.addError( "Command %r failed with error: %s" %
                           ( cmdline, e.strerror ) )
            raise CliParser.AlreadyHandledError()

      # Now it's safe to change stdout/stderr
      # save current stdout/stderr
      newProc = self.filterProc_ or self.pagerProc_
      if newProc:
         newProcStdinFd = newProc.stdin.fileno()
         self.oldStdoutFd_ = self.replaceFd( newProcStdinFd,
                                             sys.stdout.fileno() )
         self.oldStderrFd_ = self.replaceFd( newProcStdinFd,
                                             sys.stderr.fileno() )

      session.showCommandDepth_ += 1
      if self.filterCommand_ or outputToPager:
         session.filterCommandDepth_ += 1

      session.displayJsonIs( self.jsonValue_.displayJson )
      if self.jsonValue_.displayJson or not mode.session.shouldPrint():
         mode.session.outputFormat_ = "json"
      else:
         mode.session.outputFormat_ = "text"

      _setRequestedModelRevision( mode, self.jsonValue_, self.cliModel_ )
      return self

   def __exit__( self, _type, value, _traceback ):
      # in case we printed json, reset flag (affects config commands as well)
      self.mode_.session_.shouldPrintIs( self.shouldPrint_ )
      self.mode_.session_.filterCommandDepth_ = self.filterDepth_
      self.mode_.session_.showCommandDepth_ = self.showDepth_

      # restore the original stdout/stderr
      self.restoreFd( self.oldStdoutFd_, sys.stdout.fileno() )
      self.restoreFd( self.oldStderrFd_, sys.stderr.fileno() )

      self.oldStdoutFd_ = self.oldStderrFd_ = None

      # close input of the filter subprocess, wait for it to complete
      retCode = _closePipedProc( self.filterProc_ )

      if self.redirectFile_:
         if not ( retCode and self.redirectCommand_ ):
            # if redirect returns error, do not do anything
            url = self.filterCommand_.url
            try:
               url.put( self.redirectFile_.name, append=self.filterCommand_.append )
            except EnvironmentError, e:
               self.mode_.addError( "Error redirecting output to %s (%s)"
                                    % ( url.url, e.strerror ) )
         self.redirectFile_ = self.redirectCommand_ = None

      # close input of the pager subprocess, wait for it to complete
      _closePipedProc( self.pagerProc_ )

def _runShowCommandInternal( mode, showCmdFunc, jsonValue, cliModel ):
   # If displayJson is True then the rendering needs to be
   # discarded so redirect stdout to /dev/null. This is need
   # for uncoverted commands as these commands print to stdout
   # in their individual implementation of the command.  It
   # also helps to avoid printing unstructured messages like
   # "Waiting for..." for converted commands, which could cause
   # CliTestClient to fail to decode the output.
   if ( cliModel is None and mode.session.outputFormat_ == 'json' ):
      _checkResult( mode, jsonValue, cliModel, None )
      return None

   if jsonValue.displayJson: # true if cli shell and | json
      mode.session_.shouldPrintIs( False ) # dont display errors as % <msg>
      with ArPyUtils.FileHandleInterceptor( [ sys.stdout.fileno(),
                                              sys.stderr.fileno() ]
                                          ) as capturedStdout:
         result = showCmdFunc()
         sys.stdout.flush()
         output = capturedStdout.contents()
      ( wasCliPrinted, retModel ) = _checkCliPrinted( cliModel, result )
      if wasCliPrinted:
         # Handler streamed json directly to stdout for efficiency
         # Validate the output (our context is a cli shell, the http case
         # is validated later in CliApi where json.loads is needed anyway),
         # although we cannot unmarshall older revisions into today's model.
         try:
            output = simplejson.loads( output )
            if ( mode.session.requestedModelRevisionIsLatest_ and
                 CliModel.shouldValidateModel( result ) ):
               CliModel.unmarshalModel( retModel, output, degraded=False )
            print( simplejson.dumps( output, sort_keys=False, indent=4 ) )
         except Exception as e: # pylint: disable-msg=W0703
            print( "JSON ERRORS for:\n", output )
            CliCommon.printErrorMessage( "%s " % str( e ) )
         return result
   # This is the standard cli shell (text) or capi/http (text or json) case.
   # In case of DeferredModel and text, one day we could stealth run in json
   # mode first to force some baking (through unmarshall) of the json path.
   else:
      result = showCmdFunc()
      ( wasCliPrinted, _ ) = _checkCliPrinted( cliModel, result )
      if wasCliPrinted:
         # text/json was printed to stdout, nothing further to validate, if
         # the output was json, it will be fully validated in CliApi (using
         # unmarshall, which covers the ModelMismatch done for regular models
         # further below).
         return result

   _checkResult( mode, jsonValue, cliModel, result )
   return result

def _checkResult( mode, jsonValue, cliModel, result ):
   if not mode.session_:
      return

   if ( _shouldCheckModel( result, cliModel ) and
        not checkCliModel( mode, result, cliModel ) ):
      raise CliModel.ModelMismatchError( result, cliModel )

   if isinstance( result, CliModel.Model ):
      # We got a Model back: validate it first (unconditionally,
      # even though this has a runtime cost and is mostly needed
      # to catch development errors). In case of validation errors,
      # flag them as warnings, which on cli shows up as !<warn>.
      # Test-clients will convert the warning to an error (assert).
      # The only validation is to check for missing mandatory attr.
      if not mode.session.hasError():
         if mode.session.requestedModelRevisionIsLatest_:
            warnings = result.checkOptionalAttributes()
            if warnings:
               for w in warnings:
                  mode.addWarning( w )
      if jsonValue.displayJson:
         nbErrors = len( mode.session_.errors_ )
         revision = None
         if jsonValue.version is not None:
            revision = result.getRevision( jsonValue.version )
         if jsonValue.revision is not None:
            revision = result.getRevision( 1, jsonValue.revision )
         try:
            res = _makeCapiResponse( mode, result, revision=revision )
         except CliCommon.CliModelNotDegradable as e:
            mode.addError( str( e ) )
            return
         # generators can produce errors, in which case restart the response
         if nbErrors != len( mode.session_.errors_ ):
            res = _makeCapiResponse( mode )
         # have to use json module to be able to pretty print
         encoder = simplejson.JSONEncoder( indent=4, sort_keys=False )
         for chunk in encoder.iterencode( res ):
            sys.stdout.write( chunk )
         # Output is always newline terminated
         sys.stdout.write( '\n' )
      elif mode.session_.shouldPrint():
         result._renderOnce() # pylint: disable-msg=W0212
   else:
      # We didn't get a Model back, so this must be an
      # unconverted show command (unless the command actually failed)
      if jsonValue.displayJson:
         if not mode.session_.errors_:
            mode.addError( "This is an unconverted command" )
         res = _makeCapiResponse( mode )
         print( simplejson.dumps( res, sort_keys=False, indent=4 ) )
      elif not mode.session_.shouldPrint() and not mode.session_.errors_:
         # We were supposed to not print anything and just
         # return a model though...
         raise CliCommon.ModelUnsupportedError(
                                       CliCommon.UNCONVERTED_MODEL_MSG )

def _checkCliPrinted( ruleModel, result ):
   # Check if the handler decided to use CliPrint, in which case the text/json output
   # was already produced to stdout. Same (output to stdout) should happen if the
   # model registered with the cli command derives from DeferredModel, in which case
   # the result returned by the show command is not a model instance but is the class
   # of the model (the "schema" for the produced output). CliPrinting show commands
   # do return a model instance though, so we harmonize the 'result' for 'already
   # output to stdout' cases here to be the class (that is, we return a possibly
   # modified result, which is the reason we have to test for __cliPrinted__ first
   # in case people double down and return cliPrinted(model) even for models that
   # are already defined as 'deferred').
   # allow deferredModels to print nothing but set mode.addError()
   # useful in cases like "Error: feature not enabled"
   if not result:
      return ( False, None )
   if result.__dict__.get( "__cliPrinted__" ):
      return ( True, type( result ) )
   if ruleModel and issubclass( ruleModel, CliModel.DeferredModel ):
      return ( True, result )
   return ( False, result )

def runShowCommand( mode, showCmdFunc, filterCommand=None, noMore=None,
                    jsonValue=None, cliModel=None ):
   """Runs a show command, optionally piping the output to another command, such as
   'grep' or 'less', or redirecting or duplicating the output to a file.

   Returns the return value of the function implementing the show command, or
   *returns* an exception (yes, returns, not throws) if the exception is
   considered a *silent* failure (e.g. getting a SIGPIPE while running a command
   that prints something to a pipe that has been closed).
   """
   if jsonValue is None:
      jsonValue = JsonValue( False, None, None )

   with ShowCommandContext( mode, filterCommand, noMore, jsonValue, cliModel ):
      try:
         return _runShowCommandInternal( mode, showCmdFunc, jsonValue, cliModel )
      except ( OSError, IOError ), e:
         if mode.session_.filterCommandDepth_:
            if e.errno == errno.EPIPE:
               # This happens when the filter program closes its stdout before
               # I close my end of the pipe, which I can safely ignore.
               return e
         raise
      except Tac.SystemCommandError, e:
         if mode.session_.filterCommandDepth_:
            # BUG30201. When a Cli handler calls Tac.run() to run another
            # program, it throws this exception when the program exits
            # with a non-zero return code. As shown in BUG30201, this can
            # happen if it is a show command with paging enabled, and
            # 'less' is aborted by entering 'q'. So we catch this exception
            # and ignore it in this case. Unfortunately, there is no way
            # to know how the program has exited (e.g., for python scripts,
            # its return code is simply 1 if it throws an exception),
            # so it's possible we are hiding bugs in this case
            return e
         else:
            raise
      finally:
         try:
            sys.stdout.flush()
         except ( OSError, IOError, EnvironmentError ), e:
            if not e.errno == errno.EPIPE:
               raise

def _showCommandFunction( mode, valueFunction, prepareFunction, kwargs ):
   def showCmdFunc():
      return _runShowCommandFunction( mode, valueFunction, prepareFunction, kwargs )
   return showCmdFunc

def _getPipeAuthzTokens( filterCommand ):
   # We authorize filters for:
   # 1) bash commands
   # 2) built-in filters causing file operations (redirect, append and tee)
   cmd = []
   if isinstance( filterCommand, list ) and len( filterCommand ) > 2 and \
          filterCommand[ 0 : 2 ] == [ "/bin/sh", "-c" ]:
      cmd = filterCommand[ 2 : ]
   elif isinstance( filterCommand, RedirectValue ):
      if filterCommand.tee:
         cmd = [ 'tee', str( filterCommand.url ) ]
      elif filterCommand.append:
         cmd = [ 'append', str( filterCommand.url ) ]
      else:
         cmd = [ 'redirect', str( filterCommand.url ) ]
   if cmd:
      return [ '|' ] + cmd
   else:
      return []

def _authzPipeCmd( mode, filterCommand ):
   if not mode.session_.authorizationEnabled():
      return True

   tokens = _getPipeAuthzTokens( filterCommand )
   if not tokens:
      return True

   privLevel = mode.session_.privLevel_
   t0( "Requesting authorization for filter '%s' in mode %s" % (
         " ".join( tokens ), mode.name ) )
   authorized, message = CliAaa.authorizeCommand( mode, privLevel, tokens )
   if authorized:
      return True
   cmd = " ".join( tokens )
   s = "Authorization denied for filter '%s'" % cmd
   if message:
      s += ": %s" % message
   mode.addError( s )
   return False

def runShowCommandWithAuthz( mode, fnWithArgs, pipeValue ):
   if pipeValue is None:
      pipeValue = PipeValue( False, None, None )

   valueFunction, prepareFunction, kwargs, cliModel, noMore = fnWithArgs
   noMore |= pipeValue.noMore
   execFilter = _authzPipeCmd( mode, pipeValue.filterCommand )
   filterCommand = pipeValue.filterCommand if execFilter else None
   showCmdFunc = _showCommandFunction( mode, valueFunction, prepareFunction, kwargs )
   return runShowCommand( mode, showCmdFunc, filterCommand, noMore,
                          pipeValue.jsonValue, cliModel )

class ShowCommandValueFunction( object ):
   __slots__ = ( 'valueFunction_', 'cliModel_', 'func_name', 'func_code',
                 'prepareFunction_', 'noMore_' )

   def __init__( self, valueFunction, cliModel, prepareFunction=None,
                 noMore=False ):
      self.valueFunction_ = valueFunction
      self.prepareFunction_ = prepareFunction
      self.func_name = valueFunction.func_name
      self.func_code = valueFunction.func_code
      self.cliModel_ = cliModel
      self.noMore_ = noMore

   def __call__( self, mode, **kwargs ):
      return ( self.valueFunction_, self.prepareFunction_,
               kwargs, self.cliModel_, self.noMore_ )

class ShowCommandHandler( object ):
   __slots__ = ( 'handler_', 'cliModel_', 'prepareFunction_', 'noMore_' )

   def __init__( self, handler, cliModel, prepareFunction, noMore ):
      self.handler_ = CliCommand.getBoundedMethod( handler )
      self.cliModel_ = cliModel
      self.prepareFunction_ = ( CliCommand.getBoundedMethod( prepareFunction )
                                if prepareFunction else None )
      self.noMore_ = noMore

   def _parseAndAuthzFilterCommand( self, mode, args ):
      filterCommand = args.get( '<internal_filterCommand>' )
      filterCommand = filterCommand if _authzPipeCmd( mode, filterCommand ) else None
      return filterCommand

   def __call__( self, mode, args ):
      noMore = self.noMore_ | ( '<internal_noMore>' in args )
      jsonValue = args[ '<internal_jsonValue>' ]
      filterCommand = self._parseAndAuthzFilterCommand( mode, args )
      kwargs = { 'args': { k: v for k, v in args.iteritems()
                           if not k.startswith( '<internal_' ) } }
      showCmdFunc = _showCommandFunction( mode, self.handler_,
                                          self.prepareFunction_, kwargs )
      return runShowCommand( mode, showCmdFunc, filterCommand, noMore,
                             jsonValue, self.cliModel_ )

class ShowCliCommandClass( CliCommand._CliCommandBase ):
   '''
   syntax: EBNF Syntax for this command
   handler: Command handler
   data:
   authz: If authorization is enabled if this command gets authorized (default: true)
   syncAcct: Wait for accounting before handler is invoked (default: false)
   authzFunc: Authorization function to authorize this command
   acctFunc: Accounting function to account this command
   hidden: Enable autocomplete and help completion for this command (default: false)
   autoConfigSessionAllowed:
   adapter:
   privileged: If this command is privileged (changes pipe options and modes)
   noMore: Disable the more pipeing always for this command
   cliModel: CliModel returned by this show command
   prepareFunction: Function to be called before the handler is invoked
   '''
   # pylint: disable-msg=protected-access
   ALLOWED_FIELDS = ( 'syntax', 'handler', 'data',
                      'authz', 'syncAcct', 'authzFunc', 'acctFunc',
                      'hidden', 'autoConfigSessionAllowed', 'adapter', 'noMore',
                      'cliModel', 'prepareFunction', 'privileged' )
   REQUIRED_FIELDS = ( 'handler', )

   syntax = None
   privileged = False
   hidden = False
   cmdFilter = True

   allowCache = False # disable cache for show commands

   prepareFunction = None
   noMore = False
   cliModel = None
   cmdHandler = None

   unprivSyntax = None
   privSyntax = None
   unprivSyntaxRootNodes = None
   privSyntaxRootNodes = None

   dynamicSyntax_ = False
   initialized_ = False

   @classmethod
   def initialize( cls ):
      if cls.initialized_:
         return
      t1( "initialize", cls.__name__ )

      if cls.dynamicSyntax_:
         cls._generateSyntaxAndData()

      cls._validateFields()
      privileged = cls.privileged
      data = cls.data
      # when we initialize the class we take the fields that the class with created
      # and then we get to work. This is generally what we are tying to do:
      # 1) Check that the data is valid
      # 2) get all of the relevent adapters (must be done before expanding data)
      # 3) Update the syntaxes (basically expanding the syntax which are expressions)
      # 4) We don't need the Expressions in the data anymore, so also replace those
      # 5) Get the ENBF tree and cmdHandler and create function callbacks of them
      assert 'show' not in data, 'The \'show\' should not be defined'
      tokens = EbnfParser.getTerms( EbnfParser.tokenize( cls.syntax ) )
      if 'show' in tokens:
         data[ 'show' ] = showKwMatcher
      if not cls.privileged:
         cls.unprivSyntax = cls.syntax + ' [ <unprivExpr> ]'
         data[ '<unprivExpr>' ] = UnprivExpr
      cls.privSyntax = cls.syntax + ' [ <privExpr> ]'
      data[ '<privExpr>' ] = PrivExpr
      cls._checkData()
      adapters = cls._expandAdapters()
      if not privileged:
         cls._updateSyntax( 'unprivSyntax' )
      cls._updateSyntax( 'privSyntax' )
      cls.data = cls._expandData()
      # pylint: disable=no-member
      cls.cmdHandler = ShowCommandHandler( cls.handler,
                                           cls.cliModel,
                                           cls.prepareFunction,
                                           cls.noMore )
      if not privileged:
         unprivSyntaxInfo = cls._getSyntaxAndHandler( 'unprivSyntax',
                                                      'cmdHandler' )
      privSyntaxInfo = cls._getSyntaxAndHandler( 'privSyntax', 'cmdHandler' )

      def unprivNodes():
         return cls._getRootNodes( unprivSyntaxInfo[ 0 ],
                                   cls.data,
                                   unprivSyntaxInfo[ 1 ],
                                   cls.hidden,
                                   adapters )

      def privNodes():
         return cls._getRootNodes( privSyntaxInfo[ 0 ],
                                   cls.data,
                                   privSyntaxInfo[ 1 ],
                                   cls.hidden,
                                   adapters )

      if not privileged:
         cls.unprivSyntaxRootNodes = staticmethod( unprivNodes )
      cls.privSyntaxRootNodes = staticmethod( privNodes )

      # pylint: enable=no-member
      cls.initialized_ = True
   # pylint: enable-msg=protected-access
