#!/usr/bin/env python
# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Arnet
import Tac
import re
from SflowConst import defaultPort, sflowDestDefaultMsg
from SflowConst import defaultIp, sflowSourceDefaultMsg
from IpLibConsts import DEFAULT_VRF

ipAddrPattern = "[12]?\d{1,2}\.[12]?\d{1,2}\.[12]?\d{1,2}\.[12]?\d{1,2}"

def cleanupSflowVrfConfig( config, vrfName ):
   vrfConfig = config.sflowVrfConfig.get( vrfName )
   if ( vrfConfig and vrfConfig.switchIpAddr == defaultIp and
        not vrfConfig.srcIntfName and not vrfConfig.collectorHostAndPort ):
      del config.sflowVrfConfig[ vrfName ]

def addConfigCollectorInfo( config, hostname, port=defaultPort,
                            vrfName=DEFAULT_VRF ):
   if vrfName not in config.sflowVrfConfig:
      config.sflowVrfConfig.newMember( vrfName )
   config.sflowVrfConfig[ vrfName ].collectorHostAndPort.addMember(
         Tac.Value( 'Arnet::HostAndPort', hostname=hostname, port=port ) )

def removeConfigCollectorInfo( config, hostname, port=None,
                               vrfName=DEFAULT_VRF ):
   vrfConfig = config.sflowVrfConfig.get( vrfName )
   if vrfConfig:
      hostAndPort = vrfConfig.collectorHostAndPort.get( hostname )
      if hostAndPort:
         # Keyed by hostname, so only check that port matches if it is provided
         if port and port != hostAndPort.port:
            return
         del vrfConfig.collectorHostAndPort[ hostname ]
         cleanupSflowVrfConfig( config, vrfName )

def removeConfigCollectorInfoAll( config, vrfName=DEFAULT_VRF ):
   vrfConfig = config.sflowVrfConfig.get( vrfName )
   if vrfConfig:
      vrfConfig.collectorHostAndPort.clear()
      cleanupSflowVrfConfig( config, vrfName )

def addConfigCollectorInfoDict( config, hostAndPorts,
                                vrfName=DEFAULT_VRF ):
   for hostname, port in hostAndPorts.iteritems():
      addConfigCollectorInfo( config, hostname, port, vrfName )

def verifyCollectorInfoHasElement( collectorInfo, hostname,
   port=defaultPort, only=False ):
   return hostname in collectorInfo \
         and collectorInfo[ hostname ].port == port \
         and ( not only or len( collectorInfo ) == 1 )

def verifyConfigCollectorInfo( collectorHostAndPort, hostAndPortDict ):
   if len( collectorHostAndPort ) != len( hostAndPortDict ):
      return False
   for hostname, port in hostAndPortDict.iteritems():
      if not hostname in collectorHostAndPort:
         return False
      if port != collectorHostAndPort[ hostname ].port:
         return False
   return True

def verifyStatusCollectorInfo( collectorIpAndPort, hostAndIpDict ):
   if len( hostAndIpDict ) != len( collectorIpAndPort ):
      return False
   for hostname, ip in hostAndIpDict.iteritems():
      if not hostname in collectorIpAndPort:
         return False
      if ip != collectorIpAndPort[ hostname ].ip:
         return False
   return True

def addConfigSourceIp( config, sourceIp, vrfName=DEFAULT_VRF ):
   if vrfName not in config.sflowVrfConfig:
      config.sflowVrfConfig.newMember( vrfName )
   config.sflowVrfConfig[ vrfName ].switchIpAddr = sourceIp
   if sourceIp == defaultIp:
      cleanupSflowVrfConfig( config, vrfName )

def removeConfigSourceIp( config, vrfName=DEFAULT_VRF ):
   vrfConfig = config.sflowVrfConfig.get( vrfName )
   if vrfConfig:
      vrfConfig.switchIpAddr = defaultIp
      cleanupSflowVrfConfig( config, vrfName )

def addConfigSourceIntf( config, sourceIntf, vrfName=DEFAULT_VRF ):
   if vrfName not in config.sflowVrfConfig:
      config.sflowVrfConfig.newMember( vrfName )
   config.sflowVrfConfig[ vrfName ].srcIntfName = sourceIntf

def removeConfigSourceIntf( config, vrfName=DEFAULT_VRF ):
   vrfConfig = config.sflowVrfConfig.get( vrfName )
   if vrfConfig:
      vrfConfig.srcIntfName = ''
      cleanupSflowVrfConfig( config, vrfName )

def parseSflowCliOutput( lines, detail=False ):
   lines = lines[ : ]
   blankLines = []
   parsedLines = {}
   curLine = lines.pop( 0 )
   ignoreLines = [ 'waiting for counter update',
                   'Tac.waitFor: waiting for',
                   'Displaying stale counters',
                   'Displaying counters that may be stale' ]
   for ignoreLine in ignoreLines:
      if ignoreLine in curLine:
         curLine = lines.pop( 0 )

   parsedLines[ 'headerLine' ] = curLine
   parsedLines[ 'dashLine' ] = lines.pop( 0 )

   # parsedLines[ 'dstHostAndPort' ] is a list of ( destHostname, destIp, destPort,
   # vrfName ) tuples
   destLines = []
   dest6Lines = []
   curLine = lines.pop( 0 )
   if not sflowDestDefaultMsg in curLine and "Destination" in curLine:
      curLine = lines.pop( 0 )
      while 'Source' not in curLine:
         dstLine = curLine
         # This is for IPv4 source addresses.
         # The output format is either "<hostname> (<ip>):<port>" or "<ip>:<port>"
         m = re.search( "(\S+) \((%s)\):(\d+) \(VRF: (\S+)\)" %
                        ipAddrPattern, dstLine )
         if m:
            destLines.append( (
               m.group( 1 ),
               m.group( 2 ),
               int( m.group( 3 ) ),
               m.group( 4 )
               ) )
         else:
            m = re.search( "(%s):(\d+) \(VRF: (\S+)\)" % ipAddrPattern, dstLine )
            if m:
               destLines.append( (
                  None,
                  m.group( 1 ),
                  int( m.group( 2 ) ),
                  m.group( 3 )
                  ) )

         # This is for IPv6 source addresses.
         # The output format is either "<hostname> (<ip>):<port>" or "[<ip>]:<port>"
         m = re.search( "(\S+) \((%s)\)\:(\d+) \(VRF: (\S+)\)" %
                        Arnet.Ip6AddrRe, dstLine )
         if m:
            dest6Lines.append( (
               m.group( 1 ),
               m.group( 2 ),
               int( m.group( 3 ) ),
               m.group( 4 )
               ) )
         else:
            m = re.search( "\[(%s)\]:(\d+) \(VRF: (\S+)\)" %
                           Arnet.Ip6AddrRe, dstLine )
            if m:
               dest6Lines.append( (
                  None,
                  m.group( 1 ),
                  int( m.group( 2 ) ),
                  m.group( 3 )
                  ) )
         curLine = lines.pop( 0 )
   else:
      curLine = lines.pop( 0 )

   parsedLines[ 'dstHostAndPort' ] = destLines
   parsedLines[ 'dstHost6AndPort' ] = dest6Lines

   # parsedLines[ 'srcIp/srcIp6' ] is a list of ( srcIp, vrfName ) tuples
   srcIpLines = []
   srcIp6Lines = []
   if not sflowSourceDefaultMsg in curLine and "Source" in curLine:
      curLine = lines.pop( 0 )
      while 'Hardware Sample Rate for' not in curLine:
         m = re.search( '\s*(%s).*\(VRF: (\S+)\)' % ipAddrPattern, curLine )
         if m:
            srcIpLines.append( m.groups() )
         else:
            m = re.search( '\s*(%s).*\(VRF: (\S+)\)' % Arnet.Ip6AddrRe, curLine )
            if m:
               srcIp6Lines.append( m.groups() )
         curLine = lines.pop( 0 )
   else:
      curLine = lines.pop( 0 )

   parsedLines[ 'srcIp' ] = srcIpLines
   parsedLines[ 'srcIp6' ] = srcIp6Lines

   parsedLines[ 'sampleRateLine' ] = curLine
   curLine = lines.pop( 0 )
   if curLine.startswith( 'Hardware Sample Rate for HW sFlow' ):
      parsedLines[ 'accelSampleRateLine' ] = curLine
      curLine = lines.pop( 0 )
   parsedLines[ 'interval' ] = curLine
   parsedLines[ 'rewriteDscp' ] = lines.pop( 0 )
   parsedLines[ 'dscpValue' ] = lines.pop( 0 )
   if lines[ 0 ].startswith( 'Datagram transmission threshold' ):
      parsedLines[ 'hwDatagramTransmissionThreshold' ] = lines.pop( 0 )

   # blankline
   curLine = lines.pop( 0 )
   assert not curLine
   blankLines.append( curLine )

   curLine = lines.pop( 0 )
   assert curLine == 'Status'
   parsedLines[ 'statusLine' ] = curLine
   curLine = lines.pop( 0 )
   assert curLine == '------'
   parsedLines[ 'statusDashLine' ] = curLine
   parsedLines[ 'enabledLine' ] = lines.pop( 0 )

   curLine = lines.pop( 0 )
   if curLine.startswith( 'Hardware acceleration running' ):
      parsedLines[ 'accelEnabledLine' ] = curLine
      curLine = lines.pop( 0 )
   parsedLines[ 'polling' ] = curLine
   parsedLines[ 'sample' ] = lines.pop( 0 )
   parsedLines[ 'truncate' ] = lines.pop( 0 )
   curLine = lines.pop( 0 )
   if curLine.startswith( 'Sample Truncation Size for HW sFlow' ):
      parsedLines[ 'hwTruncate' ] = curLine
      curLine = lines.pop( 0 )

   # parsedLines[ 'send' ] is a list of ( sending, sendingReason, vrfName ) tuples
   sendLines = []
   if "Send Datagrams" in curLine:
      curLine = lines.pop( 0 )
      while 'BGP Export' not in curLine:
         # Two possible formats:
         #   (<Yes/No>) (VRF: <vrf>)
         #   (<Yes/No>) (<Reason>) (VRF: <vrf>)
         m = re.search( '(\S+)(\s*\(.+\))*\s*\(VRF: (\S+)\)', curLine )
         if m:
            sendLines.append( ( m.group( 1 ),
                                m.group( 2 ) if m.group( 2 ) else '',
                                m.group( 3 ) ) )
         curLine = lines.pop( 0 )
   parsedLines[ 'send' ] = sendLines

   # parsedLines[ 'export' ] is a list of ( export, vrfName ) tuples
   sendLines = []
   if "BGP Export" in curLine:
      curLine = lines.pop( 0 )
      while 'Hardware Sample' not in curLine:
         #   (<Yes/No>) (VRF: <vrf>)
         m = re.search( '(\S+)\s*\(VRF: (\S+)\)', curLine )
         if m:
            sendLines.append( ( m.group( 1 ), m.group( 2 ) ) )
         curLine = lines.pop( 0 )
   parsedLines[ 'export' ] = sendLines

   parsedLines[ 'hwRate' ] = curLine
   curLine = lines.pop( 0 )
   if curLine.startswith( 'Hardware Sample Rate for HW sFlow' ):
      parsedLines[ 'accelHwRate' ] = curLine
      curLine = lines.pop( 0 )
   if detail:
      parsedLines[ 'hwEnabled' ] = curLine
      parsedLines[ 'outputInterface' ] = lines.pop( 0 )
      parsedLines[ 'mplsExtension' ] = lines.pop( 0 )
      parsedLines[ 'switchExtension' ] = lines.pop( 0 )
      parsedLines[ 'routerExtension' ] = lines.pop( 0 )
      parsedLines[ 'tunnelIpv4EgrExtension' ] = lines.pop( 0 )
      parsedLines[ 'ingressSubintf' ] = lines.pop( 0 )
      parsedLines[ 'egressSubintf' ] = lines.pop( 0 )
      parsedLines[ 'portChannelIfindex' ] = lines.pop( 0 )
      curLine = lines.pop( 0 )
      if curLine.startswith( 'Hardware acceleration unsupported features' ):
         # We just drop all the unsupported features lines.
         curLine = lines.pop( 0 )
         while re.match( r'^\s.*', curLine ):
            curLine = lines.pop( 0 )
      # Drop the packet format line as it is tied to subinterface feature status
      curLine = lines.pop( 0 )

   # blankline
   assert not curLine
   blankLines.append( curLine )

   curLine = lines.pop( 0 )
   assert curLine == 'Statistics'
   parsedLines[ 'headerStatsLine' ] = curLine
   curLine = lines.pop( 0 )
   assert curLine == '----------'
   parsedLines[ 'statsDashLine' ] = curLine

   parsedLines[ 'totalPkts' ] = lines.pop( 0 )
   parsedLines[ 'swSamples' ] = lines.pop( 0 )
   parsedLines[ 'samplePoolLine' ] = lines.pop( 0 )
   parsedLines[ 'hwSamples' ] = lines.pop( 0 )
   parsedLines[ 'numDatagramsLine' ] = lines.pop( 0 )
   if detail:
      parsedLines[ 'samplesDiscarded' ] = lines.pop( 0 )

   parsedLines[ 'blankLines' ] = blankLines

   return parsedLines

def sflowHwStatus( sflowHwStatusDir ):
   if len( sflowHwStatusDir ):
      return sflowHwStatusDir.values()[ 0 ]
   else:
      return Tac.newInstance( 'Sflow::HwStatus', 'empty' )

def sflowHwSampleRate( sflowHwStatusDir ):
   for status in sflowHwStatusDir.values():
      if status.sampleRate:
         return status.sampleRate
   return 0
