#!/usr/bin/env python
# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import CliSave
import Tracing
import Tac
import CliSession
import ConfigMount
import LazyMount

__defaultTraceHandle__ = Tracing.Handle( "CliSessionAgent" )
t0 = Tracing.trace0
t1 = Tracing.trace1
t2 = Tracing.trace2

def sessionConfig( entityManager, disableAaa, cliSession=None, showAll=False,
                   showDetail=False, showSanitized=False, sessionName="",
                   showJson=False, showNoSeqNum=False ):
   # Modified from Cli/Url.py, localRunningConfigAll
   # These urls are for internal use and are not displayed externally to user.
   # Currently, there is no other better way to pass arguments to Url.get() method.
   # These urls are used to pass appropriate flags to saver functions.
   import Url
   context = Url.Context( entityManager, disableAaa, cliSession )
   urlString = "session:/%s-session-config" % sessionName
   if showAll or showDetail:
      urlString += ":all"
   if showDetail:
      urlString += ":detail"
   if showJson:
      urlString += ":json"
   if showSanitized:
      urlString += ":sanitized"
   if showNoSeqNum:
      urlString += ":noseqnum"
   return Url.parseUrl( urlString, context )

def saveSessionConfig( em, session, f, saveAll, saveAllDetail, showSanitized=False,
                       clean=False, showJson=False, showNoSeqNum=False,
                       errorObj=None, showProfileExpanded=False ):
   if clean:
      sStatus = getSessionStatus( em )
      if( ( not sStatus ) or
         ( not sStatus.cleanConfigStatus ) or
         ( not sStatus.cleanConfigStatus.cleanConfigComplete ) ):
         print( "Cli config sessions are not fully initialized yet." )
         return
      sessionRoot = sStatus.cleanConfig
   else:
      sessionRootDir = em.root()[ 'session' ][ 'sessionDir' ]

      if not session in sessionRootDir:
         print( "Session %s does not exist" % session )
         return
      sessionRoot = sessionRootDir.get( session )
   if not ( clean or sessionRoot ): # maybe just not mounted.
      mg = em.mountGroup()
      sessionRoot = mg.mount( "session/sessionDir/%s" % session,
                              "Tac::Dir", "ri" )
      mg.close( blocking=True )
   if not sessionRoot:
      print( "The config state for Session %s is unavailable." % session )
      return

   CliSave.saveSessionConfigInternal( em, session, sessionRoot, em.root(),
                                      f,
                                      saveAll, saveAllDetail,
                                      showSanitized=showSanitized,
                                      cleanConfig=clean,
                                      showJson=showJson,
                                      showNoSeqNum=showNoSeqNum,
                                      errorObj=errorObj,
                                      showProfileExpanded=showProfileExpanded )

def getSessionStatus( entityManager ):
   CliSession.waitForSessionStatusInitialized( entityManager )
   return CliSession.sessionStatus

saveRootSpecialCases_ = (
)

def _pathIsSaveRootSpecialCase( path ):
   for exp in saveRootSpecialCases_:
      if path.startswith( exp ):
         return True
   return False

# Determines if we assert or just trace CLI plug-in save root inconsistencies
assertOnBadSaveRoot_ = True

def processHandler( handler, handlerDir, relevantPrefixes ):
   #   if handler.kind is not None:
   #      return
   path = handler.pathPrefix.rstrip( '/' )

   # We allow a "" path. The saver presumably does not need anything from Sysdb
   if not path:
      return

   # Allow special-cases with bugs assigned
   specialCase = _pathIsSaveRootSpecialCase( path )
   assert handlerDir.configRoot is not None
   assert handlerDir.configRoot.rootsComplete
   if assertOnBadSaveRoot_ and not specialCase:
      assert handlerDir.configRoot.rootTrie.hasPrefix( path ), \
             "SaveRoot %s not covered by session roots!" % ( path, )
   if not handlerDir.configRoot.rootTrie.hasPrefix( path ):
      t1( "SaveRoot %s not covered by session roots%s" % ( path,
            ' (special case).' if specialCase else '!' ) )
      relevantPrefixes.add( path )
   for path in handler.requireMounts:
      if path and not handlerDir.configRoot.rootTrie.hasPrefix( path ):
         relevantPrefixes.add( path )

def prepareForSaveSessionConfig( entityManager, sessionName ):
   sessionStatus = getSessionStatus( entityManager )
   if not sessionStatus:
      # raise exception?
      t0( "session config infrastructure not ready" )
      return
   sStatus = sessionStatus.pendingChangeStatusDir.status.get( sessionName )
   if not sStatus:
      # raise exception?
      t0( "Cannot find session named '%s'" % sessionName )
      return

__mountsForSaveSessionMounted__ = False

def saveSessionMountsDone():
   return bool( __mountsForSaveSessionMounted__ )

def doMountsForSaveSession( entityManager, sessionStatus, callback ):
   if __mountsForSaveSessionMounted__:
      if callback:
         callback()
      return

   def finish():
      global __mountsForSaveSessionMounted__
      __mountsForSaveSessionMounted__ = True
      t0( 'Config root and requireMounts mounted.' )
      if callback:
         callback()

   # Load the CliSave plugins the first time this function is called.
   mg = entityManager.mountGroup()
   t0( 'Loading CliSavePlugins' )
   CliSave.maybeLoadCliSavePlugins( entityManager )

   relevantPrefixes = set()
   handlerDir = CliSession.handlerDir( entityManager )

   for typeHandlerInfo in CliSave.saveHandlers_:
      processHandler( typeHandlerInfo,
                      handlerDir, relevantPrefixes )

   for noTypeHandlerInfo in CliSave.noTypeSaveHandlers_:
      processHandler( noTypeHandlerInfo,
                      handlerDir, relevantPrefixes )

   for path in handlerDir.configRoot.root:
      t = handlerDir.configRoot.rootTrie.prefixTypename( path )
      flags = handlerDir.configRoot.rootTrie.prefixFlags( path )
      t0( 'Mounting root', path, ' type', t, ' flags', flags )
      mg.mount( path, t, flags )

   for path in relevantPrefixes:
      flags = LazyMount.mountFlag( entityManager, path ) or 'r'
      t0( 'Mounting', path, ', flags', flags )
      mg.mount( path, '', flags )

   # Force all config mounts too. This may help with BUG222721.
   ConfigMount.doDeferredMounts( block=False )

   mg.close( blocking=False, callback=finish )
