#!/usr/bin/env python
# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import re

def inputSection( lines ):
   ''' This is a helper generator for sectionFilter function to yield a
   "section" for show startup-config and show running-config.  A section
   is defined as any main mode config command and its subsequent sub-mode
   commands.  Currently, this only matches a single top level mode all
   sub-modes under it.  The section yielded includes newlines, so any
   subsequent actions on a section needs to bear this in mind.'''

   section = ""
   for line in lines:
      if re.findall( r'^\S', line ) and section:
         yield section
         section = ""
      section += line
   if section:
      yield section

def sectionFilter( inputText, regexList ):
   '''inputText - accepts multiline input
   regexList - list of regular expressions
   It will return a text of filtered sections
   based on the regex list provided. This function may raise an re.error,
   so you'll want to import re if you want to catch it.'''

   # Using regex lookaheads to allow order independent searches.
   # First regex matches all bang seperators, so we include them
   # in the output.
   fullRegex = r"(?=^\!$)|"
   for singleRegex in regexList:
      fullRegex += "(?=.*%s)" % singleRegex
   regexPattern = re.compile( fullRegex, re.DOTALL + re.IGNORECASE )

   matchedSections = ( matched for matched in inputSection( inputText )
                       if regexPattern.match( matched ) )
   # The following state machine makes sure that bang seperators in a
   # show run/start are only outputed once for any series of bangs.
   # Since the regex above always matches bang only lines, this
   # provides a clean output.
   formerSectionBang = False
   firstBang = True
   for section in matchedSections:
      if section.strip() == '!':
         formerSectionBang = True
      else:
         if formerSectionBang and not firstBang:
            print( "!" )
         print( section, end='' )
         firstBang = False
         formerSectionBang = False
