#!/usr/bin/env python
# Copyright (c) 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliParser
import CliCommand
import CliMatcher
import CliSave
import Tracing
import random, crypt, re

traceHandle_ = Tracing.Handle( "SecretCli" )
debug = traceHandle_.trace0

# This implements the 'secret' CLI which has the following syntax:
#
# 0     Specifies an UNENCRYPTED password will follow
# 5     Specifies an ENCRYPTED password will follow
# LINE  The UNENCRYPTED (cleartext) root password
#
# The encrypted secret is an md5 hash of the cleartext secret.

# The config in Sysdb to get the hash algorithm
defaultHashFunc = None
minPasswordLengthFunc = None

# Regular expression that matches all the valid characters in a
# cleartext password (ie one entered at the 'Password:' prompt).
# pylint: disable=W1401
cleartextPasswdRe = ( '[0-9a-zA-Z\!\@\#\$\%\^\&\*\(\)\-\_\=\+' +
                      '\[\\]\{\}\\\;\:\'\"\<\>\,\.\?\/\`\~]+' )

# Regular expression matching an MD5-encrypted string returned by the
# glibc2 version of crypt(3).
#
# From the manpage of crypt(3):
#
#   If salt is a character string starting with the three characters
#   "$1$" followed by at most eight characters, and optionally termi-
#   nated by "$", then instead of using the DES machine, the glibc
#   crypt function uses an MD5-based algorithm, and outputs up to 34
#   bytes, namely "$1$<string>$", where "<string>" stands for the up
#   to 8 characters following "$1$" in the salt, followed by 22 bytes
#   chosen from the set [a~zA~Z0~9./]. The entire key is significant
#   here (instead of only the first 8 bytes)."""
hashedStringCharsRe = r'[a-zA-Z0-9\.\/]'
md5EncryptedPasswdRe = r'\$1\$%s{1,8}[$]%s{22}' % \
                       ( hashedStringCharsRe, hashedStringCharsRe )

# Regular expression matching the SHA-512 encrypted string returned by the
# glibc2 version of crypt(3).
#
# From the manpage of crypt(3):
#
#    If salt is a character string starting with the characters "$id$" followed by
#    a string terminated by "$":
#
#         $id$salt$encrypted
#   
#    then instead of using the DES machine, id identifies the encryption method
#    used and this then determines how the rest of the password string is
#    interpreted.  The following values of id are supported:
#   
#         ID  | Method
#         ---------------------------------------------------------
#         1   | MD5
#         2a  | Blowfish (not in mainline glibc; added in some
#             | Linux distributions)
#         5   | SHA-256 (since glibc 2.7)
#         6   | SHA-512 (since glibc 2.7)
#   
#    So $5$salt$encrypted is an SHA-256 encoded password and $6$salt$encrypted is
#    an SHA-512 encoded one.
#   
#    "salt" stands for the up to 16 characters following "$id$" in the salt.  The
#    encrypted part of the password string is the actual computed password.  The
#    size of this string is fixed:
#   
#    MD5     | 22 characters
#    SHA-256 | 43 characters
#    SHA-512 | 86 characters
#   
#    The characters in "salt" and "encrypted" are drawn from the set [a-zA-Z0-9./].
#    In the MD5 and SHA implementations the entire key is significant (instead of
#    only the first 8 bytes in DES).
sha512EncryptedPasswdRe = r'\$6\$%s{1,16}[$]%s{86}' % \
                          ( hashedStringCharsRe, hashedStringCharsRe )

passwordTooShortError = "Password too short: at least %d characters required."

def setDefaultHashFunc( func ):
   global defaultHashFunc
   defaultHashFunc = func

def setMinPasswordLengthFunc( func ):
   global minPasswordLengthFunc
   minPasswordLengthFunc = func

class SecretValue( object ):
   """
   Stores a hashed password. If a cleartext pasword is
   provided SecretValue will validate the password across
   various criteria.
   """

   def __init__( self, mode, secretHash, secretCleartext=None ):
      # secretErrors is a list of error conditions for the secret
      self.secretErrors = []
      self.mode = mode
      self.secretHash = secretHash
      if secretCleartext and minPasswordLengthFunc:
         minPasswordLength = minPasswordLengthFunc()
         if len( secretCleartext ) < minPasswordLength:
            self.secretErrors.append( passwordTooShortError % minPasswordLength )

   def hash( self ):
      if self.secretErrors:
         for errorMsg in self.secretErrors:
            self.mode.addError( errorMsg )
         raise CliParser.AlreadyHandledError()
      else:
         return self.secretHash


######### Password helper functions ##########
def generalSalt( algorithm ):
   """
   Returns the salt for the algorithm given. 'algorithm' is an
   str argument of one of the following types:
   [ md5, sha512 ]
   """
   saltId = 0
   saltLength = 0
   if algorithm == 'md5':
      saltId = 1
      saltLength = 8
   elif algorithm == 'sha512':
      saltId = 6
      saltLength = 16
   else:
      assert False, "Algorithm given was not an implemented type"

   # Return a salt for a passwd encryption. It only has to be unique,
   # and not easily guessable, but doesn't need any cryptographic
   # properties. See the August 2000 discussion on sci.crypt 'What is
   # required of "salt"?', which is cached at:
   #
   #     http://www.ciphersbyritter.com/NEWS6/SALT.HTM

   # The salt may contain only the characters [a-zA-Z0-9./]
   saltChars = list( '''abcdefghijklmnopqrstuvwxyz''' + \
                     '''ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789./''' )

   # In order to get the glibc2-enhanced 'crypt(3)' to use its
   # MD5-based algorithm, the salt has to start with '$1$'.
   salt = "$%d$" % ( saltId, )
   random.seed()
   for _ in range( 0, saltLength ):
      salt += random.choice( saltChars )
   salt += "$"
   return salt

def md5Salt():
   """
   Returns an 8-character salt suitable to be passed as the second
   argument to crypt.crypt() iff the underlying crypt(3) is the
   GNU-enhanced glibc2 version with support for an MD5-based
   algorithm.
   """
   return generalSalt( 'md5' )

def sha512Salt():
   """
   Returns an 16-character salt suitable to be passed as the second
   argument to crypt.crypt() iff the underlying crypt(3) is the
   GNU-enhanced glibc2 version with support for an SHA512-based
   algorithm.
   """
   return generalSalt( 'sha512' )

def generalEncryption( cleartextPasswd, algorithm, salt=None ):
   """
   Encrypts the password using the algorithm passed in. Algorithm is
   a str chosen from [ 'md5', 'sha512' ]. This depends on a GNU extension
   to the 'crypt(3)' function.
   """
   saltFunc = None
   if algorithm == 'md5':
      saltFunc = md5Salt
   elif algorithm == 'sha512':
      saltFunc = sha512Salt
   else:
      assert False, "Algorithm given was not an implemented type"
   
   if salt is None:
      salt = saltFunc()

   return crypt.crypt( cleartextPasswd, salt )

def md5EncryptedPassword( cleartextPasswd, salt=None ):
   """
   Encrypt the cleartext password using an MD5-based algorithm.
   This depends on a GNU extension to the"crypt(3)" function.
   """
   return generalEncryption( cleartextPasswd, 'md5', salt )

def sha512EncryptedPassword( cleartextPasswd, salt=None ):
   """
   Encrypt the cleartext password using a SHA512-based algorithm.
   This depends on a GNU extension to the"crypt(3)" function.
   """
   return generalEncryption( cleartextPasswd, 'sha512', salt )

def setCleartextSecret( mode, secretString, **kargs ):
   # Make sure we get the hash algorithm from Sysdb
   defaultHash = 'md5' if not defaultHashFunc else defaultHashFunc()
   if defaultHash == 'md5':
      debug( "MD5-encrypting password" )
      secretValue = SecretValue( mode, md5EncryptedPassword( secretString ),
                                 secretCleartext=secretString )
      return secretValue
   elif defaultHash == 'sha512':
      debug( "SHA512-encrypting password" )
      secretValue = SecretValue( mode, sha512EncryptedPassword( secretString ),
                                 secretCleartext=secretString )
      return secretValue
   else:
      assert False, "Unknown hash algorithm setting"

def setCleartextMD5Secret( mode, secretString, **kargs ):
   debug( "MD5-encrypting password" )
   secretValue = SecretValue( mode, md5EncryptedPassword( secretString ),
                              secretCleartext=secretString )
   return secretValue

cleartextZeroMatcher = CliMatcher.PatternMatcher(
   cleartextPasswdRe,
   helpname='LINE',
   helpdesc='The UNENCRYPTED (cleartext) password' )

cleartextMatcher = CliMatcher.PatternMatcher(
   # Do not allow 0/5/sha512/* to avoid confusion
   r'(?!^([05\*]|sha512)$)%s' % cleartextPasswdRe,
   helpname='LINE',
   helpdesc='The UNENCRYPTED (cleartext) password' )

md5Matcher = CliMatcher.PatternMatcher(
   md5EncryptedPasswdRe,
   helpname='LINE',
   helpdesc='The MD5 ENCRYPTED password' )

sha512Matcher = CliMatcher.PatternMatcher(
   sha512EncryptedPasswdRe,
   helpname='LINE',
   helpdesc='The SHA512 ENCRYPTED password' )

sha512KwMatcher = CliMatcher.KeywordMatcher(
   'sha512',
   helpdesc='Specifies an ENCRYPTED SHA512 password will follow' )

def secretCliExpression( name, supportInvalidPassword=False,
                         extraClearTextExclude=None,
                         shaSecretGuard=None ):
   """
   name: the name in args that holds the encrypted password
   supportInvalidPassword: whether include '*' to disable login
   extraClearTextExclude: clear text password supports white spaces, but it can
                          cause issue if the command supports additional options
                          after the secret. To work around this you can pass in
                          tokens that should not match additional cleatext tokens.
   shaSecretGuard: guard function for the SHA512 option.
   """
   if extraClearTextExclude:
      cleartextExtraMatcher = CliMatcher.PatternMatcher(
         r'(?!^(%s)$)%s' % ( '|'.join( extraClearTextExclude ),
                             cleartextPasswdRe ),
         helpname='LINE',
         helpdesc='The UNENCRYPTED (cleartext) password' )
   else:
      cleartextExtraMatcher = cleartextZeroMatcher

   class SecretExpression( CliCommand.CliExpression ):
      expression = "( ( 0 $CLEARTEXT0 ) | $CLEARTEXT [ { $CLEARTEXT_EXT } ] ) " \
                   "| ( 5 $MD5 ) | ( sha512 $SHA512 )"
      data = { "$CLEARTEXT0" : CliCommand.Node( cleartextZeroMatcher,
                                                sensitive=True ),
               "$CLEARTEXT" : CliCommand.Node( cleartextMatcher,
                                                sensitive=True ),
               "$CLEARTEXT_EXT" : CliCommand.Node( cleartextExtraMatcher,
                                                   sensitive=True ),
               '0' : 'Specify an UNENCRYPTED password will follow',
               '5' : 'Specify an ENCRYPTED MD5 password will follow',
               'sha512' : CliCommand.Node( sha512KwMatcher,
                                           guard=shaSecretGuard ),
               '$MD5' : md5Matcher,
               '$SHA512' : sha512Matcher }
      if supportInvalidPassword:
         expression += ' | *'
         data[ '*' ] = 'Specify a password that cannot be used to login'

      @staticmethod
      def adapter( mode, args, argsList ):
         def _getArg( argname ):
            # If the expression is used in a set, we may get an one-elem list
            val = args[ argname ]
            if isinstance( val, list ):
               assert len( val ) == 1
               val = val[ 0 ]
            return val

         secret = None
         if supportInvalidPassword and '*' in args:
            secret = SecretValue( mode, '*' )
            del args[ '*' ]
         elif '5' in args:
            secret = SecretValue( mode, _getArg( '$MD5' ) )
            del args[ '5' ]
            del args[ '$MD5' ]
         elif 'sha512' in args:
            secret = SecretValue( mode, _getArg( '$SHA512' ) )
            del args[ 'sha512' ]
            del args[ '$SHA512' ]
         elif '$CLEARTEXT0' in args or '$CLEARTEXT' in args:
            cleartext = args.pop( '$CLEARTEXT_EXT', [] )
            if len( cleartext ) == 1 and isinstance( cleartext[ 0 ], list ):
               # handle the case where SecretExpression is used in a set
               cleartext = cleartext[ 0 ]
            if '$CLEARTEXT0' in args:
               cleartext.insert( 0, _getArg( '$CLEARTEXT0' ) )
            else:
               cleartext.insert( 0, _getArg( '$CLEARTEXT' ) )

            for arg in ( '$CLEARTEXT0', '$CLEARTEXT', '$CLEARTEXT_EXT', '0' ):
               args.pop( arg, None )
            cleartext = ' '.join( cleartext )
            if shaSecretGuard and shaSecretGuard( mode, '0' ):
               secret = setCleartextMD5Secret( mode, cleartext )
            else:
               secret = setCleartextSecret( mode, cleartext )
         if secret:
            args[ name ] = secret

   return SecretExpression

hashTypeFinder = re.compile( r'\$(\d+)\$.+' )
def getHashToken( hashString ):
   """
   Helper function for external CliSavePlugins. Given a hash string
   returns the token that would be used to save it from the CLI.
   ex:
   $1$<md5hash> would return '5'
   $6$<sha512hash> would return 'sha512'

   Unknown algorithms will return None.
   """
   hashId = hashTypeFinder.search( hashString )
   if hashId:
      if hashId.group( 1 ) == '1':
         # md5 hash id
         return '5'
      elif hashId.group( 1 ) == '6':
         # sha512 hash id
         return 'sha512'
      else:
         # Unknown hash id
         return None
   else:
      return None

def getSecretString( encryptedPasswd, saveOptions ):
   # return running-config string for encryptedPasswd
   # None for error
   assert encryptedPasswd
   if encryptedPasswd == '*':
      return '*'
   else:
      hashToken = getHashToken( encryptedPasswd )
      if hashToken:
         return hashToken + " %s" % CliSave.sanitizedOutput( saveOptions,
                                                             encryptedPasswd )
   return None
