# Copyright (c) 2008, 2009, 2010  Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import NativeTime
import PyClientBase
import TacUtils
import Tracing
import UtmpDump
import os
import sys
import time

th = Tracing.Handle( 'RunCli' )
t0 = th.trace0
t6 = th.trace6

class AuthzDeniedError( Exception ): pass

# We have to implement our own waitFor function
def waitFor( func, description="", timeout=60, sleep=True ):
   startTime = NativeTime.monotonic_time()
   endTime = startTime + timeout
   lastPrintedTime = 0
   while True:
      now = NativeTime.monotonic_time()
      if now > endTime:
         raise TacUtils.Timeout( "Aaa connection error" )
      result = func()
      if result:
         if lastPrintedTime:
            sys.stderr.write( 'ok\n' )
            sys.stderr.flush()
         return result
      if now - startTime > 10:
         # print something
         if not lastPrintedTime:
            sys.stderr.write( "waiting for %s." % description )
            sys.stderr.flush()
            lastPrintedTime = now
         elif now > lastPrintedTime + 1:
            sys.stderr.write( '.' )
            sys.stderr.flush()
            lastPrintedTime = now
      if now < startTime + 2:
         sleepTime = 0.1
      else:
         sleepTime = 1
      time.sleep( sleepTime )

def _authorizeExec( sysname, uid=None ):
   authorized = False
   message = ""
   privLevel = 1
   autoCmd = None
   if 'AAA_AUTHN_ID' in os.environ:
      sessionId = int( os.environ[ 'AAA_AUTHN_ID' ] )
   else:
      sessionId = None
   if uid is None:
      uid = os.geteuid()
   sessionPid = None
   tty = UtmpDump.getTtyName()
   if tty:
      utmpEntry = UtmpDump.getTtyLoginEntry( tty )
      if utmpEntry:
         sessionPid = int( utmpEntry[ 'pid' ] )
   try:
      t0( "RPC to Aaa: authorizeShellWithOutput(uid=", uid,
          "sessionId=", sessionId, "sessionPid=", sessionPid,
          "tty=", tty, ") with sysname", sysname )
      pc = PyClientBase.PyClientBase( 
         sysname, 'Aaa', waitFunc=waitFor,
         execMode=PyClientBase.Rpc.execModeThreadPerConnection,
         initConnectCmd="import AaaApi" )
      cmd = ( "AaaApi.authorizeShellWithOutput(uid=%r,sessionId=%r,"
              "sessionPid=%r,tty=%r)" % ( uid, sessionId, sessionPid, tty ) )
      result = pc.execute( cmd )
      t0( "RPC result is", result )
      try:
         status, privLevel, autoCmd, message = result.split( '\x01', 3 )
      except ValueError:
         # the result isn't expected, print it out
         print "Error: unexpected authorization result"
         print repr( result )
         return ( False, "bad response", 1, "" )

      if status == 'allowed':
         authorized = True
      try:
         privLevel = int( privLevel )
      except ValueError:
         pass
      if privLevel < 0 or privLevel > 15:
         print "Warning: invalid privilege level %s, default to 1" % privLevel
         privLevel = 1
   except:
      import traceback
      exc = traceback.format_exc()
      t0( "Exception occurred during authorization:", exc )
      raise
   return ( authorized, message, privLevel, autoCmd )

def checkFilesystemFull( percentage=90, tmpfsOnly=True ):
   """Check filesystem usage to see if any exceeds the threshold given as 
   percentage. If tmpfsOnly is True, only overlay/tmpfs filesystems are checked.
   Otherwise, all filesystems are checked. """
   findFsFull = False
   output = TacUtils.run( [ "df", "-Ph" ], ignoreReturnCode=True,
                          stdout=TacUtils.CAPTURE )
   dfLines = output.rstrip().splitlines()
   if len( dfLines ) <= 1:
      return
   for dfLine in dfLines[ 1: ]:
      # df output format /Filesystem/1K-blocks/Used/Available/Use%/Mounted on/
      dfWords = dfLine.split()
      if len( dfWords ) < 6:
         # We have used -P option in df. We are not supposed to see broken lines.
         # Do not know what to do with this line. Skip it.
         continue
      if tmpfsOnly and dfWords[ 0 ] not in ( 'tmpfs', 'none' ):
         continue
      # Try block for debugging very intermittent BUG258587
      # Remove when found&fixed
      try:
         tmp = int( dfWords[-2][ :-1 ] )
         t6( "df -Ph Produced value: ", tmp )
      except:
         t0( "df -Ph Produced unexpected non-integer: ", dfWords[ -2 ][ :-1 ] )
         t0( "Full output of df:" )
         t0( output )
         raise
      if int( dfWords[-2][ :-1 ] ) >= percentage:
         if not findFsFull:
            findFsFull = True
            print "Warning: the following filesystems have less than %d%% "\
                "free space left:" % ( 100 - percentage )
         print( "%-24s    %2d%% (%4s bytes) Available" %
                ( dfWords[ -1 ], 100 - int( dfWords[ -2 ][ : -1 ] ),
                  dfWords[ 3 ] ) )
   if findFsFull:
      print "Please remove configuration such as tracing and clean up the space."
   else:
      t0( "All filesystem usage below %d" % percentage )

def authorizeAndExecCli( sysname, command=None, cliPath="/usr/bin/CliShell",
                         uid=None ):
   """Performs an authorization check for executing a shell, and if allowed,
   execs /usr/bin/Cli.  Also handles special cases of the command list that
   should be handled by a different shell than the Cli, such as scp."""
   authorized, message, privLevel, autoCmd = _authorizeExec( sysname, uid=uid )
   if authorized:
      args = [ cliPath, '--sysname', sysname, '--privilege', str( privLevel ) ]
      if autoCmd:
         args += [ '-c', autoCmd ]
      elif command:
         args += [ '-c', command ]
      else:
         checkFilesystemFull()
      t0( "Authorization succeeded: execing", " ".join( args ) )
      # Setting GLIBCXX_FORCE_NEW prevents stdlibc++ from using type-specific
      # allocators for STL container types (which waste a lot of memory to achieve a
      # dubious performance gain).
      assert '_Tac' not in sys.modules
      assert 'Tac' not in sys.modules
      os.environ[ 'GLIBCXX_FORCE_NEW' ] = '1'
      os.execv( args[0], args )
   else:
      raise AuthzDeniedError( message )
