# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# Module contains utility methods for accessing Intf related information from sysdb

from __future__ import absolute_import, division, print_function

import os
import Tac

@Tac.memoize
def mgmtPrefix():
   # MGMT_INTF_PREFIX allows a test to override the prefix that
   # identifies a management interface
   return os.environ.get( "MGMT_INTF_PREFIX", "Management" )

def isManagement( intfName ):
   return intfName.startswith( mgmtPrefix() )

def isLoopback( intfName ):
   return intfName.startswith( 'Loopback' )

def isInternal( intfName ):
   return intfName.startswith( 'Internal' )

#--------------------------------------------------------------------
# Returns if the interface supports virtual routing or not
#--------------------------------------------------------------------
def intfSupportsVirtualRouter( intfName ):
   return not isManagement( intfName ) and intfName.startswith( ( "Vlan", "Test" ) )

#-------------------------------------------------------------------
# Helper Methods
#-------------------------------------------------------------------
#
# Returns True if the interface is configured as routed interface.
# If includeEligible is set then it returns True for intfs which are capable of
# routing. intfStatusDir points to interface/status/all directory.
#
def _routedIntf( intfName, intfStatusDir, includeEligible ):
   # Previously, if includeEligible was False then only non-switchport interfaces
   # were returned. This was determined by looking for 'Bridged' forwarding model
   # input for those interfaces. In the new forwarding model design, this introduced
   # a dependency on Ebra, which was not acceptable. So it was decided to just ignore
   # this parameter and always return all interfaces (except Vxlan).
   if 'Vxlan' in intfName:
      return False
   return True

#
# Returns list of interface names which are configured for routing.
# If includeEligible is set to True then the names of all interfaces capable of
# routing are returned. Else only currently configured interfaces are returned.
# If excludeLpMgmt is set then loopback and mgmt interfaces are excluded.
#
def _allRoutingIntfNames( sysdbRoot, includeEligible=False, excludeLpMgmt=False,
                          requireMounts=None ):
   if requireMounts:
      intfConfigDir = requireMounts[ 'interface/config/all' ]
      intfStatusDir = requireMounts[ 'interface/status/all' ]
   else:
      intfConfigDir = sysdbRoot[ 'interface' ][ 'config' ][ 'all' ]
      intfStatusDir = sysdbRoot[ 'interface' ][ 'status' ][ 'all' ]
   intfNames = []
   for intfName in intfConfigDir:
      if isInternal( intfName ):
         continue

      if excludeLpMgmt and ( isLoopback( intfName ) or isManagement( intfName ) ):
         continue

      if _routedIntf( intfName, intfStatusDir, includeEligible ):
         intfNames.append( intfName )

   return intfNames

#-----------------------------------------------------------------------------------
# Returns the list of interface names which support IP address configuration.
# If 'includeEligible' is set to True then the names for all interfaces capable
# of configuring IP address are returned. Else only currently configured
# L3 interface names are returned.
#-------------------------------------------------------------------------------
def allIpIntfNames( sysdbRoot, includeEligible=False,
                    requireMounts=None ):
   return _allRoutingIntfNames( sysdbRoot, includeEligible,
                                requireMounts=requireMounts )

#----------------------------------------------------------------------------
# Returns the list of interface names which support routing protocols.
# If 'includeEligible' is set to True then the names for all interfaces capable
# of configuring IP address are returned. Else only currently configured
# L3 interface names are returned.
#
# This method is currently used by CliSavePlugin, so only interfaces which
# support routing protocol Cli's are returned.
# It will be better to check interface's capability using an attribute in sysdb.
# But nothing exists for now, so filtering out loopback and mgmt using names.
#----------------------------------------------------------------------------
def allRoutingProtocolIntfNames( sysdbRoot, includeEligible=False,
                                 requireMounts=None ):
   return _allRoutingIntfNames( sysdbRoot, includeEligible, excludeLpMgmt=True,
                                requireMounts=requireMounts )
